package org.planx.msd;

import java.util.HashMap;
import java.util.Map;
import java.util.List;
import org.planx.msd.array.*;
import org.planx.msd.character.*;
import org.planx.msd.lang.*;
import org.planx.msd.list.*;
import org.planx.msd.number.*;

/**
 * This factory class provides shared discriminators for atomic types
 * and commonly used datatypes. The discriminators are <i>not</i>
 * synchronized and should be synchronized externally if used by
 * multiple threads. However, if multiple threads perform multiset
 * discrimination simultaneously, the preferred way is to create a
 * non-shared discriminator for each thread directly, and not use this
 * factory.
 * This is done by creating a {@link Memory} for each thread and passing
 * the <code>Memory</code> object for a given thread to the constructors
 * of the discriminators needed for that thread. This requires more
 * memory than synchronizing shared discriminators but avoids congestion.
 * <p>
 * For further information please see the
 * <a href="package-summary.html#package_description">Package Summary</a>.
 *
 * @author Thomas Ambus
 */
public abstract class DiscriminatorFactory {
    private static DiscriminatorFactory instance = null;

    protected DiscriminatorFactory() {}
    /**
     * Returns an instance of <code>DiscriminatorFactory</code>.
     */
    public static DiscriminatorFactory instance() {
        if (instance == null) instance = new DiscriminatorFactoryImpl();
        return instance;
    }
    /**
     * Returns a {@link PolymorphicDiscriminator} for the specified class.
     * This is useful for maintaining the ability to discriminate subtypes
     * of various classes.
     * Each implementation should make sure to register its sub-classes
     * with the <code>PolymorphicDiscriminator</code> of the appropriate
     * super class before any other part of the system needs to
     * discriminate its sub-classes.
     */
    public abstract <E> PolymorphicDiscriminator<E>
        getPolymorphicDiscriminator(Class<E> cls);
    /**
     * Returns a shared <code>Memory</code> instance for use by customized
     * discriminators.
     */
    public abstract Memory getMemory();
    /**
     * Returns a shared discriminator for <code>Character</code> objects.
     */
    public abstract Discriminator<Character> getCharacterDiscriminator();
    /**
     * Returns a shared discriminator for <code>CharSequence</code> objects.
     */
    public abstract <T extends CharSequence> Discriminator<T>
        getCharSequenceDiscriminator();
    /**
     * Returns a shared <code>ClassDiscriminator</code> for <code>Object</code>s.
     */
    public abstract ClassDiscriminator<Object> getClassDiscriminator();
    /**
     * Returns a shared discriminator for <code>Byte</code> objects.
     */
    public abstract Discriminator<Byte> getByteDiscriminator();
    /**
     * Returns a shared discriminator for <code>Short</code> objects.
     */
    public abstract Discriminator<Short> getShortDiscriminator();
    /**
     * Returns a shared discriminator for <code>Integer</code> objects.
     */
    public abstract Discriminator<Integer> getIntegerDiscriminator();
    /**
     * Returns a shared discriminator for <code>Long</code> objects.
     */
    public abstract Discriminator<Long> getLongDiscriminator();
    /**
     * Returns a shared discriminator for <code>Float</code> objects.
     */
    public abstract Discriminator<Float> getFloatDiscriminator();
    /**
     * Returns a shared discriminator for <code>Double</code> objects.
     */
    public abstract Discriminator<Double> getDoubleDiscriminator();
    /**
     * Returns a shared discriminator for <code>List</code>s of the type
     * implied by the specified <code>Discriminator</code>.
     */
    public abstract <T> Discriminator<List<T>>
        getListDiscriminator(Discriminator<T> d);
    /**
     * Returns a shared discriminator for <code>List</code>s considered as bags.
     */
    public abstract <T> Discriminator<List<T>>
        getBagDiscriminator(Discriminator<T> d);
    /**
     * Returns a shared discriminator for <code>List</code>s considered as sets.
     */
    public abstract <T> Discriminator<List<T>>
        getSetDiscriminator(Discriminator<T> d);
}

class DiscriminatorFactoryImpl extends DiscriminatorFactory {
    private static final Memory memory = new Memory();

    private Map<Class,PolymorphicDiscriminator> pdiscs =
        new HashMap<Class,PolymorphicDiscriminator>();

    public <E> PolymorphicDiscriminator<E>
                        getPolymorphicDiscriminator(Class<E> cls) {
        synchronized (pdiscs) {
            PolymorphicDiscriminator<E> pd = pdiscs.get(cls);
            if (pd == null) {
                pd = new PolymorphicDiscriminator<E>();
                pdiscs.put(cls, pd);
            }
            return pd;
        }
    }

    public Memory getMemory() {
        return memory;
    }
    public Discriminator<Character> getCharacterDiscriminator() {
        return new CharacterDiscriminator(memory);
    }
    public <T extends CharSequence> Discriminator<T>
                                    getCharSequenceDiscriminator() {
        return new CharSequenceDiscriminator<T>(memory);
    }
    public ClassDiscriminator<Object> getClassDiscriminator() {
        return new ClassDiscriminator<Object>(memory);
    }
    public Discriminator<Byte> getByteDiscriminator() {
        return new ByteDiscriminator(memory);
    }
    public Discriminator<Short> getShortDiscriminator() {
        return new ShortDiscriminator(memory);
    }
    public Discriminator<Integer> getIntegerDiscriminator() {
        return new IntegerDiscriminator(memory);
    }
    public Discriminator<Long> getLongDiscriminator() {
        return new LongDiscriminator(memory);
    }
    public Discriminator<Float> getFloatDiscriminator() {
        return new FloatDiscriminator(memory);
    }
    public Discriminator<Double> getDoubleDiscriminator() {
        return new DoubleDiscriminator(memory);
    }
    public <T> Discriminator<List<T>> getListDiscriminator(
                                        Discriminator<T> d) {
        return new ListDiscriminator<T>(d, memory);
    }
    public <T> Discriminator<List<T>> getBagDiscriminator(
                                       Discriminator<T> d) {
        return new BagDiscriminator<T>(d, memory);
    }
    public <T> Discriminator<List<T>> getSetDiscriminator(
                                       Discriminator<T> d) {
        return new SetDiscriminator<T>(d, memory);
    }
}
