/*
 * Decompiled with CFR 0.152.
 */
package org.openscience.cdk.smiles;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import org.openscience.cdk.config.Isotopes;
import org.openscience.cdk.exception.CDKException;
import org.openscience.cdk.interfaces.IAtom;
import org.openscience.cdk.interfaces.IAtomContainer;
import org.openscience.cdk.interfaces.IBond;
import org.openscience.cdk.interfaces.IDoubleBondStereochemistry;
import org.openscience.cdk.interfaces.IIsotope;
import org.openscience.cdk.interfaces.IStereoElement;
import org.openscience.cdk.interfaces.ITetrahedralChirality;
import org.openscience.cdk.smiles.SmiFlavor;
import org.openscience.cdk.stereo.ExtendedCisTrans;
import org.openscience.cdk.stereo.ExtendedTetrahedral;
import uk.ac.ebi.beam.Atom;
import uk.ac.ebi.beam.AtomBuilder;
import uk.ac.ebi.beam.Bond;
import uk.ac.ebi.beam.Configuration;
import uk.ac.ebi.beam.Edge;
import uk.ac.ebi.beam.Element;
import uk.ac.ebi.beam.Graph;
import uk.ac.ebi.beam.GraphBuilder;

final class CDKToBeam {
    private final int flavour;

    CDKToBeam() {
        this(28);
    }

    CDKToBeam(int flavour) {
        this.flavour = flavour;
    }

    Graph toBeamGraph(IAtomContainer ac) throws CDKException {
        return CDKToBeam.toBeamGraph(ac, this.flavour);
    }

    Atom toBeamAtom(IAtom atom) throws CDKException {
        return CDKToBeam.toBeamAtom(atom, this.flavour);
    }

    Edge toBeamEdge(IBond b, Map<IAtom, Integer> indices) throws CDKException {
        Preconditions.checkArgument((b.getAtomCount() == 2 ? 1 : 0) != 0, (Object)"Invalid number of atoms on bond");
        int u = indices.get(b.getBegin());
        int v = indices.get(b.getEnd());
        return CDKToBeam.toBeamEdgeLabel(b, this.flavour).edge(u, v);
    }

    static Graph toBeamGraph(IAtomContainer ac, int flavour) throws CDKException {
        int order = ac.getAtomCount();
        GraphBuilder gb = GraphBuilder.create((int)order);
        HashMap indices = Maps.newHashMapWithExpectedSize((int)order);
        for (IAtom a : ac.atoms()) {
            indices.put(a, indices.size());
            gb.add(CDKToBeam.toBeamAtom(a, flavour));
        }
        for (IBond b : ac.bonds()) {
            gb.add(CDKToBeam.toBeamEdge(b, flavour, indices));
        }
        if (SmiFlavor.isSet(flavour, 1792)) {
            for (IStereoElement se : ac.stereoElements()) {
                if (SmiFlavor.isSet(flavour, 256) && se instanceof ITetrahedralChirality) {
                    CDKToBeam.addTetrahedralConfiguration((ITetrahedralChirality)se, gb, indices);
                    continue;
                }
                if (SmiFlavor.isSet(flavour, 512) && se instanceof IDoubleBondStereochemistry) {
                    CDKToBeam.addGeometricConfiguration((IDoubleBondStereochemistry)se, flavour, gb, indices);
                    continue;
                }
                if (SmiFlavor.isSet(flavour, 1024) && se instanceof ExtendedTetrahedral) {
                    CDKToBeam.addExtendedTetrahedralConfiguration((ExtendedTetrahedral)se, gb, indices);
                    continue;
                }
                if (!SmiFlavor.isSet(flavour, 1280) || !(se instanceof ExtendedCisTrans)) continue;
                CDKToBeam.addExtendedCisTransConfig((ExtendedCisTrans)se, gb, indices, ac);
            }
        }
        return gb.build();
    }

    private static Integer getMajorMassNumber(Element e) {
        try {
            switch (e) {
                case Hydrogen: {
                    return 1;
                }
                case Boron: {
                    return 11;
                }
                case Carbon: {
                    return 12;
                }
                case Nitrogen: {
                    return 14;
                }
                case Oxygen: {
                    return 16;
                }
                case Fluorine: {
                    return 19;
                }
                case Silicon: {
                    return 28;
                }
                case Phosphorus: {
                    return 31;
                }
                case Sulfur: {
                    return 32;
                }
                case Chlorine: {
                    return 35;
                }
                case Iodine: {
                    return 127;
                }
            }
            Isotopes isotopes = Isotopes.getInstance();
            IIsotope isotope = isotopes.getMajorIsotope(e.symbol());
            if (isotope != null) {
                return isotope.getMassNumber();
            }
            return null;
        }
        catch (IOException ex) {
            throw new InternalError("Isotope factory wouldn't load: " + ex.getMessage());
        }
    }

    static Atom toBeamAtom(IAtom a, int flavour) {
        Integer massNumber;
        boolean aromatic = SmiFlavor.isSet(flavour, 16) && a.getFlag(32);
        Integer charge = a.getFormalCharge();
        String symbol = (String)Preconditions.checkNotNull((Object)a.getSymbol(), (Object)"An atom had an undefined symbol");
        Element element = Element.ofSymbol((String)symbol);
        if (element == null) {
            element = Element.Unknown;
        }
        AtomBuilder ab = aromatic ? AtomBuilder.aromatic((Element)element) : AtomBuilder.aliphatic((Element)element);
        Integer hCount = a.getImplicitHydrogenCount();
        if (element == Element.Unknown) {
            ab.hydrogens(hCount != null ? hCount : 0);
        } else {
            ab.hydrogens(((Integer)Preconditions.checkNotNull((Object)hCount, (Object)"One or more atoms had an undefined number of implicit hydrogens")).intValue());
        }
        if (charge != null) {
            ab.charge(charge.intValue());
        }
        if (SmiFlavor.isSet(flavour, 2056) && (massNumber = a.getMassNumber()) != null) {
            ab.isotope(massNumber.intValue());
        }
        Integer atomClass = (Integer)a.getProperty((Object)"cdk:AtomAtomMapping");
        if (SmiFlavor.isSet(flavour, 4) && atomClass != null) {
            ab.atomClass(atomClass.intValue());
        }
        return ab.build();
    }

    static Edge toBeamEdge(IBond b, int flavour, Map<IAtom, Integer> indices) throws CDKException {
        Preconditions.checkArgument((b.getAtomCount() == 2 ? 1 : 0) != 0, (Object)"Invalid number of atoms on bond");
        int u = indices.get(b.getBegin());
        int v = indices.get(b.getEnd());
        return CDKToBeam.toBeamEdgeLabel(b, flavour).edge(u, v);
    }

    private static Bond toBeamEdgeLabel(IBond b, int flavour) throws CDKException {
        if (SmiFlavor.isSet(flavour, 16) && b.isAromatic()) {
            if (!b.getBegin().isAromatic() || !b.getEnd().isAromatic()) {
                throw new IllegalStateException("Aromatic bond connects non-aromatic atomic atoms");
            }
            return Bond.AROMATIC;
        }
        if (b.getOrder() == null) {
            throw new CDKException("A bond had undefined order, possible query bond?");
        }
        IBond.Order order = b.getOrder();
        switch (order) {
            case SINGLE: {
                return Bond.SINGLE;
            }
            case DOUBLE: {
                return Bond.DOUBLE;
            }
            case TRIPLE: {
                return Bond.TRIPLE;
            }
            case QUADRUPLE: {
                return Bond.QUADRUPLE;
            }
        }
        if (!SmiFlavor.isSet(flavour, 16) && b.isAromatic()) {
            throw new CDKException("Cannot write Kekul\u00e9 SMILES output due to aromatic bond with unset bond order - molecule should be Kekulized");
        }
        throw new CDKException("Unsupported bond order: " + order);
    }

    private static void addGeometricConfiguration(IDoubleBondStereochemistry dbs, int flavour, GraphBuilder gb, Map<IAtom, Integer> indices) {
        IBond db = dbs.getStereoBond();
        IBond[] bs = dbs.getBonds();
        if (SmiFlavor.isSet(flavour, 16) && db.getFlag(32)) {
            return;
        }
        int u = indices.get(db.getBegin());
        int v = indices.get(db.getEnd());
        int x = indices.get(bs[0].getOther(db.getBegin()));
        int y = indices.get(bs[1].getOther(db.getEnd()));
        if (dbs.getStereo() == IDoubleBondStereochemistry.Conformation.TOGETHER) {
            gb.geometric(u, v).together(x, y);
        } else {
            gb.geometric(u, v).opposite(x, y);
        }
    }

    private static void addTetrahedralConfiguration(ITetrahedralChirality tc, GraphBuilder gb, Map<IAtom, Integer> indices) {
        IAtom[] ligands = tc.getLigands();
        int u = indices.get(tc.getChiralAtom());
        int[] vs = new int[]{indices.get(ligands[0]), indices.get(ligands[1]), indices.get(ligands[2]), indices.get(ligands[3])};
        gb.tetrahedral(u).lookingFrom(vs[0]).neighbors(vs[1], vs[2], vs[3]).winding(tc.getStereo() == ITetrahedralChirality.Stereo.CLOCKWISE ? Configuration.CLOCKWISE : Configuration.ANTI_CLOCKWISE).build();
    }

    private static void addExtendedTetrahedralConfiguration(ExtendedTetrahedral et, GraphBuilder gb, Map<IAtom, Integer> indices) {
        IAtom[] ligands = et.peripherals();
        int u = indices.get(et.focus());
        int[] vs = new int[]{indices.get(ligands[0]), indices.get(ligands[1]), indices.get(ligands[2]), indices.get(ligands[3])};
        gb.extendedTetrahedral(u).lookingFrom(vs[0]).neighbors(vs[1], vs[2], vs[3]).winding(et.winding() == ITetrahedralChirality.Stereo.CLOCKWISE ? Configuration.CLOCKWISE : Configuration.ANTI_CLOCKWISE).build();
    }

    private static void addExtendedCisTransConfig(ExtendedCisTrans ect, GraphBuilder gb, Map<IAtom, Integer> indices, IAtomContainer container) {
        IAtom[] ends = ExtendedCisTrans.findTerminalAtoms((IAtomContainer)container, (IBond)((IBond)ect.getFocus()));
        IBond[] carriers = ect.getCarriers().toArray(new IBond[2]);
        if (ends != null) {
            Configuration.DoubleBond config = ect.getConfigOrder() == 2 ? Configuration.DoubleBond.TOGETHER : (ect.getConfigOrder() == 1 ? Configuration.DoubleBond.OPPOSITE : Configuration.DoubleBond.UNSPECIFIED);
            gb.extendedGeometric(indices.get(ends[0]).intValue(), indices.get(ends[1]).intValue()).configure(indices.get(carriers[0].getOther(ends[0])).intValue(), indices.get(carriers[1].getOther(ends[1])).intValue(), config);
        }
    }
}

