/* *##%
 * Copyright (C) 2010 Code Lutin, Chatellier Eric
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.nuiton.wikitty;

import java.io.Serializable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jgroups.Channel;
import org.jgroups.ChannelClosedException;
import org.jgroups.ChannelException;
import org.jgroups.ChannelNotConnectedException;
import org.jgroups.JChannel;
import org.jgroups.Message;
import org.jgroups.ReceiverAdapter;

/**
 * JGroups notifier.
 * 
 * @author chatellier
 * @version $Revision: 123 $
 * 
 * Last update : $Date: 2010-06-11 16:50:28 +0200 (ven., 11 juin 2010) $
 * By : $Author: echatellier $
 */
public class JGroupsNotifier extends ReceiverAdapter implements WikittyServiceListener {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(JGroupsNotifier.class);

    /** Notifier service reference reference. */
    protected WikittyServiceNotifier ws;

    /** Message type (put, remove, clear...). */
    static enum WikittyJGroupType {
        PUT_WIKITTY,
        REMOVE_WIKITTY,
        CLEAR_WIKITTY,
        PUT_EXTENSION,
        REMOVE_EXTENSION,
        CLEAR_EXTENSION
    }

    static class WikittyJGroupMessage implements Serializable {
        /** serialVersionUID. */
        private static final long serialVersionUID = 1914969328584238081L;

        public WikittyJGroupType type;
        public WikittyServiceEvent event;

        private WikittyJGroupMessage(WikittyJGroupType type, WikittyServiceEvent event) {
            this.type = type;
            this.event = event;
        }

        public String toString() {
            String toString = type + " " + event;
            return toString;
        }
    }

    /**
     * Indique si les objects sont propages (true) vers les autres caches ou
     * simplement supprimes des autres caches (false).
     * 
     * @see WikittyServiceCached#WIKITTY_PROPAGATE_CACHE_OPTION
     */
    protected boolean propagateCache = false;

    /** JGroup channel. */
    protected JChannel channel;

    public JGroupsNotifier(WikittyServiceNotifier ws, String channelName, boolean propagateCache) {
        this.ws = ws;
        this.propagateCache = propagateCache;
        initChannel(channelName);
    }

    /**
     * Init jgroup channel.
     * 
     * @param channelName channel name
     */
    protected void initChannel(String channelName) {
        if (log.isDebugEnabled()) {
            log.debug("Init jgroup communication channel...");
        }

        try {
            // use default udp.xml in classpath
            channel = new JChannel();
            channel.connect(channelName);
            channel.setReceiver(this);

            // don't receive messages sent by myself
            channel.setOpt(Channel.LOCAL, false); 

            if (log.isDebugEnabled()) {
                log.debug("JGroup communication channel initialized to " + channel.getAddressAsString());
            }
        }
        catch (ChannelException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't init jgroup channel");
            }
        }
    }

    /**
     * Send a jgroup message to all other channel member.
     * 
     * @param wikittyMessage message to send
     */
    protected void sendJGroupMessage(WikittyJGroupMessage wikittyMessage) {
        if (log.isInfoEnabled()) {
            log.info("Send message : " + wikittyMessage);
        }
        Message msg = new Message(null, null, wikittyMessage);
        try {
            channel.send(msg);
        } catch (ChannelNotConnectedException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't send jgroup message", eee);
            }
        } catch (ChannelClosedException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't send jgroup message", eee);
            }
        }
    }

    /*
     * @see org.jgroups.ReceiverAdapter#receive(org.jgroups.Message)
     */
    @Override
    public void receive(Message msg) {
        
        Object message = msg.getObject();
        
        if (log.isInfoEnabled()) {
            log.info("Receive message : " + msg.getObject());
        }

        if (message instanceof WikittyJGroupMessage) {
            WikittyJGroupMessage wikittyMessage = (WikittyJGroupMessage)message;
            WikittyJGroupType type = wikittyMessage.type;
            WikittyServiceEvent event = wikittyMessage.event;
            
            //source is transient, add it here :
            event.setSource(ws);
            event.setRemote(true); // received event became remote
            
            switch(type) {
                case PUT_WIKITTY:
                    ws.firePutWikitty(event); break;
                case REMOVE_WIKITTY:
                    ws.fireRemoveWikitty(event); break;
                case CLEAR_WIKITTY:
                    ws.fireClearWikitty(event); break;
                case PUT_EXTENSION:
                    ws.firePutExtension(event); break;
                case REMOVE_EXTENSION:
                    ws.fireRemoveExtension(event); break;
                case CLEAR_EXTENSION:
                    ws.fireClearExtension(event); break;
                default:
                    if (log.isDebugEnabled()) {
                        log.debug("Not managed jgroup message " + wikittyMessage.type);
                    }
            }
        }
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#putWikitty(org.nuiton.wikitty.Wikitty[])
     */
    @Override
    public void putWikitty(WikittyServiceEvent event) {
        if (propagateCache) {
            sendJGroupMessage(new WikittyJGroupMessage(WikittyJGroupType.PUT_WIKITTY, event));
        }
        else {
            if (log.isDebugEnabled()) {
                log.debug("Put wikitty event skipped");
            }
        }
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#removeWikitty(java.lang.String[])
     */
    @Override
    public void removeWikitty(WikittyServiceEvent event) {
        if (propagateCache) {
            sendJGroupMessage(new WikittyJGroupMessage(WikittyJGroupType.REMOVE_WIKITTY, event));
        }
        else {
            if (log.isDebugEnabled()) {
                log.debug("Remove wikitty event skipped");
            }
        }
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#clearWikitty()
     */
    @Override
    public void clearWikitty(WikittyServiceEvent event) {
        if (propagateCache) {
            sendJGroupMessage(new WikittyJGroupMessage(WikittyJGroupType.CLEAR_WIKITTY, event));
        }
        else {
            if (log.isDebugEnabled()) {
                log.debug("Clear wikitty event skipped");
            }
        }
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#putExtension(org.nuiton.wikitty.WikittyExtension[])
     */
    @Override
    public void putExtension(WikittyServiceEvent event) {
        if (propagateCache) {
            sendJGroupMessage(new WikittyJGroupMessage(WikittyJGroupType.PUT_EXTENSION, event));
        }
        else {
            if (log.isDebugEnabled()) {
                log.debug("Put extension event skipped");
            }
        }
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#removeExtension(java.lang.String[])
     */
    @Override
    public void removeExtension(WikittyServiceEvent event) {
        if (propagateCache) {
            sendJGroupMessage(new WikittyJGroupMessage(WikittyJGroupType.REMOVE_EXTENSION, event));
        }
        else {
            if (log.isDebugEnabled()) {
                log.debug("Remove extension event skipped");
            }
        }
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#clearExtension()
     */
    @Override
    public void clearExtension(WikittyServiceEvent event) {
        if (propagateCache) {
            sendJGroupMessage(new WikittyJGroupMessage(WikittyJGroupType.CLEAR_EXTENSION, event));
        }
        else {
            if (log.isDebugEnabled()) {
                log.debug("Clear extension event skipped");
            }
        }
    }
}
