/* *##% 
 * ToPIA :: SOA
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/

package org.nuiton.topia.generator;

import org.apache.commons.lang.StringUtils;
import static org.nuiton.topia.generator.TopiaGeneratorUtil.hasUnidirectionalRelationOnAbstractType;

import java.io.File;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.models.object.ObjectModelGenerator;
import org.nuiton.eugene.models.object.ObjectModelAssociationClass;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelParameter;
import org.nuiton.topia.service.TopiaApplicationService;

/**
* ServiceInterfaceGenerator.java
*
* Created: 14 juin 2007
*
* @author ndupont
* @version $Revision: 1732 $
*
* L'interface du service etend TopiaApplicationService
* 
* Genere les interfaces des services, genere les signatures de toutes les methodes des 
* DAO et des entites utilises. Les signatures de methodes sont modifiees pour eviter les 
* doublons.
* 
* Par exemple :
* 
* findAllByAddress d'un PersonDAO et d'un EmployeeDAO deviennent respectivement 
* findAllPersonByAddress et findAllEmployeeByAddress.
* 
* getTask d'entites Cost et Bill deviennent respectivement getTaskOfBill, getTaskOfCost
* 
* @see TopiaApplicationService
*
* Mise a jour: $Date: 2009-12-20 17:29:38 +0100 (dim. 20 déc. 2009) $
* par : $Author: tchemit $
*  @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.topia.generator.ServiceInterfaceGenerator" 
*/
public class ServiceInterfaceGenerator extends ObjectModelGenerator {

    /**
     * Logger for this class
     */
    private static final Log log = LogFactory
            .getLog(ServiceInterfaceGenerator.class);
    
    private static final String DAO = "dao";
    
	// methodes du service <signature methode>
	private List<String> methods = null;

    @Override
    public String getFilenameForInterface(ObjectModelInterface interfacez) {
        return interfacez.getQualifiedName().replace('.', File.separatorChar) + ".java";
    }

    @Override
    public void generateFromInterface(Writer output, ObjectModelInterface interfacez) throws IOException {

        // reset la liste a chaque interface traitée
        methods = new ArrayList<String>();
        
    	if (!interfacez.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_SERVICE)) {
            return;
        }
    	    	
    	// génèrer l'entête de l'interface du service
        generateInterfaceHeader(output, interfacez);
        
        // génèrer les signatures des méthodes métiers du service
        generateInterfaceOperations(output, interfacez);
        
        // génèrer les signatures des méthodes des DAOs
        generateInheritedInterfaceOperations(output, interfacez);

        // génèrer les signatures des méthodes des entités (getter/setter) 
        generateMethodsGetter(output, interfacez);
        
        // génèrer les signatures des méthodes métier des entités (getter/setter)
        generateAssociatedClassOperations(output, interfacez);
        
output.write("\n");
output.write("    // methodes du service\n");
output.write("    public String[] methods = {\n");
output.write("");
        String comma=" ";
    	for(String method : methods){
output.write("        "+comma+"\""+method+"\"\n");
output.write("");
    		comma=", ";
    	}
output.write("    };\n");
output.write("		 \n");
output.write("} //"+interfacez.getName()+"\n");
output.write("");
    }

    private void generateInterfaceHeader(Writer output, ObjectModelClassifier classifier) throws IOException {
        String copyright = TopiaGeneratorUtil.getCopyright(model);
        String classifierName = classifier.getName();
        if (TopiaGeneratorUtil.notEmpty(copyright)) {
output.write(""+copyright+"\n");
output.write("");
        }
output.write("package "+classifier.getPackageName()+";\n");
output.write("\n");
output.write("import java.util.List;\n");
output.write("import java.util.Collection;\n");
output.write("import org.nuiton.topia.TopiaException;\n");
output.write("import org.nuiton.topia.service.TopiaApplicationService;\n");
output.write("\n");
output.write("");
        if (TopiaGeneratorUtil.hasDocumentation(classifier)) {
        	String documentation = classifier.getDocumentation();
output.write("\n");
output.write("/**\n");
output.write(" * "+documentation+"\n");
output.write(" */\n");
output.write("");
        }
output.write("public interface "+classifierName+" extends ");
        String extendClass = "";
        for (ObjectModelClassifier parent : classifier.getInterfaces()) {
            // pas pour les interfaces de DAO
            if(!parent.hasStereotype(DAO)){
            	extendClass += parent.getQualifiedName();
                extendClass += ", ";    
            }
        }
        
        if (classifier instanceof ObjectModelClass) {
            ObjectModelClass clazz = (ObjectModelClass)classifier;
            for (ObjectModelClass parent : clazz.getSuperclasses()) {
                if (parent.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_SERVICE)) {
                    extendClass += parent.getQualifiedName();
                }
                extendClass += ", ";            
            }
        }
        
output.write(""+extendClass+"TopiaApplicationService {\n");
output.write("");
    }

    private void generateMethodsGetter(Writer output, ObjectModelClassifier classifier) throws IOException {
output.write("\n");
output.write("    public abstract String[] getMethods();\n");
output.write("\n");
output.write("	// methodes utilisant celles des entites\n");
output.write(""); 	
    }
    
    
    private void generateInterfaceOperations(Writer output, ObjectModelClassifier classifier) throws IOException {
        for (ObjectModelOperation op : classifier.getOperations()) {
            String opName = op.getName();
            String opVisibility = op.getVisibility();
            String opType = op.getReturnType();
            String method = opType + " " + opName + "(";
output.write("\n");
output.write("    /** \n");
output.write("     * Methode "+opName+".\n");
output.write("");
            if (TopiaGeneratorUtil.hasDocumentation(op)) {
            	String opDocumentation = op.getDocumentation();
output.write("     * "+opName+" : "+opDocumentation+"\n");
output.write("");
            }
            Collection<ObjectModelParameter> params = (Collection<ObjectModelParameter>)op.getParameters();
            for(ObjectModelParameter param : params) {
                if (TopiaGeneratorUtil.hasDocumentation(param)) {
                	String paramName = param.getName();
                	String paramDocumentation = param.getDocumentation();
output.write("     * @param "+paramName+" "+paramDocumentation+"\n");
output.write("");
                }
            }
output.write("     */\n");
output.write("    "+opVisibility+" "+opType+" "+opName+"(");
            String comma = "";
            for(ObjectModelParameter param : params){
            	String paramName = param.getName();
            	String paramType = param.getType();
                method += comma + paramType + " " + paramName;
output.write(""+comma+""+paramType+" "+paramName+"");
                comma = ", ";
            }
output.write(")");
            Set<String> exceptions = op.getExceptions();
            comma = " throws ";
            for (String exception : exceptions) {
output.write(""+comma+""+exception+"");
                comma = ", ";
            }
output.write(";\n");
output.write("");
        // ajout des methodes
        methods.add(method + ")");
        }
    }
    
    private void generateInheritedInterfaceOperations(Writer output, ObjectModelClassifier classifier) throws IOException {
output.write("\n");
output.write("	// methodes utilisant celles des daos\n");
output.write("");    	
        for (ObjectModelInterface parent : classifier.getInterfaces()) {
            // pour tous les DAOInterface
            if(parent.hasStereotype(DAO)){
            	
            	// récupèrer la classe de l'object model correspondant à la DAO
            	String entityClassName = null;
            	entityClassName = parent.getQualifiedName().replace("DAO", "");
            	if(getModel().hasClass(entityClassName)){
            		ObjectModelClass clazz =  getModel().getClass(entityClassName);
            		String clazzName = clazz.getName();
            		String clazzFQN = clazz.getQualifiedName();
            		
            		if (!clazz.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_ENTITY)) {
            			return;  
            		}

            		// ajouter les methodes à la liste
                    methods.add("void delete"+clazzName+"("+clazzName+" entity)");
                    methods.add(clazzName+" create"+clazzName+"(Object ... properties)");
                    methods.add(clazzName+" update"+clazzName+"("+clazzName+")");
                    methods.add("List<"+clazzName+"> findAll"+clazzName+"()");
                    methods.add(clazzName+" find"+clazzName+"ByTopiaId(String v)");

output.write("\n");
output.write("    /**\n");
output.write("     * Supprime l'entite "+clazzName+" passee en parametre\n");
output.write("     * @param entity l'entite a supprimer\n");
output.write("     */\n");
output.write("	public abstract void delete"+clazzName+"("+clazzFQN+" entity) throws TopiaException;\n");
output.write("	\n");
output.write("    /**\n");
output.write("     * Creer l'entite "+clazzName+" avec les proprietes passees en parametre\n");
output.write("     * @param entity l'entite a mettre a jour\n");
output.write("     */\n");
output.write("    public abstract "+clazzFQN+" create"+clazzName+"(Object ... properties) throws TopiaException;\n");
output.write("    \n");
output.write("    /**\n");
output.write("     * Mise a jour de l'entite "+clazzName+" passee en parametre\n");
output.write("     * @param properties les proprietes de l'entite a creer\n");
output.write("     */\n");
output.write("    public abstract "+clazzFQN+" update"+clazzName+"("+clazzFQN+" entity) throws TopiaException;\n");
output.write("    \n");
output.write("	/**\n");
output.write("     * Retourne tous les "+clazzName+" \n");
output.write("     * @return une liste\n");
output.write("     */\n");
output.write("    public abstract List<"+clazzFQN+"> findAll"+clazzName+"() throws TopiaException;\n");
output.write("    \n");
output.write("    /**\n");
output.write("     * Retourne le "+clazzName+" par son TopiaId \n");
output.write("     * @return le "+clazzName+"\n");
output.write("     */\n");
output.write("    public abstract "+clazzFQN+" find"+clazzName+"ByTopiaId(String v) throws TopiaException;\n");
output.write("");
            	    for (ObjectModelAttribute attr : clazz.getAttributes()) {
            		    if (!attr.isNavigable()) {
            		 	   continue;
                        }

            	        if (!GeneratorUtil.isNMultiplicity(attr)) {
            	            generateNoNMultiplicity(output, attr, clazz, false);
            	        } else {
            	            generateNMultiplicity(output, attr, clazz, false);
            	        }
            	    }

            	    if (clazz instanceof ObjectModelAssociationClass) {
            	        ObjectModelAssociationClass assocClass = (ObjectModelAssociationClass)clazz;
            	        for (ObjectModelAttribute attr : assocClass.getParticipantsAttributes()) {
            	            if (attr != null) {
            	                if (!GeneratorUtil.isNMultiplicity(attr)) {
            	                    generateNoNMultiplicity(output, attr, clazz, true);
            	                } else {
            	                    generateNMultiplicity(output, attr, clazz, true);
            	                }
            	            }
            	        }
            		} 	    	    
            	}
            }
        }
    }

    protected void generateNoNMultiplicity(Writer output, ObjectModelAttribute attr, ObjectModelClass clazz, boolean isAssoc) throws IOException {
    	String attrName = attr.getName();
	    String propertyName = attrName;
	    String attrType = attr.getType();
	    String clazzName = clazz.getName();
	    String clazzFQN = clazz.getQualifiedName();
	    if (!isAssoc && attr.hasAssociationClass()) {
	        propertyName = TopiaGeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName()) + "." + propertyName;
	    }

		// ajouter les methodes à la liste
        methods.add(clazzName+" find"+clazzName+"By"+StringUtils.capitalize(attrName)+"("+attrType+" v)");
        methods.add("List<"+clazzName+"> findAll"+clazzName+"By"+StringUtils.capitalize(attrName)+"("+attrType+" v)");
output.write("\n");
output.write("    /**\n");
output.write("     * Retourne le premier element trouve ayant comme valeur pour l'attribut\n");
output.write("     * "+attrName+" le parametre\n");
output.write("     * @param v la valeur que doit avoir "+attrName+"\n");
output.write("     * @return un element ou null\n");
output.write("     */\n");
output.write("    public abstract "+clazzFQN+" find"+clazzName+"By"+StringUtils.capitalize(attrName)+"("+attrType+" v) throws TopiaException;\n");
output.write("    /**\n");
output.write("     * Retourne les éléments ayant comme valeur pour l'attribut\n");
output.write("     * "+attrName+" le paramêtre\n");
output.write("     * @param v la valeur que doit avoir "+attrName+"\n");
output.write("     * @return une liste\n");
output.write("     */\n");
output.write("    public abstract  List<"+clazzFQN+"> findAll"+clazzName+"By"+StringUtils.capitalize(attrName)+"("+attrType+" v) throws TopiaException;\n");
output.write("");
        if (attr.hasAssociationClass()) {
        	String assocClassName = attr.getAssociationClass().getName();
        	String assocClassFQN = attr.getAssociationClass().getQualifiedName();
            methods.add(clazzName +" find"+clazzName+"By"+StringUtils.capitalize(assocClassName)+"("+assocClassFQN+" value)");
            methods.add("List<"+clazzName+"> findAll"+clazzName+"By"+StringUtils.capitalize(assocClassName)+"("+assocClassFQN+" value)");
output.write("\n");
output.write("    /**\n");
output.write("     * Retourne le premier element trouve ayant comme valeur pour l'attribut\n");
output.write("     * "+TopiaGeneratorUtil.toLowerCaseFirstLetter(assocClassName)+" le parametre\n");
output.write("     * @param value la valeur que doit avoir "+TopiaGeneratorUtil.toLowerCaseFirstLetter(assocClassName)+"\n");
output.write("     * @return un element ou null\n");
output.write("     */\n");
output.write("    public abstract "+clazzFQN+" find"+clazzName+"By"+StringUtils.capitalize(assocClassName)+"("+assocClassFQN+" value) throws TopiaException;\n");
output.write("    /**\n");
output.write("     * Retourne les éléments ayant comme valeur pour l'attribut\n");
output.write("     * "+TopiaGeneratorUtil.toLowerCaseFirstLetter(assocClassName)+" le paramêtre\n");
output.write("     * @param value la valeur que doit avoir "+TopiaGeneratorUtil.toLowerCaseFirstLetter(assocClassName)+"\n");
output.write("     * @return une liste\n");
output.write("     */\n");
output.write("    public abstract List<"+clazzFQN+"> findAll"+clazzName+"By"+StringUtils.capitalize(assocClassName)+"("+assocClassFQN+" value) throws TopiaException;\n");
output.write("");
	    }
	}        
	        
    protected void generateNMultiplicity(Writer output, ObjectModelAttribute attr, ObjectModelClass clazz, boolean isAssoc) throws IOException {
    	String attrName = attr.getName();
    	String attrType = attr.getType();
	    String clazzName = clazz.getName();
	    String clazzFQN = clazz.getQualifiedName();
        methods.add(clazzName + " find"+clazzName+"Contains"+StringUtils.capitalize(attrName)+"("+attrType+" ... v)");
        methods.add("List<"+clazzName+"> findAll"+clazzName+"Contains"+StringUtils.capitalize(attrName)+"("+attrType+" ... v)");
output.write("\n");
output.write("    /**\n");
output.write("     * Retourne le premier element trouve dont l'attribut\n");
output.write("     * "+attrName+" contient le parametre\n");
output.write("     * @param v la valeur que doit contenir "+attrName+"\n");
output.write("     * @return un element ou null\n");
output.write("     */\n");
output.write("    public abstract "+clazzFQN+" find"+clazzName+"Contains"+StringUtils.capitalize(attrName)+"("+attrType+" ... v) throws TopiaException;\n");
output.write("    /**\n");
output.write("     * Retourne les elements trouve dont l'attribut\n");
output.write("     * "+attrName+" contient le parametre\n");
output.write("     * @param v la valeur que doit contenir "+attrName+"\n");
output.write("     * @return une liste\n");
output.write("     */\n");
output.write("    public abstract List<"+clazzFQN+"> findAll"+clazzName+"Contains"+StringUtils.capitalize(attrName)+"("+attrType+" ... v) throws TopiaException;\n");
output.write("");        
	}
    
    private void generateAssociatedClassOperations(Writer output, ObjectModelClassifier classifier) throws IOException {
        for (Iterator i=classifier.getInterfaces().iterator(); i.hasNext();) {
            ObjectModelClassifier parent = (ObjectModelClassifier)i.next();
            // pour tous les DAOInterface
            if(parent.hasStereotype(DAO)){
            	// récupérer la classe de l'object model correspondant à la DAO
            	String entityClassName = null;
            	entityClassName = parent.getQualifiedName().replace("DAO", "");
            	if(getModel().hasClass(entityClassName)){
            		ObjectModelClass clazz =  getModel().getClass(entityClassName);
            		generateFromDAOClass(output, clazz);
            		
            	}
            }		
        }
    }
    
    private void generateFromDAOClass(Writer output, ObjectModelClass clazz) throws IOException {
        if (!clazz.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_ENTITY)) {
            return;
        }

        String clazzName = clazz.getName();
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
        	String attrName = attr.getName();
        	String attrType = attr.getType();
            ObjectModelAttribute reverse = attr.getReverseAttribute();
            if (!attr.isNavigable()
                    && !hasUnidirectionalRelationOnAbstractType(reverse, model)) {
                continue;
            }
            if (!GeneratorUtil.isNMultiplicity(attr)) {
                if (!attr.hasAssociationClass()) {
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                        methods.add("void set"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId, "+attrType+ " "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+")");
                        
output.write("     * "+attrName+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+" La valeur de l'attribut "+attrName+" à positionner.\n");
output.write("     */\n");
output.write("    public abstract void set"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId, "+attrType+" "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+") throws TopiaException;\n");
output.write("\n");
output.write("");
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                        methods.add(attrType+" get"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId)");
output.write("     * "+attrName+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @return La valeur de l'attribut "+attrName+".\n");
output.write("     */\n");
output.write("    public abstract "+attrType+" get"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId) throws TopiaException;\n");
output.write("\n");
output.write("");
                } else {
                    String assocAttrName = TopiaGeneratorUtil.getAssocAttrName(attr);
                    String assocClassName = attr.getAssociationClass().getName();
                    String assocClassFQN = attr.getAssociationClass().getQualifiedName();
                    if (log.isTraceEnabled()) { log.trace("assocAttrName: " + assocAttrName); }
                    methods.add("void set"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+assocClassName+" "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+")");
                    methods.add(assocClassName+" get"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId)");
output.write("    /**\n");
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+" La valeur de l'attribut "+assocClassName+" à positionner.\n");
output.write("     */\n");
output.write("    public abstract void set"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+assocClassFQN+" "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+")  throws TopiaException;\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @return La valeur de l'attribut "+assocClassName+".\n");
output.write("     */\n");
output.write("    public abstract "+assocClassFQN+" get"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId)  throws TopiaException;\n");
output.write("\n");
output.write("");
                }
            } else { //NMultiplicity
                if (!attr.hasAssociationClass()) { //Méthodes remplacées par des add/set sur les classes d'assoc
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                        methods.add("void add"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId, "+attrType+" "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+")");
output.write("     * "+attrName+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+" L'instance de "+attrName+" à ajouter.\n");
output.write("     */\n");
output.write("    public abstract void add"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId, "+attrType+" "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+")  throws TopiaException;\n");
output.write("\n");
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                        methods.add("void addAll"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId, "+(attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+attrType+"> "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+")");;
                        		
output.write("     * "+attrName+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+" Les instances de "+attrName+" à ajouter.\n");
output.write("     */\n");
output.write("    public abstract void addAll"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId, "+(attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+attrType+"> "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+")  throws TopiaException;\n");
output.write("\n");
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                    	methods.add("void set"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId, "+ (attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+attrType+"> "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+")");
output.write("     * "+attrName+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+" La Collection de "+attrName+" à positionner.\n");
output.write("     */\n");
output.write("    public abstract void set"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId, "+(attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+attrType+"> "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+")  throws TopiaException;\n");
output.write("\n");
output.write("    /** \n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                    	methods.add("void remove"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId, "+attrType+ " "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+")");
output.write("     * "+attrName+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+" L'instance de "+attrName+" à retirer.\n");
output.write("     */\n");
output.write("    public abstract void remove"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId, "+attrType+" "+GeneratorUtil.toLowerCaseFirstLetter(attrName)+")  throws TopiaException;\n");
output.write("\n");
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                    	methods.add("void clear"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId)");
output.write("     * "+attrName+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * Vide la Collection de "+attrName+".\n");
output.write("     */\n");
output.write("    public abstract void clear"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId) throws TopiaException;\n");
output.write("\n");
output.write("");
                } else {
                	String assocClassName = attr.getAssociationClass().getName();
                	String assocClassFQN = attr.getAssociationClass().getQualifiedName();
                    String assocAttrName = TopiaGeneratorUtil.getAssocAttrName(attr);
                    if (log.isTraceEnabled()) { log.trace("assocAttrName: " + assocAttrName); }
                	methods.add("void add"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+assocClassName+" "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+")");
                	methods.add("void addAll"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+(attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+assocClassName+"> "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+")");
                	methods.add("void set"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+(attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+assocClassName+"> "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+")");
                	methods.add("void remove"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+assocClassName+" "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+")");
                	methods.add("void clear"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId)");
output.write("    /**\n");
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+" L'instance de "+assocClassName+" à ajouter.\n");
output.write("     */\n");
output.write("    public abstract void add"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+assocClassFQN+" "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+") throws TopiaException;\n");
output.write("\n");
output.write("    /** \n");
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+" Les instances de "+assocClassName+" à ajouter.\n");
output.write("     */\n");
output.write("    public abstract void addAll"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+(attr.isOrdered()?"List":"Collection")+"<"+assocClassFQN+"> "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+") throws TopiaException;\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+" La Collection de "+assocClassName+" à positionner.\n");
output.write("     */\n");
output.write("    public abstract void set"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+(attr.isOrdered()?"List":"Collection")+"<"+assocClassFQN+"> "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+") throws TopiaException;\n");
output.write("\n");
output.write("    /** \n");
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+" L'instance de "+assocClassName+" à retirer.\n");
output.write("     */\n");
output.write("    public abstract void remove"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+assocClassFQN+" "+GeneratorUtil.toLowerCaseFirstLetter(assocClassName)+") throws TopiaException;\n");
output.write("\n");
output.write("    /**\n");
output.write("     * Vide la Collection de "+assocClassName+".\n");
output.write("     */\n");
output.write("    public abstract void clear"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId) throws TopiaException;\n");
output.write("\n");
output.write("");
                }

                if (!attr.hasAssociationClass()) {
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                    	methods.add((attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+attrType+"> get"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId)");
                    	methods.add("int size"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId)");
                    	
output.write("     * "+attrName+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @return La Liste de "+attrName+".\n");
output.write("     */\n");
output.write("    public abstract "+(attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+attrType+"> get"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId) throws TopiaException;\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @return Le nombre d'éléments de la collection "+attrName+".\n");
output.write("     */\n");
output.write("    public abstract int size"+StringUtils.capitalize(attrName)+"Of"+clazzName+"(String topiaId) throws TopiaException;\n");
output.write("\n");
output.write("");
                } else {
                    String assocAttrName = TopiaGeneratorUtil.getAssocAttrName(attr);
                    String assocClassFQN = attr.getAssociationClass().getQualifiedName();
                    String assocClassName = attr.getAssociationClass().getName();
                    if (log.isTraceEnabled()) { log.trace("assocAttrName: " + assocAttrName); }
                    methods.add((attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+assocClassName+"> get"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"Of"+clazzName+"(String topiaId)");
                    methods.add(assocClassName+" get"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+attrType+" value)");
                    methods.add("int size"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId)");
output.write("    /**\n");
output.write("     * @return La liste des attributs "+assocClassName+".\n");
output.write("     */\n");
output.write("    public abstract "+(attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+assocClassFQN+"> get"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId) throws TopiaException;\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @return L'attribut "+assocClassName+" associé à la valeur <code>value</code> de l'attribut "+attrName+".\n");
output.write("     */\n");
output.write("    public abstract "+assocClassFQN+" get"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+attrType+" value) throws TopiaException;\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @return Le nombre d'éléments de la collection "+attrName+".\n");
output.write("     */\n");
output.write("    public abstract int size"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId) throws TopiaException;\n");
output.write("\n");
output.write("");
                }
            }
        }

        //Méthodes d'accès aux attributs d'une classe d'associations
        if (clazz instanceof ObjectModelAssociationClass) {
            ObjectModelAssociationClass assoc = (ObjectModelAssociationClass) clazz;
            for (Iterator i = assoc.getParticipantsAttributes().iterator(); i.hasNext(); ) {
               ObjectModelAttribute attr = (ObjectModelAttribute) i.next();
               if (attr != null) {
                   String type = attr.getType();
                   String name = attr.getName();
                   generateAssociationAccessors(output, clazz, name, type);
                   if (attr.getReverseAttribute() == null) {
                       type = ((ObjectModelClassifier)attr.getDeclaringElement()).getQualifiedName();
                       name = attr.getDeclaringElement().getName();
                       generateAssociationAccessors(output, clazz, name, type);
                   }
               }
            }
        }

        generateInterfaceOperationsOfClass(output, clazz);

    }

    private void generateInterfaceOperationsOfClass(Writer output, ObjectModelClassifier classifier) throws IOException {
        for (Iterator it = classifier.getOperations().iterator(); it.hasNext();) {
            ObjectModelOperation op = (ObjectModelOperation)it.next();
            String method = op.getReturnType()+ " "+op.getName()+"(";
output.write("    /**\n");
output.write("");
            if (TopiaGeneratorUtil.hasDocumentation(op)) {
output.write("     * "+op.getName()+" : "+op.getDocumentation()+"\n");
output.write("");
            }
            Collection<ObjectModelParameter> params = (Collection<ObjectModelParameter>)op.getParameters();
            for(ObjectModelParameter param : params) {
                if(log.isTraceEnabled()) {log.trace("Param" + param);}
output.write("     * @param "+param.getName()+" "+param.getDocumentation()+"\n");
output.write(" "); 
            }
output.write("     */\n");
output.write("    "+op.getVisibility()+" abstract "+op.getReturnType()+" "+op.getName()+"Of"+classifier.getName()+"(String topiaId");
            if(params.size()>0){
output.write(", ");            	
            }
            String vir = "";
            for(ObjectModelParameter param : params){
                if(log.isTraceEnabled()) {log.trace("Param" + param + " vir" + vir);}
                method += vir + param.getType()+" "+param.getName();
output.write(""+vir+""+param.getType()+" "+param.getName()+"");
                vir = ", ";
            }
output.write(")");
            Set<String> exceptions = (Set<String>)op.getExceptions();
            if(exceptions.isEmpty()){
output.write("throws TopiaException");     	
            }
            vir = " throws ";
            for (String exception : exceptions) {
                if(log.isTraceEnabled()) {log.trace("exception" + exception + " vir" + vir);}
output.write(""+vir+""+exception+"");
                vir = ", ";
            }
output.write(";\n");
output.write("\n");
output.write("");
         methods.add(method);
        }
    }

    private void generateAssociationAccessors(Writer output, ObjectModelClass clazz, String name, String type) throws IOException {
    	String clazzName = clazz.getName();
        methods.add("=> "+"void set"+StringUtils.capitalize(name)+"Of"+clazzName+"(String topiaId, "+type+" value)");
        methods.add("=> "+type+" get"+ StringUtils.capitalize(name)+"Of"+clazzName+"(String topiaId)");
output.write("    /**\n");
output.write("     * @param value La valeur de l'attribut "+name+" à positionner.\n");
output.write("     */\n");
output.write("    public abstract void set"+StringUtils.capitalize(name)+"Of"+clazzName+"(String topiaId, "+type+" value) throws TopiaException;\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @return La valeur de l'attribut "+name+".\n");
output.write("     */\n");
output.write("    public abstract "+type+" get"+StringUtils.capitalize(name)+"Of"+clazzName+"(String topiaId) throws TopiaException;\n");
output.write("\n");
output.write("");
    }
 
} //ServiceInterfaceGenerator
