/* *##% 
 * ToPIA :: Persistence
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/
/*******************************************************************************
 * GeneratorUtil.java
 *
 * Created: 13 déc. 2005
 *
 * @author Arnaud Thimel <thimel@codelutin.com>
 *
 * @version $Revision: 1298 $
 *
 * Mise a jour: $Date: 2009-01-15 00:01:45 +0100 (jeu 15 jan 2009) $ par : $Author: tchemit $
 */
package org.nuiton.topia.generator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.nuiton.eugene.AbstractGenerator;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.models.Model;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAssociationClass;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelElement;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelParameter;

/**
 * Classe regroupant divers méthodes utiles pour la génération des entités
 */
public class TopiaGeneratorUtil extends GeneratorUtil {

    /**
     * Stéréotype pour les interfaces devant être générées sous forme de facades
     */
    public final static String STEREOTYPE_FACADE = "facade";
    /**
     * Stéréotype pour les objets devant être générées sous forme d'entités
     */
    public static final String STEREOTYPE_ENTITY = "entity";
    /**
     * Stéréotype pour les objets devant être générées sous forme de DTO
     */
    public static final String STEREOTYPE_DTO = "dto";
    /**
     * Stéréotype pour les objets devant être générées sous forme de bean
     */
    public static final String STEREOTYPE_BEAN = "bean";
    /**
     * Stéréotype pour les interfaces devant être générées sous forme de
     * services
     */
    public static final String STEREOTYPE_SERVICE = "service";
    /**
     * Stéréotype pour les interfaces devant être générées sous forme de DAO
     */
    public static final String STEREOTYPE_DAO = "dao";
    /**
     * Stéréotype pour les attributs à indexer en base
     */
    public static final String STEREOTYPE_INDEXED = "indexed";
    /**
     * Stéréotype pour les collections avec unicité
     */
    public static final String STEREOTYPE_UNIQUE = "unique";
    /**
     * Stéréotype pour les attributs étant des clés primaires
     */
    public static final String STEREOTYPE_PRIMARYKAY = "primaryKey";
    /**
     * Stéréotype pour les attributs considérés comme des tableaux
     */
    public static final String STEREOTYPE_ARRAY = "array";
    /**
     * Tag pour le type de persistence
     */
    public static final String TAG_PERSISTENCE_TYPE = "persistenceType";
    /**
     * Tag pour le nom du champ / entité en BD
     */
    public static final String TAG_DB_NAME = "dbName";
    /**
     * Tag pour le nom du schema en BD
     */
    public static final String TAG_SCHEMA_NAME = "dbSchema";
    /**
     * Tag pour la taille du champ en BD
     */
    public static final String TAG_LENGTH = "length";
    /**
     * Tag pour ajouter une annotation à un champ
     */
    public static final String TAG_ANNOTATION = "annotation";
    /**
     * Tag pour ajouter specifier le copyright d'un fichier
     */
    public static final String TAG_COPYRIGHT = "copyright";
    /**
     * Tag pour specfier le type d'acces a un champ
     */
    public static final String TAG_ACCESS = "access";
    /**
     * Tag pour specfier si on doit générer i18n
     */
    public static final String TAG_I18N_PREFIX = "i18n";
    /**
     * Tag pour ajouter un attribut dans une clef métier
     */
    public static final String TAG_NATURAL_ID = "naturalId";
    /**
     * Tag pour specifier si une clef metier est mutable
     */
    public static final String TAG_NATURAL_ID_MUTABLE = "naturalIdMutable";
    /**
     * Tag pour spécifier la caractère lazy d'une association multiple
     */
    public static final String TAG_LAZY = "lazy";
    /**
     * Tag pour spécifier la caractère fetch d'une association multiple
     */
    public static final String TAG_FETCH = "fetch";
    /**
     * Tag pour spécifier la caractère order-by d'une association multiple
     */
    public static final String TAG_ORDER_BY = "orderBy";
    /**
     * Tag pour spécifier la caractère not-null d'un attribut
     */
    public static final String TAG_NOT_NULL = "notNull";
    /**
     * Tag pour spécifier la caractère embed-xml d'une association
     */
    public static final String TAG_EMBED_XML = "embedXml";
    /**
     * Tag pour configurer l'interface du proxy sur autre chose que l'implementation par defaut.
     * <p/>
     * Par defaut :
     * null > generere le proxy sur l'interface de l'implementation
     * Autre valeur :
     * "none" > laisse la configuration par defaut d'hibernate
     */
    public static final String TAG_PROXY_INTERFACE = "hibernateProxyInterface";
    /**
     * Tag pour spécifier le permissions à la création
     */
    public static final String TAG_SECURITY_CREATE = "securityCreate";
    /**
     * Tag pour spécifier le permissions au chargement
     */
    public static final String TAG_SECURITY_LOAD = "securityLoad";
    /**
     * Tag pour spécifier le permissions à la mise à jour
     */
    public static final String TAG_SECURITY_UPDATE = "securityUpdate";
    /**
     * Tag pour spécifier le permissions à la suppression
     */
    public static final String TAG_SECURITY_DELETE = "securityDelete";
    /**
     * Tag pour specifier de ne pas generer la methode toString
     */
    public static final String TAG_NOT_GENERATE_TO_STRING = "notGenerateToString";
    /**
     * Tag pour specifier de trier les attributs par nom lors de la generation
     */
    public static final String TAG_SORT_ATTRIBUTE = "sortAttribute";
    /**
     * Tag pour specfier si on doit générer la methode getOperator dans les daohelpers )
     */
    public static final String TAG_GENERATE_OPERATOR_FOR_DAO_HELPER = "generateOperatorForDAOHelper";
    /**
     * Type de persistence Hibernate
     */
    public static final String PERSISTENCE_TYPE_HIBERNATE = "hibernate";
    /**
     * Type de persistence LDAP
     */
    public static final String PERSISTENCE_TYPE_LDAP = "ldap";
    /**
     * Type de persistence par défaut (si aucun précisé)
     */
    public static final String PERSISTENCE_TYPE_DEFAULT = PERSISTENCE_TYPE_HIBERNATE;
    /**
     * Propriété des générateurs indiquant le package par défaut
     */
    public static final String PROPERTY_DEFAULT_PACKAGE = "defaultPackage";
    /**
     * Le package par défaut si aucun n'est spécifié
     */
    public static final String DEFAULT_PACKAGE = "org.codelutin.malo";

    /**
     * Renvoie le package par défaut pour le générateur donné
     *
     * @param generator le générateur donné
     * @return le package par défaut du générator donné
     */
    public static String getDefaultPackage(AbstractGenerator<?> generator) {
        String packageName = generator.getProperty(PROPERTY_DEFAULT_PACKAGE);
        if (packageName == null || "".equals(packageName)) {
            packageName = DEFAULT_PACKAGE;
        }
        return packageName;
    }

//    /**
//     * @see GeneratorUtil#hasDocumentation
//     * @deprecated
//     */
//    @Deprecated
//    public static boolean hasDocumentation(ObjectModelElement element) {
//        return notEmpty(element.getDocumentation());
//    }

//    /**
//     * @see GeneratorUtil#notEmpty
//     * @deprecated
//     */
//    @Deprecated
//    public static boolean notEmpty(String s) {
//        return (s != null && !"".equals(s));
//    }

    /**
     * Renvoie l'interface DAO associée à la classe passée en paramètre
     *
     * @param clazz la classe à tester
     * @param model le modele utilisé
     * @return l'interface trouvée ou null sinon
     */
    public static ObjectModelInterface getDAOInterface(ObjectModelClass clazz,
                                                       ObjectModel model) {
        for (Object o : model.getInterfaces()) {
            ObjectModelInterface daoInterface = (ObjectModelInterface) o;
            if (daoInterface.getName().equals(clazz.getName() + "DAO")) {
                if (daoInterface.hasStereotype(STEREOTYPE_DAO)) {
                    return daoInterface;
                }
            }
        }
        return null;
    }

    /**
     * Renvoie le type de persistence pour l'élément donné. Si aucun n'est
     * trouvé, le type par défaut est utilisé
     *
     * @param element l'élément à tester
     * @return le type de persitence pour l'élément donné.
     */
    public static String getPersistenceType(ObjectModelElement element) {
        String tag = element.getTagValue(TAG_PERSISTENCE_TYPE);
        if (tag == null) {
            tag = PERSISTENCE_TYPE_DEFAULT;
        }
        return tag;
    }

    public static String getReverseDBName(ObjectModelAttribute attr) {
        if (attr.getReverseAttribute() != null) {
            return getDBName(attr.getReverseAttribute());
        } else {
            return getDBName(attr) + "_id";
        }
    }

    /**
     * Renvoie le nom BD de l'élement passé en paramètre. Elle se base sur le
     * tag associé si il existe, sinon sur le nom de l'élément
     *
     * @param element l'élément à tester
     * @return le nom de table
     */
    public static String getDBName(ObjectModelElement element) {
        if (element == null) {
            return null;
        }
        if (notEmpty(element.getTagValue(TAG_DB_NAME))) {
            return element.getTagValue(TAG_DB_NAME);
        }
        return toLowerCaseFirstLetter(element.getName());
    }

    /**
     * Cherche et renvoie le schema a utiliser sur cet element, sinon sur le model.
     *
     * @param element l'élément à tester
     * @param model   le modele utilisé
     * @return le nom du schema ou null
     */
    public static String getSchemaName(ObjectModelElement element,
                                       ObjectModel model) {
        return findTagValue(TAG_SCHEMA_NAME, element, model);
    }

    /**
     * Cherche et renvoie le prefixe i18n à utiliser sur cet element, sinon sur le model.
     *
     * @param element l'élément à tester
     * @param model   le modele utilisé
     * @return le prefix i18n ou <code>null</code> si non spécifié
     */
    public static String getI18nPrefix(ObjectModelElement element,
                                       ObjectModel model) {
        return GeneratorUtil.findTagValue(TAG_I18N_PREFIX, element, model);
    }

    /**
     * Cherche et renvoie le prefixe i18n à utiliser sur cet element, sinon sur le model.
     *
     * @param element l'élément à tester
     * @param model   le modele utilisé
     * @return le prefix i18n ou <code>null</code> si non spécifié
     */
    public static boolean shouldgenerateOperatorForDAOHelper(ObjectModelElement element,
                                                             ObjectModel model) {
        String tagValue = GeneratorUtil.findTagValue(TAG_GENERATE_OPERATOR_FOR_DAO_HELPER, element, model);
        boolean generate = GeneratorUtil.notEmpty(tagValue) && Boolean.valueOf(tagValue);
        return generate;
    }

    /**
     * Cherche et renvoie la liste des attributs constituant la clef metier d'une classe.
     *
     * @param clazz la classe à tester
     * @return la liste des attributs de la clef métier ou null si pas de clef métier.
     */
    public static List<String> getNaturalId(ObjectModelClass clazz) {
        String value = clazz.getTagValue(TAG_NATURAL_ID);
        if (value == null || value.trim().isEmpty()) {
            return java.util.Collections.emptyList();
        }
        List<String> result = new ArrayList<String>();
        for (String attribute : value.split(",")) {
            result.add(attribute.trim());
        }
        return result;
    }

    /**
     * Cherche et renvoie la liste des attributs constituant la clef metier d'une classe.
     *
     * @param clazz la classe à tester
     * @param model le modele
     * @return la liste des attributs de la clef métier ou null si pas de clef métier.
     */
    public static boolean generateToString(ObjectModelClass clazz,
                                           ObjectModel model) {
        String value;
        value = model.getTagValue(TAG_NOT_GENERATE_TO_STRING);
        if (value != null && !value.trim().isEmpty()) {
            return false;
        }
        value = clazz.getTagValue(TAG_NOT_GENERATE_TO_STRING);
        return (value == null || value.trim().isEmpty());
    }

    /**
     * Cherche et renvoie la liste des attributs constituant la clef metier d'une classe.
     *
     * @param clazz la classe à tester
     * @param model le modele
     * @return la liste des attributs de la clef métier ou null si pas de clef métier.
     */
    public static boolean sortAttribute(ObjectModelClass clazz,
                                        ObjectModel model) {
        String value;
        value = clazz.getTagValue(TAG_SORT_ATTRIBUTE);
        if (value == null || value.trim().isEmpty() || "false".equals(value.trim())) {
            return false;
        }
        if ("true".equals(value.trim())) {
            return true;
        }

        value = model.getTagValue(TAG_SORT_ATTRIBUTE);
        if (value == null || value.trim().isEmpty() || "false".equals(value.trim())) {
            return false;
        }
        if ("true".equals(value.trim())) {
            return true;
        }
        return true;
    }

    /**
     * Detecte si un attribut fait partie d'une clef metier.
     *
     * @param attribute l'attribut à tester
     * @return <code>true</code> si l'attribut fait partie d'une clef metier, <code>false</cdoe> sinon.
     */
    public static boolean isNaturalId(ObjectModelAttribute attribute) {
        String value = attribute.getTagValue(TAG_NATURAL_ID);
        if (!GeneratorUtil.notEmpty(value)) {
            // valeur null, donc pas positionnee
            return false;
        }
        try {
            return Boolean.valueOf(value.trim());
        } catch (Exception e) {
            // on a pas reussi a convertir en boolean.
            //todo peut-être declancher une exception ?
            return false;
        }
    }

    /**
     * Cherches et renvoie le copyright a utiliser sur le model.
     *
     * @param model le modele utilisé
     * @return le texte du copyright ou null
     */
    public static String getCopyright(Model model) {
        return findTagValue(TAG_COPYRIGHT, null, model);
    }

//    /**
//     * @see GeneratorUtil#findTagValue
//     * @deprecated
//     */
//    @Deprecated
//    public static String findTagValue(String tagName,
//            ObjectModelElement element, Model model) {
//        if (element == null) {
//            if (model != null) {
//                if (notEmpty(model.getTagValue(tagName))) {
//                    return model.getTagValue(tagName);
//                }
//            }
//            return null;
//        }
//        if (notEmpty(element.getTagValue(tagName))) {
//            return element.getTagValue(tagName);
//        }
//        //On va chercher sur l'element declarant
//        return findTagValue(tagName, element.getDeclaringElement(), model);
//    }

    public static <Type extends ObjectModelElement> Collection<Type> getElementsWithStereotype(
            Collection<Type> elements, String... stereotypes) {
        Collection<Type> result = new ArrayList<Type>();
        for (Type element : elements) {
            if (hasStereotypes(element, stereotypes)) {
                result.add(element);
            }
        }
        return result;
    }

    public static boolean hasStereotypes(ObjectModelElement element,
                                         String... stereotypes) {
        for (String stereotype : stereotypes) {
            if (!element.hasStereotype(stereotype)) {
                return false;
            }
        }
        return true;
    }

    public static String getPrimaryKeyAttributesListDeclaration(
            ObjectModelClass clazz, boolean includeName) {
        String attributes = "";
        for (ObjectModelAttribute attr : getElementsWithStereotype(clazz.getAttributes(), STEREOTYPE_PRIMARYKAY)) {
            attributes += attr.getType();
            if (includeName) {
                attributes += " " + attr.getName();
            }
            attributes += ", ";
        }
        if (attributes.length() > 0) {
            attributes = attributes.substring(0, attributes.length() - 2);
        }
        return attributes;
    }

//    public static String capitalize(String s) {
//        return StringUtils.capitalize(s);
//    }

    public static boolean isAssociationClassDoublon(ObjectModelAttribute attr) {
        return (attr.getReverseAttribute() != null) && (attr.getDeclaringElement().equals(attr.getReverseAttribute().getDeclaringElement())) && (!GeneratorUtil.isFirstAttribute(attr));
    }

    /**
     * Renvoie le nom de l'attribut de classe d'association en fonction des cas:
     * Si l'attribut porte le même nom que le type (extrémité inverse de
     * l'association), on lui ajoute le nom de la classe d'association
     *
     * @param attr l'attribut a traiter
     * @return le nom de l'attribut de classe d'association
     */
    public static String getAssocAttrName(ObjectModelAttribute attr) {
        String typeName = attr.getType().substring(
                attr.getType().lastIndexOf(".") + 1);
        String result = attr.getName();
        if (attr.getName().equalsIgnoreCase(typeName)) {
            result += StringUtils.capitalize(attr.getAssociationClass().getName());
        }
        return result;
    }

    public static String getDOType(ObjectModelElement elem, ObjectModel model) {
        String type = elem.getName();
        if (elem instanceof ObjectModelAttribute) {
            type = ((ObjectModelAttribute) elem).getType();
        }
        if (elem instanceof ObjectModelClass) {
            type = ((ObjectModelClass) elem).getQualifiedName();
        }
        return getDOType(type, model);
    }

    public static String getDOType(String type, ObjectModel model) {
        if (!model.hasClass(type)) {
            return type;
        }
        ObjectModelClass clazz = model.getClass(type);
        if (clazz.hasStereotype(STEREOTYPE_ENTITY)) {
            if (shouldBeAbstract(clazz)) {
                type += "Abstract";
            } else {
                type += "Impl";
            }
        }
        return type;
    }

    private static final Set<String> numberTypes = new HashSet<String>();
    private static final Set<String> textTypes = new HashSet<String>();
    private static final Set<String> booleanTypes = new HashSet<String>();
    private static final Set<String> primitiveTypes = new HashSet<String>();
    private static final String VOID_TYPE = "void";

    static {
        numberTypes.add("byte");
        numberTypes.add("java.lang.Byte");
        numberTypes.add("Byte");
        numberTypes.add("short");
        numberTypes.add("java.lang.Short");
        numberTypes.add("Short");
        numberTypes.add("int");
        numberTypes.add("java.lang.Integer");
        numberTypes.add("Integer");
        numberTypes.add("long");
        numberTypes.add("java.lang.Long");
        numberTypes.add("Long");
        numberTypes.add("float");
        numberTypes.add("java.lang.Float");
        numberTypes.add("Float");
        numberTypes.add("double");
        numberTypes.add("java.lang.Double");
        numberTypes.add("Double");

        textTypes.add("char");
        textTypes.add("java.lang.Char");
        textTypes.add("Char");
        textTypes.add("java.lang.String");
        textTypes.add("String");

        booleanTypes.add("boolean");
        booleanTypes.add("java.lang.Boolean");
        booleanTypes.add("Boolean");

        primitiveTypes.addAll(numberTypes);
        primitiveTypes.addAll(textTypes);
        primitiveTypes.addAll(booleanTypes);
    }

    public static boolean isNumericType(ObjectModelAttribute attr) {
        return numberTypes.contains(attr.getType());
    }

    public static boolean isTextType(ObjectModelAttribute attr) {
        return textTypes.contains(attr.getType());
    }

    public static boolean isDateType(ObjectModelAttribute attr) {
        return "java.util.Date".equals(attr.getType());
    }

    public static boolean isBooleanType(ObjectModelAttribute attr) {
        return booleanTypes.contains(attr.getType());
    }

    public static boolean isPrimitiveType(ObjectModelAttribute attr) {
        return primitiveTypes.contains(attr.getType());
    }

    /**
     * Indique si la classe specifiee n'a aucune ou que des methodes abstraites
     *
     * @param clazz l'instance de ObjectModelClass
     * @return true si la classe n'a que des operations abstraite ou aucune
     *         operation
     */
    public static boolean hasNothingOrAbstractMethods(ObjectModelClass clazz) {
        boolean result = true;
        Iterator<?> operations = clazz.getOperations().iterator();
        while (result && operations.hasNext()) {
            ObjectModelOperation op = (ObjectModelOperation) operations.next();
            result = op.isAbstract();
        }
        return result;
    }

    /**
     * Indique si la classe specifiee devrait etre abstraite
     *
     * @param clazz l'instance de ObjectModelClass
     * @return true dans ce cas, false sinon
     */
    public static boolean shouldBeAbstract(ObjectModelClass clazz) {
        return clazz != null && (clazz.isAbstract() && hasNothingOrAbstractMethods(clazz));
    }

    /**
     * <p>
     * Cette méthode permet de détecter si
     * - l'attribut représente une relation 1-n
     * - cette relation est unidirectionnelle
     * - le type de l'attribut représente un entité
     * - cette entité a des sous-classes dans le modèle
     * <p/>
     * Ce cas correspond à une incompatibilité d'Hibernate qui nous oblige a
     * adopter un comportement particulier.
     * </p>
     *
     * @param attr  l'attribut a tester
     * @param model le model
     * @return true si et seulement si il s'agit bien de ce type de relation
     */
    public static boolean hasUnidirectionalRelationOnAbstractType(
            ObjectModelAttribute attr, ObjectModel model) {
        ObjectModelAttribute reverse = attr.getReverseAttribute();
        //relation 1-n
        if (reverse != null && isNMultiplicity(attr) && !isNMultiplicity(reverse)) {
            //Pas de navigabilité
            if (!reverse.isNavigable()) {
                //Il s'agit d'une entity
                ObjectModelClass clazz = model.getClass(attr.getType());
                if (clazz != null && clazz.hasStereotype(STEREOTYPE_ENTITY)) {
                    //Cette classe a des sous-classes dans le modèle
                    for (ObjectModelClass subClass : model.getClasses()) {
                        if (subClass.getSuperclasses().contains(clazz)) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }

    /**
     * Renvoie le nom unique de table pour une relation ManyToMany en fonction
     * de l'attribut <code>attr</code>
     * <p/>
     * Plusieurs cas de figure:
     * <li>
     *
     * @param attr l'attribut servant de base au calcul du nom
     * @return le nom de la table
     */
    public static String getManyToManyTableName(ObjectModelAttribute attr) {
        String result;

        if (attr.hasAssociationClass()) {
            result = TopiaGeneratorUtil.getDBName(attr.getAssociationClass());
        } else {
            String name = attr.getName();
            String revers = attr.getReverseAttributeName();

            if (name.compareToIgnoreCase(revers) < 0) {
                result = name + "_" + revers;
            } else {
                result = revers + "_" + name;
            }
        }
        //        String result;
        //        if (!Util.isFirstAttribute(attr)) {
        //            result = attr.getDeclaringElement().getName() + "_" + attr.getReverseAttribute().getDeclaringElement().getName();
        //        } else {
        //            result = attr.getReverseAttribute().getDeclaringElement().getName() + "_" + attr.getDeclaringElement().getName();
        //        }
        return result.toLowerCase();
    }

    /**
     * Renvoie le type d'interface à utiliser en fonction de l'attribut
     *
     * @param attr l'attribut a traiter
     * @return String
     */
    public static String getNMultiplicityInterfaceType(ObjectModelAttribute attr) {
        if (attr.hasStereotype(STEREOTYPE_UNIQUE)) {
            return Set.class.getName();
        } else if (attr.isIndexed() || attr.isOrdered()) {
            return List.class.getName();
        }
        return Collection.class.getName();
    }

    /**
     * Renvoie le type d'objet (instance) à utiliser en fonction de l'attribut
     *
     * @param attr l'attribut a traiter
     * @return String
     */
    public static String getNMultiplicityObjectType(ObjectModelAttribute attr) {
        if (attr.hasStereotype(STEREOTYPE_UNIQUE)) {
            return HashSet.class.getName();
        } else if (attr.isIndexed() || attr.isOrdered()) {
            //On considère qu'on ne sait pas traiter vraiment l'attribut "ordered"
            //  puisqu'on va conserver l'ordre d'insertion, et non un ordre en
            //  fonction d'un élément donné. Donc on renvoi une ArrayList
            return ArrayList.class.getName();
        }
        LinkedList.class.getName();
        return ArrayList.class.getName();
    }

    /**
     * Renvoie le type d'interface à utiliser en fonction de l'attribut
     *
     * @param attr l'attribut a traiter
     * @return String
     */
    public static String getNMultiplicityHibernateType(ObjectModelAttribute attr) {
        if (attr.hasStereotype(STEREOTYPE_UNIQUE)) {
            return "set";
        } else if (attr.isIndexed()) {
            return "list";
        }
        //attr.isOrdered() - On génère le ordered en bag  
        return "bag";
    }

    /**
     * Obtain the list of entities classes with the possibility to sort the result.
     *
     * @param model the current model to scan
     * @param sort  flag to allow sort the result
     * @return the list of filtred classes by their stereotype
     */
    public static List<ObjectModelClass> getEntityClasses(ObjectModel model,
                                                          boolean sort) {
        return getClassesByStereotype(STEREOTYPE_ENTITY, model, sort);
    }

    /**
     * Obtain the list of classes for a given stereotype with the possibility to sort the result.
     *
     * @param stereotype filter stereotype
     * @param model      the current model to scan
     * @param sort       flag to allow sort the result
     * @return the list of filtred classes by their stereotype
     */
    public static List<ObjectModelClass> getClassesByStereotype(
            String stereotype, ObjectModel model, boolean sort) {
        List<ObjectModelClass> classes = new ArrayList<ObjectModelClass>();
        for (ObjectModelClass clazz : model.getClasses()) {
            if (clazz.hasStereotype(stereotype)) {
                classes.add(clazz);
            }
        }
        if (sort && !classes.isEmpty()) {
            java.util.Collections.sort(classes,
                    new java.util.Comparator<ObjectModelClass>() {

                        @Override
                        public int compare(ObjectModelClass o1,
                                           ObjectModelClass o2) {
                            return o1.getQualifiedName().compareTo(
                                    o2.getQualifiedName());
                        }
                    });
        }
        return classes;
    }

    /**
     * Detecte si la clef metier d'une classe est mutable ou pas.
     * <p/>
     * On respecte la valeur par defaut d'hibernate, à savoir que par default une clef metier est non mutable.
     *
     * @param clazz la classe a tester
     * @return <code>true</code> si le tag value a ete positionne sur la classe via le tag
     *         {@link #TAG_NATURAL_ID_MUTABLE}, , <code>false</code> sinon.
     */
    public static boolean isNaturalIdMutable(ObjectModelClass clazz) {
        String value = clazz.getTagValue(TAG_NATURAL_ID_MUTABLE);
        if (!notEmpty(value)) {
            // valeur null, donc par default positionnee
            return false;
        }
        try {
            return Boolean.valueOf(value.trim());
        } catch (Exception e) {
            // on a pas reussi a convertir en boolean.
            //todo peut-être declancher une exception ?
            return false;
        }
    }

    /**
     * Obtain the list of fqn of object involed in the given class.
     *
     * @param aClass       the clazz to inspect
     * @param incomingFqns incoming fqns
     * @return the list of fqn of attributes
     */
    public static List<String> getImports(ObjectModelClass aClass, String... incomingFqns) {
        Set<String> tmp = new HashSet<String>();
        tmp.addAll(Arrays.asList(incomingFqns));
        getImports(aClass, tmp);
        List<String> result = cleanImports(aClass.getPackageName(), tmp);
        return result;
    }

    /**
     * Obtain the list of fqn of object involed in the given interface.
     *
     * @param anInterface  the interface to inspect
     * @param incomingFqns incoming fqns
     * @return the list of fqn of attributes
     */
    public static List<String> getImports(ObjectModelInterface anInterface, String... incomingFqns) {
        Set<String> tmp = new HashSet<String>();
        tmp.addAll(Arrays.asList(incomingFqns));
        getImports(anInterface, tmp);
        List<String> result = cleanImports(anInterface.getPackageName(), tmp);
        return result;
    }

    public static String getSimpleName(String fqn) {
        int lasIndex = fqn.lastIndexOf(".");
        if (lasIndex == 1) {
            // primitive type
            return fqn;
        }
        return fqn.substring(lasIndex + 1);
        /*if (lasIndex == aClass.getPackageName().length()) {
        // same package
        return fqn.substring(lasIndex + 1);
        }

        return fqn;*/
    }

    /**
     * Obtain the list of fqn of object involed in the given class.
     *
     * @param aClass the class to inspect
     * @param fqns   where to store found fqns
     */
    protected static void getImports(ObjectModelClass aClass, Set<String> fqns) {
        // scan attributes
        for (ObjectModelAttribute attr : aClass.getAttributes()) {
            fqns.add(attr.getType());
            if (isNMultiplicity(attr)) {
                String collectionType = getNMultiplicityInterfaceType(attr);
                fqns.add(collectionType);
                String collectionObject = getNMultiplicityObjectType(attr);
                fqns.add(collectionObject);
            }
        }
        for (ObjectModelAttribute attribute : aClass.getAllOtherAttributes()) {
            fqns.add(attribute.getType());
        }
        // scan associations
        if (aClass instanceof ObjectModelAssociationClass) {
            ObjectModelAssociationClass assoc = (ObjectModelAssociationClass) aClass;
            for (ObjectModelAttribute attr : assoc.getParticipantsAttributes()) {
                if (attr == null) {
                    continue;
                }
                fqns.add(attr.getType());
                if (isNMultiplicity(attr)) {
                    String collectionType = getNMultiplicityInterfaceType(attr);
                    fqns.add(collectionType);
                    String collectionObject = getNMultiplicityObjectType(attr);
                    fqns.add(collectionObject);
                }
            }
        }
        // scan operations
        for (ObjectModelOperation operation : aClass.getOperations()) {
            getImports(operation, fqns);
        }
        // scan super interfaces
        for (ObjectModelInterface modelInterface : aClass.getInterfaces()) {
            fqns.add(modelInterface.getQualifiedName());
            getImports(modelInterface, fqns);
        }
        // scan super classes
        for (ObjectModelClass modelClass : aClass.getSuperclasses()) {
            fqns.add(modelClass.getQualifiedName());
            getImports(modelClass);
        }
    }

    /**
     * Obtain the list of fqn of object involed in the given interface.
     *
     * @param anInterface the interface to inspect
     * @param fqns        where to store found fqns
     */
    protected static void getImports(ObjectModelInterface anInterface, Set<String> fqns) {
        // scan operations
        for (ObjectModelOperation operation : anInterface.getOperations()) {
            getImports(operation, fqns);
        }
        // scan super interfaces
        for (ObjectModelInterface modelInterface : anInterface.getInterfaces()) {
            fqns.add(modelInterface.getQualifiedName());
            getImports(modelInterface, fqns);
        }
    }

    /**
     * Obtain the fqn's list of all involed type in a givne operation.
     *
     * @param operation operation to inspect
     * @param fqns      where to store found fqns
     */
    protected static void getImports(ObjectModelOperation operation, Set<String> fqns) {
        String fqn = operation.getReturnType();
        fqns.add(fqn);
        for (ObjectModelParameter parameter : operation.getParameters()) {
            fqns.add(parameter.getType());
        }
    }

    /**
     * Clean a set of fqns, transform it into a {@link List} and sort it.
     *
     * @param packageName the current package name
     * @param fqns        the dirty set of fqns
     * @return the sorted cleaned list of fqns.
     */
    protected static List<String> cleanImports(String packageName, Set<String> fqns) {
        fqns.removeAll(primitiveTypes);
        fqns.remove(VOID_TYPE);
        int packageLength = packageName.length();
        List<String> genericType = new ArrayList<String>();
        for (Iterator<String> it = fqns.iterator(); it.hasNext();) {
            String fqn = it.next();
            int lastIndex = fqn.lastIndexOf(".");
            if (lastIndex == packageLength && fqn.startsWith(packageName)) {
                // same package
                it.remove();
                continue;
            }
            int genericIndex = fqn.indexOf('<');
            if (genericIndex != -1) {
                genericType.add(fqn.substring(0, genericIndex));
                it.remove();
            }
        }
        fqns.addAll(genericType);

        ArrayList<String> result = new ArrayList<String>(fqns);
        java.util.Collections.sort(result);
        return result;
    }

    /**
     * Convertit un nom de variable en nom de constante.
     *
     * @param variableName le nom de variable a convertir
     * @return le nom de la constante à partir du nom de la variable
     */
    public static String convertVariableNameToConstantName(String variableName) {
        //TODO Faire des tests pour savoir si variableName est non null et valide
        //TODO Ameliorer l'algo pour tenir compte des caractères non alpha
        //TODO pour le moment cela convient, donc...
        StringBuilder buffer = new StringBuilder();
        boolean lastCarIsUp = false;
        for (int i = 0, j = variableName.length(); i < j; i++) {
            char c = variableName.charAt(i);
            boolean carIsUp = Character.isUpperCase(c);
            if (i > 0 && !lastCarIsUp && carIsUp) {
                // ajout d'un _
                buffer.append('_');
            }
            if (carIsUp) {
                buffer.append(c);
            } else {
                buffer.append(Character.toUpperCase(c));
            }
            lastCarIsUp = carIsUp;
        }
        return buffer.toString();
    }
} // GeneratorUtil

