package org.nuiton.util.version;

/*
 * #%L
 * Nuiton Utils
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

/**
 * Useful class around {@link Version}.
 *
 * Created on 8/23/14.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 3.0-rc-6
 */
public class Versions {

    /**
     * Shortcut method to get a version from his string representation.
     *
     * @param version string representation of the version
     * @return converted version from the string representation
     */
    public static Version valueOf(String version) {
        Version v = VersionBuilder.create(version).build();
        return v;
    }

    /**
     * Tests if two versions are equals.
     *
     * @param version0 the first version
     * @param version1 the second version
     * @return {@code true} if versions are equals, {@code false} otherwise.
     */
    public static boolean equals(String version0, String version1) {
        Version v0 = valueOf(version0);
        Version v1 = valueOf(version1);
        boolean result = v0.equals(v1);
        return result;
    }

    /**
     * Tests if the first version is smaller than the second version.
     *
     * @param version0 the first version
     * @param version1 the second version
     * @return {@code true} if {@code version0} is before {@code version1},
     * {@code false} otherwise.
     */
    public static boolean smallerThan(String version0, String version1) {
        Version v0 = valueOf(version0);
        Version v1 = valueOf(version1);
        boolean result = v0.before(v1);
        return result;
    }

    /**
     * Tests if the first version is greater than the second version.
     *
     * @param version0 the first version
     * @param version1 the second version
     * @return {@code true} if {@code version0} is after {@code version1},
     * {@code false} otherwise.
     */
    public static boolean greaterThan(String version0, String version1) {
        Version v0 = valueOf(version0);
        Version v1 = valueOf(version1);
        boolean result = v0.after(v1);
        return result;
    }

    /**
     * Create a version from the given one and set to it the {@code snapshot} state to {@code true}.
     *
     * @param version version to clone
     * @return the cloned version with the {@code snapshot} state to {@code true}
     * @throws IllegalArgumentException if {@code snapshot} state is already set to {@code true} on
     *                                  the given {@code version}.
     */
    public static Version addSnapshot(Version version) {
        if (version.isSnapshot()) {
            throw new IllegalArgumentException(
                    "version " + version + "is already a snapshot");
        }
        Version result = VersionBuilder.create(version).setSnapshot(true).build();
        return result;
    }

    /**
     * Create a version from the given one and set to it the {@code snapshot} state to {@code false}.
     *
     * @param version version to clone
     * @return the cloned version with the {@code snapshot} state to {@code true}
     * @throws IllegalArgumentException if {@code snapshot} state is already set to {@code false} on
     *                                  the given {@code version}
     */
    public static Version removeSnapshot(Version version) {
        if (!version.isSnapshot()) {
            throw new IllegalArgumentException(
                    "version " + version + "is already a snapshot");
        }
        Version result = VersionBuilder.create(version).setSnapshot(false).build();
        return result;
    }
}
