/*
 * *##% 
 * JRedmine maven plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.jredmine.plugin;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.apache.maven.plugin.MojoFailureException;
import org.nuiton.jredmine.model.I18nAble;
import org.nuiton.jredmine.model.IdAble;
import org.nuiton.jredmine.model.RedmineModelEnum;
import org.nuiton.jredmine.model.RedmineModelScope;
import org.nuiton.jredmine.RedmineServiceImplementor;

/**
 * Display in console some data from redmine's server.
 *
 * @goal display-data
 *
 * @author tchemit
 * @since 1.0.0
 */
public class DisplayDataMojo extends AbstractRedmineMojo {

    /**
     * The data types to display : {@code issueSatus}, {@code issuePriority},
     * {@code issueCategory}, {@code tracker} and {@code user}.
     * <p/>
     * {@code issueStatus} and {@code issuePriority} are common for all
     * projects, so for those datas, no extrat configuration is required.
     * <p/>
     * {@code isseCategoriy}, {@code tracker} and {@code user} are project
     * specific, so you have to fill also the {@code projectId} parameter to
     * obtain data for those types.
     * <p/>
     * Note : you can specifiy several types separated by comma
     *
     * @parameter expression="${redmine.types}"
     * @required
     * @since 1.0.0
     */
    protected String types;
    /**
     * list of actions to perform
     */
    protected List<RedmineModelEnum> actions;

    public DisplayDataMojo() {
        super(false, false, false);
    }

    @Override
    protected boolean checkRunOnceDone() {
        return false;
    }

    @Override
    protected boolean isGoalSkip() {
        return false;
    }

    @Override
    protected boolean isRunOnce() {
        return false;
    }

    @Override
    protected boolean init() throws Exception {

        boolean init = super.init();
        if (init) {

            actions = getDownloadActions(types,
                    RedmineModelEnum.issueStatus,
                    RedmineModelEnum.issuePriority,
                    RedmineModelEnum.issueCategory,
                    RedmineModelEnum.tracker,
                    RedmineModelEnum.user);

            if (actions.isEmpty()) {
                // no data to treate
                getLog().warn("no data types detected, you must fill the required parameter dataTypes");
                init = false;
            }
        }
        return init;
    }

    @Override
    protected void doAction() throws Exception {

        StringBuilder buffer = new StringBuilder("\n");
        for (RedmineModelEnum entry : actions) {

            RedmineModelScope scope = entry.getScope();

            Object[] datas = ((RedmineServiceImplementor) service).getDatas(entry.getModelType(), projectId, versionId);
            buffer.append("\nValues of '").append(entry).append('\'');
            if (scope == RedmineModelScope.project || scope == RedmineModelScope.version) {
                buffer.append(" for project ").append(projectId);
            }
            buffer.append('\n');
            for (Object data : datas) {
                buffer.append("  - ").append(((IdAble) data).getId()).append(" = ").append(((I18nAble) data).getName()).append('\n');
            }
        }
        System.out.println(buffer.toString());
    }

    protected List<RedmineModelEnum> getDownloadActions(String dataTypes, RedmineModelEnum... universe) throws MojoFailureException {

        List<RedmineModelEnum> results = new ArrayList<RedmineModelEnum>();

        List<RedmineModelEnum> universeList = new ArrayList<RedmineModelEnum>(Arrays.asList(universe));

        for (String dataType : dataTypes.split(",")) {

//            if (dataType.endsWith("s")) {
//                dataType = dataType.substring(0, dataType.length() - 1);
//            }

            RedmineModelEnum result;
            try {
                result = RedmineModelEnum.valueOf(dataType.trim());
            } catch (Exception e) {
                throw new MojoFailureException("could not obtain data type " + dataType + " in " + Arrays.toString(RedmineModelEnum.values()));
            }
            RedmineModelScope scope = result.getScope();

            if (scope == RedmineModelScope.project || scope == RedmineModelScope.version) {

                // check projectId used
                if (projectId == null || projectId.isEmpty()) {
                    throw new MojoFailureException("could not use the data type " + result + " since it is not a project requires and projectId was not filled");
                }
            }
            if (scope == RedmineModelScope.version) {
                // check versionId use
                if (versionId == null || versionId.isEmpty()) {
                    throw new MojoFailureException("could not use the data type " + result + " since it a version requires and versionId was not filled");
                }
            }

            if (!universeList.isEmpty() && !universeList.contains(result)) {
                // can not accept these value
                getLog().warn("the type " + result + " is not authorized, universe : " + universeList);
                continue;
            }
            results.add(result);
        }
        return results;
    }
}
