package org.nuiton.jredmine.rest;
/*
 * #%L
 * JRedmine :: Client
 * $Id: RedmineRequestFactory.java 320 2012-07-19 09:09:40Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.4/jredmine-client/src/main/java/org/nuiton/jredmine/rest/RedmineRequestFactory.java $
 * %%
 * Copyright (C) 2009 - 2012 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Strings;
import org.nuiton.io.rest.AbstractRequestFactory;
import org.nuiton.io.rest.RestMethod;
import org.nuiton.io.rest.RestRequestBuilder;
import org.nuiton.jredmine.model.Attachment;
import org.nuiton.jredmine.model.ModelHelper;
import org.nuiton.jredmine.model.News;
import org.nuiton.jredmine.model.TimeEntry;
import org.nuiton.jredmine.model.Version;
import org.nuiton.jredmine.model.VersionStatusEnum;

import java.io.File;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * Request factory for the redmine client.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4
 */
public class RedmineRequestFactory extends AbstractRequestFactory {

    public static String getRequestScope(RestRequestBuilder requestBuilder) {
        String scope;
        if (requestBuilder instanceof DefaultRequestBuilder) {

            scope = ((DefaultRequestBuilder) requestBuilder).getScope();
        } else {
            scope = null;
        }
        return scope;
    }

    @Override
    public void addDefaultRequests() {

        // misc requests

        addRequestBuilder(new DefaultRequestBuilder(ModelHelper.PING_REQUEST_NAME, RestMethod.GET, "jredmine", "ping"));
        addRequestBuilder(new DefaultRequestBuilder(ModelHelper.LOGOUT_REQUEST_NAME, RestMethod.GET, "jredmine", "logout"));
        addRequestBuilder(new DefaultRequestBuilder(ModelHelper.LOGIN_REQUEST_NAME, RestMethod.POST, "jredmine", "login") {

            private static final long serialVersionUID = 1L;

            @Override
            public void checkRequestArgs(Object... args) {
                checkRequestArgs(2, "userName, password", args);
            }

            @Override
            public String[] getParameters(Object... args) {
                String login = (String) args[0];
                String password = (String) args[1];
                return new String[]{"username", login, "password", password};
            }
        });

        addRequestBuilder(new DefaultRequestBuilder(ModelHelper.LOGIN_BY_API_KEY_REQUEST_NAME, RestMethod.POST, "jredmine", "login") {

            private static final long serialVersionUID = 1L;

            @Override
            public void checkRequestArgs(Object... args) {
                checkRequestArgs(1, "apiKey", args);
            }

//            @Override
//            public Map<String, String> getHeaders(Object... args) {
//                Map<String, String> headers = Maps.newHashMap();
//                String apiKey = (String) args[0];
//                headers.put("X-Redmine-API-Key", apiKey);
//                return headers;
//            }
        });

        // data with no scope requests

        addRequestBuilder(new DefaultRequestBuilder(ModelHelper.GET_ALL_PROJECT_REQUEST_NAME, RestMethod.GET, "jredmine", "get_projects.xml"));
        addRequestBuilder(new DefaultRequestBuilder(ModelHelper.GET_USER_PROJECTS_REQUEST_NAME, RestMethod.GET, "jredmine", "get_user_projects.xml"));
        addRequestBuilder(new DefaultRequestBuilder(ModelHelper.GET_ALL_ISSUE_STATUS_REQUEST_NAME, RestMethod.GET, "jredmine", "get_issue_statuses.xml"));
        addRequestBuilder(new DefaultRequestBuilder(ModelHelper.GET_ALL_ISSUE_PRIORITY_REQUEST_NAME, RestMethod.GET, "jredmine", "get_issue_priorities.xml"));

        // data with project scope requests

        addRequestBuilder(new ProjectScopeRequestBuilder(ModelHelper.GET_PROJECT_REQUEST_NAME, RestMethod.GET, "jredmine", "get_project.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(ModelHelper.GET_ALL_PROJECT_ISSUES_REQUEST_NAME, RestMethod.GET, "jredmine", "get_project_issues.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(ModelHelper.GET_ALL_PROJECT_OPENED_ISSUES_REQUEST_NAME, RestMethod.GET, "jredmine", "get_project_opened_issues.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(ModelHelper.GET_ALL_PROJECT_CLOSED_ISSUES_REQUEST_NAME, RestMethod.GET, "jredmine", "get_project_closed_issues.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(ModelHelper.GET_ALL_VERSION_REQUEST_NAME, RestMethod.GET, "jredmine", "get_project_versions.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(ModelHelper.GET_ALL_ISSUE_CATEGORY_REQUEST_NAME, RestMethod.GET, "jredmine", "get_issue_categories.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(ModelHelper.GET_ALL_TRACKER_REQUEST_NAME, RestMethod.GET, "jredmine", "get_project_trackers.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(ModelHelper.GET_ALL_USER_REQUEST_NAME, RestMethod.GET, "jredmine", "get_project_users.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(ModelHelper.GET_ALL_NEWS_REQUEST_NAME, RestMethod.GET, "jredmine", "get_project_news.xml"));

        addRequestBuilder(new ProjectScopeRequestBuilder(ModelHelper.ADD_VERSION_REQUEST_NAME, RestMethod.POST, "jredmine", "add_version.xml") {
            private static final long serialVersionUID = 1L;

            @Override
            public void checkRequestArgs(Object... args) {
                checkRequestArgs(2, "projectName, version", args);
            }

            @Override
            public String[] getParameters(Object... args) {
                Version version = (Version) args[1];
                String date = getVersionEffectiveDate(version);
                String status = getVersionStatus(version);
                return new String[]{
                        "version[name]", version.getName(),
                        "version[description]", version.getDescription(),
                        "version[effective_date]", date,
                        "version[status]", status
                };
            }
        });

        addRequestBuilder(new ProjectScopeRequestBuilder(ModelHelper.UPDATE_VERSION_REQUEST_NAME, RestMethod.POST, "jredmine", "update_version.xml") {
            private static final long serialVersionUID = 1L;

            @Override
            public void checkRequestArgs(Object... args) {
                checkRequestArgs(2, "projectName, version", args);
            }

            @Override
            public String[] getParameters(Object... args) {
                Version version = (Version) args[1];
                String date = getVersionEffectiveDate(version);
                String status = getVersionStatus(version);
                return new String[]{
                        "version[name]", version.getName(),
                        "version[description]", version.getDescription(),
                        "version[effective_date]", date,
                        "version[status]", status
                };
            }
        });

        addRequestBuilder(new ProjectScopeRequestBuilder(ModelHelper.NEXT_VERSION_REQUEST_NAME, RestMethod.POST, "jredmine", "next_version.xml") {
            private static final long serialVersionUID = 1L;

            @Override
            public void checkRequestArgs(Object... args) {
                checkRequestArgs(3,
                                 "projectName, version, oldVersionName", args);
            }

            @Override
            public String[] getParameters(Object... args) {

                Version version = (Version) args[1];
                String date = getVersionEffectiveDate(version);
                String status = getVersionStatus(version);
                String oldVersionName = (String) args[2];
                return new String[]{
                        "oldVersionName", oldVersionName,
                        "version[name]", version.getName(),
                        "version[description]", version.getDescription(),
                        "version[effective_date]", date,
                        "version[status]", status
                };
            }
        });

        addRequestBuilder(new ProjectScopeRequestBuilder(ModelHelper.ADD_NEWS_REQUEST_NAME, RestMethod.POST, "jredmine", "add_news.xml") {
            private static final long serialVersionUID = 1L;

            @Override
            public void checkRequestArgs(Object... args) {
                checkRequestArgs(2, "projectName, news", args);
            }

            @Override
            public String[] getParameters(Object... args) {
                News news = (News) args[1];
                return new String[]{
                        "news[title]", news.getTitle(),
                        "news[summary]", news.getSummary(),
                        "news[description]", news.getDescription()
                };
            }
        });

        // version scope requests

        addRequestBuilder(new VersionScopeRequestBuilder(ModelHelper.GET_VERSION_REQUEST_NAME, RestMethod.GET, "jredmine", "get_version.xml"));
        addRequestBuilder(new VersionScopeRequestBuilder(ModelHelper.GET_ALL_ISSUES_REQUEST_NAME, RestMethod.GET, "jredmine", "get_version_issues.xml"));
        addRequestBuilder(new VersionScopeRequestBuilder(ModelHelper.GET_ALL_ATTACHMENTS_REQUEST_NAME, RestMethod.GET, "jredmine", "get_version_attachments.xml"));
        addRequestBuilder(new VersionScopeRequestBuilder(ModelHelper.ADD_ATTACHMENT_REQUEST_NAME, RestMethod.POST, "jredmine", "add_version_attachment.xml") {
            private static final long serialVersionUID = 1L;

            @Override
            public void checkRequestArgs(Object... args) {
                checkRequestArgs(3,
                                 "projectName, versionName, attachment", args);
            }

            @Override
            public String[] getParameters(Object... args) {
                String versionId = (String) args[1];
                Attachment attachment = (Attachment) args[2];
                return new String[]{
                        "version_name", versionId,
                        "attachment[description]", attachment.getDescription()
                };
            }

            @Override
            public Map<String, File> getAttachments(Object... args) {
                Map<String, File> upload = new HashMap<String, File>();
                Attachment attachment = (Attachment) args[2];
                upload.put("attachment[file]", attachment.getToUpload());
                return upload;
            }
        });

        // issue scope requests

        addRequestBuilder(new IssueScopeRequestBuilder(ModelHelper.GET_ALL_ISSUE_TIME_ENTRY_REQUEST_NAME, RestMethod.GET, "jredmine", "get_issue_times.xml"));

        addRequestBuilder(new IssueScopeRequestBuilder(ModelHelper.ADD_ISSUE_TIME_ENTRY_REQUEST_NAME, RestMethod.POST, "jredmine", "add_issue_time.xml") {
            private static final long serialVersionUID = 1L;

            @Override
            public void checkRequestArgs(Object... args) {
                checkRequestArgs(3,
                                 "projectName, issueId, timeEntry", args);
            }

            @Override
            public String[] getParameters(Object... args) {
                String issueId = (String) args[1];
                TimeEntry timeEntry = (TimeEntry) args[2];
                Date d = timeEntry.getSpentOn();
                if (d == null) {
                    d = new Date();
                }
                String date = DATE_FORMAT.format(d);
                return new String[]{
                        "issue_id", issueId,
                        "time_entry[activity_id]", timeEntry.getActivityId() + "",
                        "time_entry[spent_on]", date,
                        "time_entry[hours]", timeEntry.getHours() + "",
                        "time_entry[comments]", timeEntry.getComments() == null ? "" : timeEntry.getComments()
                };
            }
        });
    }

    protected static String getVersionStatus(Version version) {
        String status = version.getStatus();
        if (Strings.isNullOrEmpty(status)) {

            // use default open status
            status = VersionStatusEnum.open.name();
        }
        return status;
    }

    protected static String getVersionEffectiveDate(Version version) {
        return version.getEffectiveDate() == null ? "" :
               DATE_FORMAT.format(version.getEffectiveDate());
    }

    /**
     * Default {@link RestRequestBuilder} to be used in {@link RedmineRestClient}.
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 1.0.0
     */
    protected static class DefaultRequestBuilder extends AbstractRequestBuilder {

        private static final long serialVersionUID = 1L;

        protected String scope;

        public DefaultRequestBuilder(String name,
                                     RestMethod method,
                                     String... action) {
            this(name, null, method, action);
        }

        public DefaultRequestBuilder(String name,
                                     String scope,
                                     RestMethod method,
                                     String... action) {
            super(name, method, action);
            this.scope = scope;
        }

        /**
         * Gets the scope of this request (scope can be null,project,
         * version issue,...).
         *
         * @return the scope of this request
         */
        public final String getScope() {
            return scope;
        }

        @Override
        public void checkRequestArgs(Object... args) {

            // by default request have no ags
            checkRequestArgs(0, "", args);
        }
    }

    /**
     * Builder of requests which are in a project context.
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 1.0.0
     */
    protected static class ProjectScopeRequestBuilder extends DefaultRequestBuilder {

        private static final long serialVersionUID = 1L;

        public ProjectScopeRequestBuilder(String name,
                                          RestMethod method,
                                          String... action) {
            this(name, ModelHelper.PROJECT_SCOPE, method, action);
        }

        public ProjectScopeRequestBuilder(String name,
                                          String scope,
                                          RestMethod method,
                                          String... action) {
            super(name, scope, method, action);
        }

        @Override
        public void checkRequestArgs(Object... args) {
            checkRequestArgs(1, "projectName", args);
        }

        @Override
        public String[] getPath(Object... args) {

            // one args : projectName

            String projectName = (String) args[0];

            String[] result = new String[action.length + 1];
            System.arraycopy(action, 0, result, 0, action.length);
            result[action.length] = projectName;
            return result;
        }
    }

    /**
     * Builder of requests which are in a project and issue context.
     * <p/>
     * Created: 1 janv. 2010
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 1.0.3
     */
    protected static class IssueScopeRequestBuilder extends ProjectScopeRequestBuilder {

        private static final long serialVersionUID = 1L;

        public IssueScopeRequestBuilder(String name,
                                        RestMethod method,
                                        String... action) {
            super(name, ModelHelper.ISSUE_SCOPE, method, action);
        }

        @Override
        public void checkRequestArgs(Object... args) {
            checkRequestArgs(2, "projectName, issueId", args);
        }

        @Override
        public String[] getParameters(Object... args) {

            // args 1 = project id
            // args 2 = issue id

            String issueId = (String) args[1];

            return new String[]{
                    "issue_id", issueId
            };
        }
    }

    /**
     * Builder of requests which are in a project and version context.
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 1.0.0
     */
    protected static class VersionScopeRequestBuilder extends ProjectScopeRequestBuilder {

        private static final long serialVersionUID = 1L;

        public VersionScopeRequestBuilder(String name,
                                          RestMethod method,
                                          String... action) {
            super(name, ModelHelper.VERSION_SCOPE, method, action);
        }

        @Override
        public void checkRequestArgs(Object... args) {
            checkRequestArgs(2, "projectName, versionName", args);
        }

        @Override
        public String[] getParameters(Object... args) {

            // args 1 = project id
            // args 2 = version name

            String versionName = (String) args[1];

            return new String[]{
                    "version_name", versionName
            };
        }
    }

}
