/*
 * #%L
 * JRedmine :: Client
 * 
 * $Id: DefaultRedmineService.java 344 2012-10-04 11:57:42Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.4/jredmine-client/src/main/java/org/nuiton/jredmine/DefaultRedmineService.java $
 * %%
 * Copyright (C) 2009 - 2012 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.jredmine.model.Attachment;
import org.nuiton.jredmine.model.Issue;
import org.nuiton.jredmine.model.IssueCategory;
import org.nuiton.jredmine.model.IssuePriority;
import org.nuiton.jredmine.model.IssueStatus;
import org.nuiton.jredmine.model.ModelHelper;
import org.nuiton.jredmine.model.News;
import org.nuiton.jredmine.model.Project;
import org.nuiton.jredmine.model.TimeEntry;
import org.nuiton.jredmine.model.Tracker;
import org.nuiton.jredmine.model.User;
import org.nuiton.jredmine.model.Version;

/**
 * Default {@link RedmineService} implementation.
 *
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role="org.nuiton.jredmine.RedmineService" role-hint="default"
 * @since 1.0.0
 */
public class DefaultRedmineService implements RedmineService {

    protected static final Log log = LogFactory.getLog(DefaultRedmineService.class);

    /**
     * Delegate all the technical layer to a implementor.
     *
     * @see RedmineServiceImplementor
     * @since 1.4
     */
    protected final RedmineServiceImplementor delegateImplementor;

    /**
     * Delegate all anonymous call to the anonymous service.
     *
     * @see RedmineAnonymousService
     * @since 1.4
     */
    protected final RedmineAnonymousService anonymousService;

    /**
     * Default constructor.
     * <p/>
     * Make this service use the default {@link RedmineServiceImplementor}.
     *
     * @see DefaultRedmineServiceImplementor
     * @since 1.4
     */
    public DefaultRedmineService() {
        this(new DefaultRedmineServiceImplementor());
    }

    /**
     * Service constructor using the givne implementator to deal this technical layer.
     *
     * @param delegateImplementor the implementator to use
     * @since 1.4
     */
    public DefaultRedmineService(RedmineServiceImplementor delegateImplementor) {
        this.delegateImplementor = delegateImplementor;
        this.anonymousService = new DefaultRedmineAnonymousService(delegateImplementor);
    }

    ///////////////////////////////////////////////////////////////////////////
    /// RedmineAnonymousService implementation
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public void init(RedmineServiceConfiguration configuration) throws RedmineServiceException {
        if (log.isDebugEnabled()) {
            log.debug("init service with configuration: " + configuration);
        }
        if (!delegateImplementor.isInit()) {

            // only init once
            delegateImplementor.init(configuration);
        }
    }

    @Override
    public void destroy() throws RedmineServiceException {
        delegateImplementor.destroy();
    }

    @Override
    public RedmineServiceImplementor getDelegateImplementor() {
        return delegateImplementor;
    }

    @Override
    public IssueStatus[] getIssueStatuses() throws RedmineServiceException {
        IssueStatus[] result = anonymousService.getIssueStatuses();
        return result;
    }

    @Override
    public IssuePriority[] getIssuePriorities() throws RedmineServiceException {
        IssuePriority[] result = anonymousService.getIssuePriorities();
        return result;
    }

    @Override
    public Project[] getProjects() throws RedmineServiceException {
        Project[] result = anonymousService.getProjects();
        return result;
    }

    @Override
    public IssueCategory[] getIssueCategories(String projectName) throws RedmineServiceException {
        IssueCategory[] result = anonymousService.getIssueCategories(projectName);
        return result;
    }

    @Override
    public Project getProject(String projectName) throws RedmineServiceException {
        Project result = anonymousService.getProject(projectName);
        return result;
    }

    @Override
    public Tracker[] getTrackers(String projectName) throws RedmineServiceException {
        Tracker[] result = anonymousService.getTrackers(projectName);
        return result;
    }

    @Override
    public News[] getNews(String projectName) throws RedmineServiceException {
        News[] result = anonymousService.getNews(projectName);
        return result;
    }

    @Override
    public User[] getProjectMembers(String projectName) throws RedmineServiceException {
        User[] result = anonymousService.getProjectMembers(projectName);
        return result;
    }

    @Override
    public Version[] getVersions(String projectName) throws RedmineServiceException {
        Version[] result = anonymousService.getVersions(projectName);
        return result;
    }

    @Override
    public Version getVersion(String projectName,
                              String versionName) throws RedmineServiceException {
        Version result = anonymousService.getVersion(projectName, versionName);
        return result;
    }

    @Override
    public Attachment[] getAttachments(String projectName,
                                       String versionName) throws RedmineServiceException {
        Attachment[] result = anonymousService.getAttachments(projectName, versionName);
        return result;
    }

    @Override
    public Issue[] getIssues(String projectName,
                             String versionName) throws RedmineServiceException {
        Issue[] result = anonymousService.getIssues(projectName, versionName);
        return result;
    }

    @Override
    public TimeEntry[] getIssueTimeEntries(String projectName,
                                           String issueId) throws RedmineServiceException {
        TimeEntry[] result = anonymousService.getIssueTimeEntries(projectName, issueId);
        return result;
    }

    @Override
    public Issue[] getIssues(String projectName) throws RedmineServiceException {
        Issue[] result = anonymousService.getIssues(projectName);
        return result;
    }

    @Override
    public Issue[] getOpenedIssues(String projectName) throws RedmineServiceException {
        Issue[] result = anonymousService.getOpenedIssues(projectName);
        return result;
    }

    @Override
    public Issue[] getClosedIssues(String projectName) throws RedmineServiceException {
        Issue[] result = anonymousService.getClosedIssues(projectName);
        return result;
    }

    ///////////////////////////////////////////////////////////////////////////
    /// RedmineLogguedService implementation
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public Project[] getUserProjects() throws RedmineServiceException {
        delegateImplementor.checkLoggued();
        Project[] result = delegateImplementor.getDatas(ModelHelper.GET_USER_PROJECTS_REQUEST_NAME,
                                                        Project.class);
        return result;
    }

    @Override
    public Version addVersion(String projectName,
                              Version version) throws RedmineServiceException {

        // send data and obtain created version
        Version result = delegateImplementor.sendData(ModelHelper.ADD_VERSION_REQUEST_NAME,
                                                      Version.class,
                                                      projectName,
                                                      version
        );
        return result;
    }

    @Override
    public Version updateVersion(String projectName,
                                 Version version) throws RedmineServiceException {

        // send data and obtain updated version
        Version result = delegateImplementor.sendData(ModelHelper.UPDATE_VERSION_REQUEST_NAME,
                                                      Version.class,
                                                      projectName,
                                                      version
        );
        return result;
    }

    @Override
    public Version nextVersion(String projectName,
                               String oldVersionName,
                               Version newVersion) throws RedmineServiceException {

        // send data and obtain updated or created new version
        Version result = delegateImplementor.sendData(ModelHelper.NEXT_VERSION_REQUEST_NAME,
                                                      Version.class,
                                                      projectName,
                                                      newVersion,
                                                      oldVersionName
        );
        return result;
    }

    @Override
    public Attachment addAttachment(String projectName,
                                    String versionName,
                                    Attachment attachement) throws RedmineServiceException {

        // send data and obtain created attachment
        Attachment result = delegateImplementor.sendData(ModelHelper.ADD_ATTACHMENT_REQUEST_NAME,
                                                         Attachment.class,
                                                         projectName,
                                                         versionName,
                                                         attachement
        );
        return result;
    }

    @Override
    public News addNews(String projectName,
                        News news) throws RedmineServiceException {

        // send data and obtain created news
        News result = delegateImplementor.sendData(ModelHelper.ADD_NEWS_REQUEST_NAME,
                                                   News.class,
                                                   projectName,
                                                   news
        );
        return result;
    }

    @Override
    public TimeEntry addIssueTimeEntry(String projectName,
                                       String issueId,
                                       TimeEntry entry) throws RedmineServiceException {

        // send data and obtain created time entry
        TimeEntry result = delegateImplementor.sendData(ModelHelper.ADD_ISSUE_TIME_ENTRY_REQUEST_NAME,
                                                        TimeEntry.class,
                                                        projectName,
                                                        issueId,
                                                        entry
        );
        return result;
    }

}
