/*
 * *##% 
 * JRedmine client
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.jredmine;

import org.nuiton.jredmine.model.Attachment;
import org.nuiton.jredmine.model.Issue;
import org.nuiton.jredmine.model.IssueCategory;
import org.nuiton.jredmine.model.IssuePriority;
import org.nuiton.jredmine.model.IssueStatus;
import org.nuiton.jredmine.model.News;
import org.nuiton.jredmine.model.Project;
import org.nuiton.jredmine.model.Tracker;
import org.nuiton.jredmine.model.User;
import org.nuiton.jredmine.model.Version;

/**
 * The service that mirrors the rails {@code jredmine} plugin.
 *
 * @author chemit
 * @since 1.0.0
 */
public interface RedmineService {

    /**
     * Plexus lookup name
     */
    String ROLE = RedmineService.class.getName();

    /**
     * Obtain all accessible projets for the loggued user.
     *
     * @return all the projects
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Project
     */
    Project[] getProjects() throws RedmineServiceException;

    /**
     * Obtain all the priorities defined on a {@link Issue}.
     * <p/>
     * <b>Note : </b> The priorities are common for all projects.
     *
     * @return all the issue properties
     * @throws RedmineServiceException if any pb while retriving datas
     * @see IssuePriority
     */
    IssuePriority[] getIssuePriorities() throws RedmineServiceException;

    /**
     * Obtain all the statuses defined on a {@link Issue}.
     * <p/>
     * <b>Note : </b> The statuses are common for all projects.
     *
     * @return all the issue statuses
     * @throws RedmineServiceException if any pb while retriving datas
     * @see IssueStatus
     */
    IssueStatus[] getIssueStatuses() throws RedmineServiceException;

    /**
     * Obtain a project given his name.
     *
     * @param projectName the name of the project
     * @return the project
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Project
     */
    Project getProject(String projectName) throws RedmineServiceException;

    /**
     * Obtain all categories defined on issues for a given project.
     *
     * @param projectName the name of the project
     * @return the categories of issues for the given project.
     * @throws RedmineServiceException if any pb while retriving datas
     * @see IssueCategory
     */
    IssueCategory[] getIssueCategories(String projectName) throws RedmineServiceException;

    /**
     * Obtain all trackers defined on a given project.
     *
     * @param projectName the name of the project
     * @return the trackers for the given project.
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Tracker
     */
    Tracker[] getTrackers(String projectName) throws RedmineServiceException;

    /**
     * Obtain all news defined on a given project.
     *
     * @param projectName the name of the project
     * @return the news for the given project.
     * @throws RedmineServiceException if any pb while retriving datas
     * @see News
     */
    News[] getNews(String projectName) throws RedmineServiceException;

    /**
     * Obtain all users defined on a given project.
     *
     * @param projectName the name of the project
     * @return the users for the given project.
     * @throws RedmineServiceException if any pb while retriving datas
     * @see User
     */
    User[] getUsers(String projectName) throws RedmineServiceException;

    /**
     * Obtain all versions defined on a given project.
     *
     * @param projectName the name of the project
     * @return the versions of the given project.
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Version
     */
    Version[] getVersions(String projectName) throws RedmineServiceException;

    /**
     * Obtain a specific version for a given project.
     *
     * @param projectName the name of the project
     * @param versionName the name of the version
     * @return the version
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Version
     */
    Version getVersion(String projectName, String versionName) throws RedmineServiceException;

    /**
     * Obtain all issues for a specific version on a given project
     *
     * @param projectName the name of the project
     * @param versionName the name of the version
     * @return the issues
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Issue
     */
    Issue[] getIssues(String projectName, String versionName) throws RedmineServiceException;

    /**
     * Obtain all attachments for a specific version on a given project
     *
     * @param projectName the name of the project
     * @param versionName the name of the version
     * @return the attachments
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Attachment
     */
    Attachment[] getAttachments(String projectName, String versionName) throws RedmineServiceException;

    /**
     * Add a version for a given project.
     *
     * @param projectName the name of the project
     * @param version     the version to add
     * @return the added version
     * @throws RedmineServiceException if any pb while sending and retriving datas to redmine server
     * @see Version
     */
    Version addVersion(String projectName, Version version) throws RedmineServiceException;

    /**
     * Update a version for a given project.
     *
     * @param projectName the name of the project
     * @param version     the version to update
     * @return the updated version
     * @throws RedmineServiceException if any pb while sending and retriving datas to redmine server
     * @see Version
     */
    Version updateVersion(String projectName, Version version) throws RedmineServiceException;

    /**
     * Prepare a new version (create it or update it).
     * <p/>
     * If the {@code oldVersionName} is given, then all issues unclosed from this
     * old version will be move to the new version.
     *
     * @param projectName    the name of the project
     * @param oldVersionName the name of the old version (optional)
     * @param newVersion     the newVersion to create or update
     * @return the created version
     * @throws RedmineServiceException if any pb while sending and retriving datas to redmine server
     * @see Version
     */
    Version nextVersion(String projectName, String oldVersionName, Version newVersion) throws RedmineServiceException;

    /**
     * Add a news for a given project.
     *
     * @param projectName the name of the project
     * @param news        the news to add
     * @return the added news.
     * @throws RedmineServiceException if any pb while sending and retriving datas to redmine server
     * @see News
     */
    News addNews(String projectName, News news) throws RedmineServiceException;

    /**
     * Add a attachment for a given version of a given project.
     *
     * @param projectName the name of the project
     * @param versionName the name of the version
     * @param attachement the attachment to add
     * @return the added attachment
     * @throws RedmineServiceException if any pb while sending and retriving datas to redmine server
     * @see Attachment
     */
    Attachment addAttachment(String projectName, String versionName, Attachment attachement) throws RedmineServiceException;
}
