/* *##% 
 * JAXX Maven plugin
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/
package org.nuiton.jaxx.plugin;

import org.apache.maven.model.Resource;
import org.apache.maven.project.MavenProject;
import org.nuiton.plugin.AbstractPlugin;
import org.nuiton.util.FileUtil;

import java.io.File;
import java.io.IOException;
import java.util.List;

/**
 * Abstract Jaxx Mojo.
 *
 * @author chemit
 * @since 1.3
 */
public abstract class AbstractJaxxMojo extends AbstractPlugin {

    public abstract File getTargetDirectory();

    public abstract void setTargetDirectory(File targetDirectory);

    /**
     * Dépendance du projet.
     *
     * @parameter default-value="${project}"
     * @required
     * @readonly
     */
    private MavenProject project;
    /**
     * Encoding pour la generation des fichiers
     *
     * @parameter expression="${jaxx.encoding}" default-value="${project.build.sourceEncoding}"
     * @since 2.0.0
     */
    private String encoding;
    /**
     * verbose flag
     *
     * @parameter expression="${jaxx.verbose}" default-value="false"
     * @since 1.3
     */
    private boolean verbose;

    /**
     * to force generation of java source for any jaxx files with no timestamp checking.
     * <p/>
     * By default, never force generation.
     *
     * @parameter expression="${jaxx.force}" default-value="false"
     */
    private boolean force;
    /**
     * Flag to activate help id detection while parsing jaxx files.
     * <p/>
     * By default, not active.
     *
     * @parameter expression="${jaxx.generateHelp}" default-value="false"
     * @since 1.3
     */
    protected boolean generateHelp;

    @Override
    protected boolean checkPackaging() {
        return project != null && !"pom".equals(project.getPackaging()) && !"site".equals(project.getPackaging());
    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    public String getEncoding() {
        return encoding;
    }

    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    public boolean isForce() {
        return force;
    }

    public void setForce(boolean force) {
        this.force = force;
    }

    /**
     * TODO-TC20091221 Should move this to AbstractPlugin
     * Create the directory if necessary.
     *
     * @param dir the directory to create if not already existing
     * @throws IOException if could not create the directory
     */
    protected void createDirectoryIfNecessary(File dir) throws IOException {
        if (!dir.exists()) {
            if (verbose) {
                getLog().info("mkdir " + dir);
            }
            boolean b = dir.mkdirs();
            if (!b) {
                throw new IOException("could not create directory " + dir);
            }
        }
    }

    //TODO-TC20091221 move this api to PluginHelper
    public static boolean addResourceDir(File dir, MavenProject project, String... includes) {
        List<?> resources = project.getResources();
        boolean added = addResourceDir(dir, project, resources, includes);
        return added;
    }

    //TODO-TC20091221 move this api to PluginHelper
    public static boolean addTestResourceDir(File newresourceDir, MavenProject project, String... includes) {
        List<?> resources = project.getTestResources();
        boolean added = addResourceDir(newresourceDir, project, resources, includes);
        return added;
    }

    //TODO-TC20091221 move this api to PluginHelper
    public static boolean addResourceDir(File dir, MavenProject project, List<?> resources, String... includes) {
        String newresourceDir = dir.getAbsolutePath();
        boolean shouldAdd = true;
        for (Object o : resources) {
            Resource r = (Resource) o;
            if (!r.getDirectory().equals(newresourceDir)) {
                continue;
            }

            for (String i : includes) {
                if (!r.getIncludes().contains(i)) {
                    r.addInclude(i);
                }
            }
            shouldAdd = false;
            break;
        }
        if (shouldAdd) {
            Resource r = new Resource();
            r.setDirectory(newresourceDir);
            for (String i : includes) {
                if (!r.getIncludes().contains(i)) {
                    r.addInclude(i);
                }
            }
            project.addResource(r);
        }
        return shouldAdd;
    }


    class GetLastModifiedFileAction implements FileUtil.FileAction {

        protected File lastFile;

        public GetLastModifiedFileAction(File lastFile) {
            this.lastFile = lastFile;

        }

        @Override
        public boolean doAction(File f) {

            if (f.lastModified() > lastFile.lastModified()) {
                lastFile = f;
            }
            return true;
        }

        public File getLastFile() {
            return lastFile;
        }
    }

    protected Long getLastModified(File dir) {
        if (!dir.exists()) {
            return null;
        }
        GetLastModifiedFileAction fileAction = new GetLastModifiedFileAction(dir);
        FileUtil.walkAfter(dir, fileAction);
        return fileAction.getLastFile().lastModified();
    }
}
