package jaxx.demo.config;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;

import java.util.Locale;

import static org.nuiton.i18n.I18n._;

/**
 * La configuration de l'application.
 * <p/>
 * Il s'agit de l'objet partagé par toutes les démos.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.0.2
 */
public class DemoConfig extends ApplicationConfig {

    /** Logger */
    static private Log log = LogFactory.getLog(DemoConfig.class);

    /**
     * le fichier de configuration de l'application avec les informations sur
     * le projet (version, license,...) et la configuration des ui (icons, ...)
     */
    public static final String APPLICATION_PROPERTIES = "/jaxx-demo.properties";

    public static final String PROPERTY_FULLSCREEN = "fullscreen";

    public static final String PROPERTY_LOCALE = "locale";

    public static final String PROPERTY_FONT_SIZE = "fontSize";

    public DemoConfig() {

        setConfigFileName(Option.CONFIG_FILE.defaultValue);

        for (Option o : Option.values()) {
            setDefaultOption(o.key, o.defaultValue);
        }

        installSaveUserAction(PROPERTY_FULLSCREEN,
                              PROPERTY_FONT_SIZE,
                              PROPERTY_LOCALE);
    }

    public boolean isFullScreen() {
        Boolean result = getOptionAsBoolean(Option.FULL_SCREEN.key);
        return result != null && result;
    }

    public Locale getLocale() {
        Locale result = getOption(Locale.class, Option.LOCALE.key);
        return result;
    }

    public Float getFontSize() {
        Float result = getOption(Float.class, Option.FONT_SIZE.key);
        return result;
    }

    public void setFullscreen(boolean fullscreen) {
        Object oldValue = null;
        setOption(Option.FULL_SCREEN.key, fullscreen + "");
        firePropertyChange(PROPERTY_FULLSCREEN, oldValue, fullscreen);
    }

    public void setLocale(Locale newLocale) {
        setOption(Option.LOCALE.key, newLocale.toString());
        firePropertyChange(PROPERTY_LOCALE, null, newLocale);
    }

    public void setFontSize(Float newFontSize) {
        Float oldValue = getFontSize();
        if (log.isDebugEnabled()) {
            log.debug("changing font-size to " + newFontSize);
        }
        setOption(Option.FONT_SIZE.key, newFontSize.toString());
        firePropertyChange(PROPERTY_FONT_SIZE, oldValue, newFontSize);
    }

    //////////////////////////////////////////////////
    // Toutes les options disponibles
    //////////////////////////////////////////////////

    public enum Option implements OptionDef {

        CONFIG_FILE(
                CONFIG_FILE_NAME,
                _("jaxxdemo.config.configFileName.description"),
                "jaxxdemo",
                String.class,
                true,
                true),
        FULL_SCREEN(
                "ui.fullscreen",
                _("jaxxdemo.config.ui.fullscreen"),
                "false",
                Boolean.class,
                false,
                false),
        LOCALE(
                "ui." + PROPERTY_LOCALE,
                _("jaxxdemo.config.ui.locale"),
                Locale.FRANCE.toString(),
                Locale.class,
                false,
                false),
        FONT_SIZE(
                "ui." + PROPERTY_FONT_SIZE,
                _("jaxxdemo.config.ui.fontSize"),
                "10f",
                Float.class,
                false,
                false);

        public final String key;

        public final String description;

        public String defaultValue;

        public final Class<?> type;

        public boolean _transient;

        public boolean _final;

        Option(String key,
               String description,
               String defaultValue,
               Class<?> type,
               boolean _transient,
               boolean _final) {
            this.key = key;
            this.description = description;
            this.defaultValue = defaultValue;
            this.type = type;
            this._final = _final;
            this._transient = _transient;
        }

        @Override
        public boolean isFinal() {
            return _final;
        }

        @Override
        public void setDefaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        @Override
        public void setTransient(boolean _transient) {
            this._transient = _transient;
        }

        @Override
        public void setFinal(boolean _final) {
            this._final = _final;
        }

        @Override
        public boolean isTransient() {
            return _transient;
        }

        @Override
        public String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public String getDescription() {
            return description;
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public Class<?> getType() {
            return type;
        }
    }
}
