/*
 * #%L
 * JAXX :: Compiler
 * 
 * $Id: AbstractFinalizer.java 2243 2011-03-18 11:19:23Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.5.23/jaxx-compiler/src/main/java/jaxx/compiler/finalizers/AbstractFinalizer.java $
 * %%
 * Copyright (C) 2008 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package jaxx.compiler.finalizers;

import jaxx.compiler.java.JavaElementFactory;
import jaxx.compiler.java.JavaField;
import jaxx.compiler.java.JavaFile;
import jaxx.compiler.java.JavaMethod;

/**
 * Base implementation of a {@link JAXXCompilerFinalizer}.
 * <p/>
 * Contains commons methods and constants.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.4
 */
public abstract class AbstractFinalizer implements JAXXCompilerFinalizer {

    /**
     * Clones the given {@code field} and adds it to the {@code file} as a
     * property via the method {@link JavaFile#addField(JavaField)}.
     *
     * @param file  the file where to add the cloned field
     * @param field the field to clone
     * @since 2.4
     */
    protected void addField(JavaFile file, JavaField field) {

        JavaField clonedField = JavaElementFactory.cloneField(field);
        file.addField(clonedField);
    }

    /**
     * Clones the given {@code method} and adds it to the {@code file} as a
     * simple method using the method {@link JavaFile#addMethod(JavaMethod)}.
     *
     * @param file  the file where to add the cloned field
     * @param field the field to clone
     * @since 2.4
     */
    protected void addSimpleField(JavaFile file, JavaField field) {

        JavaField clonedField = JavaElementFactory.cloneField(field);
        file.addSimpleField(clonedField);
    }

    /**
     * Clones the given {@code field} and adds it to the {@code file} as a
     * simple field using the method {@link JavaFile#addSimpleField(JavaField)}.
     *
     * @param file   the file where to add the cloned method
     * @param method the method to clone
     * @param types optional types to use to simplify the body of the method
     * @since 2.4
     */
    protected void addMethod(JavaFile file, JavaMethod method, String... types) {

        JavaMethod clonedMethod = JavaElementFactory.cloneMethod(method);
        if (types.length > 0) {
            String body = clonedMethod.getBody();
            String simplifiedBody = file.simplifyCode(body, types);
            clonedMethod.setBody(simplifiedBody);
        }
        file.addMethod(clonedMethod);
    }

}
