package org.nuiton.eugene.plugin.modelextension;

/*-
 * #%L
 * EUGene :: Maven plugin
 * %%
 * Copyright (C) 2006 - 2016 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.nuiton.eugene.models.object.reader.extension.ModelExtensionFileParser;
import org.nuiton.eugene.models.object.reader.extension.ModelExtensionFileParserCallback;
import org.nuiton.eugene.models.object.reader.extension.ModelExtensionFileParserFactory;
import org.nuiton.eugene.models.stereotype.InvalidStereotypeSyntaxException;
import org.nuiton.eugene.models.tagvalue.InvalidTagValueSyntaxException;

import java.io.File;
import java.io.IOException;

/**
 * Created on 10/09/16.
 *
 * @author Tony Chemit - chemit@codelutin.com
 */
public class ModelBeanBuilder implements ModelExtensionFileParserCallback {

    protected final ModelBean modelBean;
    protected int modelStereotypeHits;
    protected int modelTagValueHits;
    protected int packageStereotypeHits;
    protected int packageTagValueHits;
    protected int classStereotypeHits;
    protected int classTagValueHits;
    protected int classAttributeStereotypeHits;
    protected int classAttributeTagValueHits;
    protected boolean strictLoading;

    public ModelBeanBuilder(boolean strictLoading, String modelName) {
        this.modelBean = new ModelBean(modelName);
    }

    public ModelBean build() {
        return modelBean;
    }

    public int getStereotypeHits() {
        return modelStereotypeHits + packageStereotypeHits + classStereotypeHits + classAttributeStereotypeHits;
    }

    public int getTagValueHits() {
        return modelTagValueHits + packageTagValueHits + classTagValueHits + classAttributeTagValueHits;
    }

    public int getModelStereotypeHits() {
        return modelStereotypeHits;
    }

    public int getModelTagValueHits() {
        return modelTagValueHits;
    }

    public int getPackageStereotypeHits() {
        return packageStereotypeHits;
    }

    public int getPackageTagValueHits() {
        return packageTagValueHits;
    }

    public int getClassStereotypeHits() {
        return classStereotypeHits;
    }

    public int getClassTagValueHits() {
        return classTagValueHits;
    }

    public int getClassAttributeStereotypeHits() {
        return classAttributeStereotypeHits;
    }

    public int getClassAttributeTagValueHits() {
        return classAttributeTagValueHits;
    }

    @Override
    public boolean onModelTagValueFound(String tag, String value) {
        addTagValue(modelBean, tag, value);
        modelTagValueHits++;
        return true;
    }

    @Override
    public boolean onModelStereotypeFound(String stereotype) {
        addStereotype(modelBean, stereotype);
        modelStereotypeHits++;
        return true;
    }

    @Override
    public boolean onPackageTagValueFound(String packageName, String tag, String value) {
        addTagValue(modelBean.getOrCreatePackage(packageName), tag, value);
        packageTagValueHits++;
        return true;
    }

    @Override
    public boolean onPackageStereotypeFound(String packageName, String stereotype) {
        addStereotype(modelBean.getOrCreatePackage(packageName), stereotype);
        packageStereotypeHits++;
        return true;
    }

    @Override
    public boolean onClassTagValueFound(String className, String tag, String value) {
        addTagValue(modelBean.getOrCreateClass(className), tag, value);
        classTagValueHits++;
        return true;
    }

    @Override
    public boolean onClassStereotypeFound(String className, String stereotype) {
        addStereotype(modelBean.getOrCreateClass(className), stereotype);
        classStereotypeHits++;
        return true;
    }

    @Override
    public boolean onAttributeTagValueFound(String className, String attributeName, String tag, String value) {
        addTagValue(modelBean.getOrCreateClassAttribute(className, attributeName), tag, value);
        classAttributeTagValueHits++;
        return true;
    }

    @Override
    public boolean onAttributeStereotypeFound(String className, String attributeName, String stereotype) {
        addStereotype(modelBean.getOrCreateClassAttribute(className, attributeName), stereotype);
        classAttributeStereotypeHits++;
        return true;
    }

    public void addFile(File inputFile) throws InvalidTagValueSyntaxException, InvalidStereotypeSyntaxException, IOException {

        ModelExtensionFileParser parser = ModelExtensionFileParserFactory.newParser(strictLoading, inputFile);
        parser.parse(inputFile, this);

    }

    protected void addTagValue(ElementBean elementBean, String tag, String value) {
        elementBean.getTagValues().put(tag, value);
    }

    protected void addStereotype(ElementBean elementBean, String stereotype) {
        elementBean.getStereotypes().add(stereotype);
    }
}
