/*
 * #%L
 * EUGene :: Java templates
 * 
 * $Id: JavaBeanTransformer.java 1346 2014-04-27 12:17:19Z tchemit $
 * $HeadURL: https://svn.nuiton.org/eugene/tags/eugene-2.9/eugene-java-templates/src/main/java/org/nuiton/eugene/java/JavaBeanTransformer.java $
 * %%
 * Copyright (C) 2004 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.eugene.java;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.EugeneTagValues;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;




/**
 * JavaBeanTransformer generates simple bean with pcs support
 * (and nothing else) according to the JavaBeans 1.1 norm.
 * <p/>
 * Since version 2.2.1, it is possible to
 * <ul>
 * <li>generate a simple POJO (says with no PCS support) by using the tag value {@link JavaTemplatesTagValues#TAG_NO_PCS}.</li>
 * <li>generate i18n keys using the tag value {@link EugeneTagValues#TAG_I18N_PREFIX}.</li>
 * </ul>
 *
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.eugene.java.JavaBeanTransformer"
 * @since 2.0.2
 */
public class JavaBeanTransformer extends AbstractJavaBeanTransformer {

    private static final Log log = LogFactory.getLog(JavaBeanTransformer.class);

    @Override
    public void transformFromClass(ObjectModelClass input) {

        if (!JavaTemplatesStereoTypes.hasBeanStereotype(input)) {

            //  not a bean
            return;
        }

        ObjectModelClass output = null;

        if (canGenerateAbstract(input)) {

            // nothing more to do
            output = generateAbstract(input);
        }

        if (canGenerateImpl(input, output)) {

            generateImpl(input);
        }
    }

    protected boolean canGenerateAbstract(ObjectModelClass input) {
        boolean b = !isInClassPath(input);
        return b;
    }

    protected boolean canGenerateImpl(ObjectModelClass input,
                                      ObjectModelClass abstractOutput) {
        String fqn = input.getQualifiedName() + "Impl";

        if (isInClassPath(fqn)) {

            // already in class-path
            return false;
        }

        Collection<ObjectModelOperation> operations = input.getOperations();
        if (!operations.isEmpty()) {

            // this input have some specific operations (so Impl can not be generated)
            return false;
        }

        Collection<ObjectModelOperation> allOtherOperations =
                input.getAllOtherOperations(true);

        if (!allOtherOperations.isEmpty()) {

            Collection<ObjectModelOperation> allExistingOperations =
                    new ArrayList<ObjectModelOperation>(input.getAllSuperclassOperations(true));

            if (abstractOutput != null) {
                allExistingOperations.addAll(abstractOutput.getOperations());
            }

            for (ObjectModelOperation operation : allOtherOperations) {
                if (!allExistingOperations.contains(operation)) {

                    // found one method not on super classe, so can't generate Impl
                    return false;
                }
            }
        }
        return true;
    }


    protected void createAbstractOperations(ObjectModelClass ouput,
                                            Iterable<ObjectModelOperation> operations) {
        JavaTemplatesGeneratorUtil.cloneOperations(
                this,
                operations,
                ouput,
                true,
                ObjectModelJavaModifier.ABSTRACT
        );
    }

    protected ObjectModelClass generateAbstract(ObjectModelClass input) {
        // test if a super class has bean stereotype
        boolean superClassIsBean = false;
        Collection<ObjectModelClass> superclasses = input.getSuperclasses();
        if (CollectionUtils.isNotEmpty(superclasses)) {
            for (ObjectModelClass superclass : superclasses) {
                if (JavaTemplatesStereoTypes.hasBeanStereotype(superclass)) {
                    superClassIsBean = true;
                    break;
                }
            }
        }

        String superClass = null;

        if (!superClassIsBean) {

            // try to find a super class by tag-value
            superClass =
                    JavaTemplatesTagValues.getBeanSuperClassTagValue(
                            input, model);
            if (superClass != null) {

                // will act as if super class is a bean
                superClassIsBean = true;
            }
        }

        ObjectModelClass output =
                createAbstractClass(input.getName(), input.getPackageName());

        if (superClass != null) {
            setSuperClass(output, superClass);
        }
        if (log.isDebugEnabled()) {
            log.debug("will generate " + output.getQualifiedName());
        }

        addSuperClass(input, output);

        boolean serializableFound = addInterfaces(input, output, null);

        if (superClassIsBean) {
            serializableFound = true;
        }

        addSerializable(input, output, serializableFound);

        generateI18nBlockAndConstants(input, output);

        boolean usePCS = !JavaTemplatesTagValues.isNoPCS(input, model);

        boolean generateBooleanGetMethods = !
                EugeneTagValues.isDoNotGenerateBooleanGetMethods(input, model);
//        boolean generateBooleanGetMethods =
//                StringUtils.isEmpty(noGenerateBooleanGetMethods) ||
//                !"true".equals(noGenerateBooleanGetMethods.trim());

        // Get available properties
        List<ObjectModelAttribute> properties = getProperties(input);

        // Add properties field + javabean methods
        for (ObjectModelAttribute attr : properties) {

            createProperty(output, attr, usePCS, generateBooleanGetMethods);
        }

        // Add operations
        createAbstractOperations(output, input.getOperations());

        if (!superClassIsBean) {
            addDefaultMethodForNoneBeanSuperClass(output, usePCS, properties);
        }

        return output;
    }

    protected ObjectModelClass generateImpl(ObjectModelClass input) {

        ObjectModelClass resultClassImpl = createClass(
                input.getName() + "Impl",
                input.getPackageName()
        );

        // set the abstract resulClass as the resultClassImpl super class
        setSuperClass(resultClassImpl, input.getQualifiedName());

        // add a fix serialVersionUID, since the class has no field nor method
        addConstant(resultClassImpl,
                    JavaTemplatesGeneratorUtil.SERIAL_VERSION_UID,
                    "long",
                    "1L",
                    ObjectModelJavaModifier.PRIVATE
        );
        return resultClassImpl;
    }

    protected void addSuperClass(ObjectModelClass input,
                                 ObjectModelClass output) {
        // Set superclass
        Iterator<ObjectModelClass> j = input.getSuperclasses().iterator();
        if (j.hasNext()) {
            ObjectModelClass p = j.next();
            // We want to set the inheritance to the implementation class of the father
            // Ex for model : A -> B (a inherits B) we want : A -> BImpl -> B
            String qualifiedName = p.getQualifiedName() + "Impl";
            setSuperClass(output, qualifiedName);
        }
    }
}
