/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.logging;

import java.text.MessageFormat;

/**
 * An abstract base class for logger plugin instances.
 */
public abstract class AbstractLoggerPluginInstance implements LoggerPluginInstance {

    private final Logger logger;
    private final LoggerPlugin loggerPlugin;

    /**
     * Construct a new instance.
     *
     * @param name the logger name
     * @param resourceBundleName the resource bundle name
     * @param loggerPlugin
     */
    protected AbstractLoggerPluginInstance(final String name, final String resourceBundleName, final LoggerPlugin loggerPlugin) {
        this.loggerPlugin = loggerPlugin;
        //noinspection ThisEscapedInObjectConstruction
        logger = new Logger(name, resourceBundleName, this);
    }

    public LoggerPlugin getLoggerPlugin() {
        return loggerPlugin;
    }

    /** {@inheritDoc} */
    public Logger getLogger() {
        return logger;
    }

    /** {@inheritDoc} */
    public void logf(final Logger.Level level, final String loggerFqcn, final String format, final Object[] params, final Throwable t) {
        if (isEnabled(level)) {
            log(level, loggerFqcn, params == null || params.length == 0 ? format : String.format(String.valueOf(format), params), t);
        }
    }

    /** {@inheritDoc} */
    public void log(final Logger.Level level, final String loggerFqcn, final Object message, final Object[] params, final Throwable t) {
        if (isEnabled(level)) {
            final String msgStr = String.valueOf(message);
            log(level, loggerFqcn, params == null || params.length == 0 ? msgStr : MessageFormat.format(msgStr, params), t);
        }
    }

    /**
     * Simple log method.  Implemented by logger plugins which do not have any special formatting support.
     *
     * @param level the level
     * @param loggerFqcn the logger class name
     * @param message the log message
     * @param t the throwable cause
     */
    protected void log(final Logger.Level level, final String loggerFqcn, final String message, final Throwable t) {
    }
}
