/*
 * #%L
 * Pollen :: VoteCounting Api
 * $Id: AbstractVoteCountingStrategy.java 3593 2012-08-12 11:01:15Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4.5.1/pollen-votecounting-api/src/main/java/org/chorem/pollen/votecounting/strategy/AbstractVoteCountingStrategy.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.votecounting.strategy;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.chorem.pollen.votecounting.model.ChoiceIdAble;
import org.chorem.pollen.votecounting.model.ChoiceScore;
import org.chorem.pollen.votecounting.model.VoteCountingResult;
import org.chorem.pollen.votecounting.model.VoteForChoice;
import org.chorem.pollen.votecounting.model.Voter;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;

import static org.nuiton.i18n.I18n.l_;

/**
 * Base abstract implementation of a {@link VoteCountingStrategy}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4.5
 */
public abstract class AbstractVoteCountingStrategy implements VoteCountingStrategy {

    public static final BigDecimal ZERO_D = BigDecimal.valueOf(0.);

    @Override
    public String getStrategyName(Locale locale) {
        String result = l_(locale, getI18nName());
        return result;
    }

    @Override
    public String getStrategyHelp(Locale locale) {
        String voteName = l_(locale, getI18nName());
        String voteHelp = l_(locale, getI18nHelp());
        String result = l_(locale, "pollen.voteCountingType.help", voteName, voteHelp);
        return result;
    }

    public SortedMap<String, ChoiceScore> votersToResult(Set<Voter> voters) {
        // get all choice Id
        Set<String> choiceIds = getAllChoiceIds(voters);

        SortedMap<String, ChoiceScore> resultByChoice = Maps.newTreeMap();

        // creates all empty result for choice
        for (String choiceId : choiceIds) {
            ChoiceScore choiceScore = ChoiceScore.newScore(choiceId, null);
            resultByChoice.put(choiceId, choiceScore);
        }
        return resultByChoice;
    }

    public VoteCountingResult resultToList(Map<String, ChoiceScore> resultByChoice) {
        List<ChoiceScore> score = toChoiceScore(resultByChoice);
        return VoteCountingResult.newResult(score);
    }

    public List<ChoiceScore> toChoiceScore(Map<String, ChoiceScore> resultByChoice) {
        List<ChoiceScore> score = Lists.newArrayList(resultByChoice.values());
        Collections.sort(score);
        Collections.reverse(score);
        return score;
    }

    public Set<String> getAllChoiceIds(Set<Voter> voters) {
        ChoiceIdAble.ChoiceIdAbleById function = new ChoiceIdAble.ChoiceIdAbleById();
        Set<String> result = Sets.newHashSet();
        for (Voter voter : voters) {
            Iterable<String> transform = Iterables.transform(
                    voter.getVoteForChoices(), function);
            Iterables.addAll(result, transform);

        }
        return result;
    }

    public Map<Voter, List<Set<String>>> buildVoterSortedChoices(Set<Voter> voters) {

        VoteForChoiceComparator comparator = new VoteForChoiceComparator();

        Map<Voter, List<Set<String>>> voterSortedChoices = Maps.newHashMap();

        for (Voter voter : voters) {

            List<Set<String>> sortedChoices = sortVoteForChoices(
                    voter.getVoteForChoices(), comparator);
            voterSortedChoices.put(voter, sortedChoices);
        }
        return voterSortedChoices;
    }

    public List<Set<String>> sortVoteForChoices(Set<VoteForChoice> voteForChoices,
                                                Comparator<VoteForChoice> comparator) {
        // get sort vote for choices
        List<VoteForChoice> sortedChoices = Lists.newArrayList(voteForChoices);
        Collections.sort(sortedChoices, comparator);

        // build ranks
        List<Set<String>> result = Lists.newArrayList();

        Set<String> set = Sets.newHashSet();
        result.add(set);
        VoteForChoice lastVoteForChoice = null;
        for (VoteForChoice voteForChoice : sortedChoices) {
            if (lastVoteForChoice != null &&
                comparator.compare(lastVoteForChoice, voteForChoice) != 0) {

                // new rank found
                // register it
                result.add(set = Sets.newHashSet());
            }

            set.add(voteForChoice.getChoiceId());
            lastVoteForChoice = voteForChoice;
        }
        return result;
    }

    public Set<String> getDirectWinners(Collection<List<Set<String>>> values) {
        Set<String> result = null;
        for (List<Set<String>> value : values) {

            Set<String> topChoices = value.get(0);

            if (result == null) {
                // first time coming here
                result = topChoices;
            } else {

                if (!result.equals(topChoices)) {

                    // not same first ranking, not direct winner
                    result = null;
                    break;
                }
            }
        }
        return result;
    }

    public static class VoteForChoiceComparator implements Comparator<VoteForChoice>, Serializable {

        private static final long serialVersionUID = 1L;

        @Override
        public int compare(VoteForChoice o1, VoteForChoice o2) {
            Double v1 = o1.getVoteValue();
            Double v2 = o2.getVoteValue();
            if (v1 == null) {
                v1 = Double.MAX_VALUE;
            }
            if (v2 == null) {
                v2 = Double.MAX_VALUE;
            }
            return v1.intValue() - v2.intValue();
        }
    }
}
