/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service.snapshot;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.service.snapshot.SnapshotManifest;
import org.apache.cassandra.utils.DirectorySizeCalculator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TableSnapshot {
    private static final Logger logger = LoggerFactory.getLogger(TableSnapshot.class);
    private final String keyspaceName;
    private final String tableName;
    private final UUID tableId;
    private final String tag;
    private final Instant createdAt;
    private final Instant expiresAt;
    private final Set<File> snapshotDirs;

    public TableSnapshot(String keyspaceName, String tableName, UUID tableId, String tag, Instant createdAt, Instant expiresAt, Set<File> snapshotDirs) {
        this.keyspaceName = keyspaceName;
        this.tableName = tableName;
        this.tableId = tableId;
        this.tag = tag;
        this.createdAt = createdAt;
        this.expiresAt = expiresAt;
        this.snapshotDirs = snapshotDirs;
    }

    public String getId() {
        return TableSnapshot.buildSnapshotId(this.keyspaceName, this.tableName, this.tableId, this.tag);
    }

    public String getKeyspaceName() {
        return this.keyspaceName;
    }

    public String getTableName() {
        return this.tableName;
    }

    public String getTag() {
        return this.tag;
    }

    public Instant getCreatedAt() {
        long minCreation;
        if (this.createdAt == null && (minCreation = this.snapshotDirs.stream().mapToLong(File::lastModified).min().orElse(0L)) != 0L) {
            return Instant.ofEpochMilli(minCreation);
        }
        return this.createdAt;
    }

    public Instant getExpiresAt() {
        return this.expiresAt;
    }

    public boolean isExpired(Instant now) {
        if (this.createdAt == null || this.expiresAt == null) {
            return false;
        }
        return this.expiresAt.compareTo(now) < 0;
    }

    public boolean exists() {
        return this.snapshotDirs.stream().anyMatch(File::exists);
    }

    public boolean isExpiring() {
        return this.expiresAt != null;
    }

    public long computeSizeOnDiskBytes() {
        return this.snapshotDirs.stream().mapToLong(FileUtils::folderSize).sum();
    }

    public long computeTrueSizeBytes() {
        SnapshotTrueSizeCalculator visitor = new SnapshotTrueSizeCalculator();
        for (File snapshotDir : this.snapshotDirs) {
            try {
                Files.walkFileTree(snapshotDir.toPath(), visitor);
            }
            catch (IOException e) {
                logger.error("Could not calculate the size of {}.", (Object)snapshotDir, (Object)e);
            }
        }
        return visitor.getAllocatedSize();
    }

    public Collection<File> getDirectories() {
        return this.snapshotDirs;
    }

    public Optional<File> getManifestFile() {
        for (File snapshotDir : this.snapshotDirs) {
            File manifestFile = Directories.getSnapshotManifestFile(snapshotDir);
            if (!manifestFile.exists()) continue;
            return Optional.of(manifestFile);
        }
        return Optional.empty();
    }

    public Optional<File> getSchemaFile() {
        for (File snapshotDir : this.snapshotDirs) {
            File schemaFile = Directories.getSnapshotSchemaFile(snapshotDir);
            if (!schemaFile.exists()) continue;
            return Optional.of(schemaFile);
        }
        return Optional.empty();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        TableSnapshot snapshot = (TableSnapshot)o;
        return Objects.equals(this.keyspaceName, snapshot.keyspaceName) && Objects.equals(this.tableName, snapshot.tableName) && Objects.equals(this.tableId, snapshot.tableId) && Objects.equals(this.tag, snapshot.tag) && Objects.equals(this.createdAt, snapshot.createdAt) && Objects.equals(this.expiresAt, snapshot.expiresAt) && Objects.equals(this.snapshotDirs, snapshot.snapshotDirs);
    }

    public int hashCode() {
        return Objects.hash(this.keyspaceName, this.tableName, this.tableId, this.tag, this.createdAt, this.expiresAt, this.snapshotDirs);
    }

    public String toString() {
        return "TableSnapshot{keyspaceName='" + this.keyspaceName + '\'' + ", tableName='" + this.tableName + '\'' + ", tableId=" + this.tableId + ", tag='" + this.tag + '\'' + ", createdAt=" + this.createdAt + ", expiresAt=" + this.expiresAt + ", snapshotDirs=" + this.snapshotDirs + '}';
    }

    protected static String buildSnapshotId(String keyspaceName, String tableName, UUID tableId, String tag) {
        return String.format("%s:%s:%s:%s", keyspaceName, tableName, tableId, tag);
    }

    static File getLiveFileFromSnapshotFile(Path snapshotFilePath) {
        Path liveDir = snapshotFilePath.getParent().getParent().getParent();
        if (Directories.isSecondaryIndexFolder(snapshotFilePath.getParent())) {
            liveDir = Paths.get(liveDir.getParent().toString(), snapshotFilePath.getParent().getFileName().toString());
        }
        return new File(liveDir.toString(), snapshotFilePath.getFileName().toString());
    }

    public static class SnapshotTrueSizeCalculator
    extends DirectorySizeCalculator {
        @Override
        public boolean isAcceptable(Path snapshotFilePath) {
            return !TableSnapshot.getLiveFileFromSnapshotFile(snapshotFilePath).exists();
        }
    }

    static class Builder {
        private final String keyspaceName;
        private final String tableName;
        private final UUID tableId;
        private final String tag;
        private Instant createdAt = null;
        private Instant expiresAt = null;
        private final Set<File> snapshotDirs = new HashSet<File>();

        Builder(String keyspaceName, String tableName, UUID tableId, String tag) {
            this.keyspaceName = keyspaceName;
            this.tableName = tableName;
            this.tag = tag;
            this.tableId = tableId;
        }

        void addSnapshotDir(File snapshotDir) {
            this.snapshotDirs.add(snapshotDir);
            File manifestFile = new File(snapshotDir, "manifest.json");
            if (manifestFile.exists() && this.createdAt == null && this.expiresAt == null) {
                this.loadTimestampsFromManifest(manifestFile);
            }
        }

        private void loadTimestampsFromManifest(File manifestFile) {
            try {
                logger.trace("Loading snapshot manifest from {}", (Object)manifestFile);
                SnapshotManifest manifest = SnapshotManifest.deserializeFromJsonFile(manifestFile);
                this.createdAt = manifest.createdAt;
                this.expiresAt = manifest.expiresAt;
            }
            catch (IOException e) {
                logger.warn("Cannot read manifest file {} of snapshot {}.", new Object[]{manifestFile, this.tag, e});
            }
        }

        TableSnapshot build() {
            return new TableSnapshot(this.keyspaceName, this.tableName, this.tableId, this.tag, this.createdAt, this.expiresAt, this.snapshotDirs);
        }
    }
}

