/*
 * Decompiled with CFR 0.152.
 */
package net.timewalker.ffmq4.storage.data.impl.journal;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import net.timewalker.ffmq4.storage.data.impl.journal.JournalException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public final class JournalFile {
    private static final Log log = LogFactory.getLog(JournalFile.class);
    public static final String SUFFIX = ".journal";
    public static final String RECYCLED_SUFFIX = ".recycled";
    private String baseName;
    private int storageSyncMethod;
    private File file;
    private FileChannel channel;
    private RandomAccessFile output;
    private byte[] writeBuffer;
    private int usedBufferSize;
    private long size;
    private long lastTransactionId;
    private boolean complete;

    public JournalFile(int index, String baseName, File dataFolder, long maxJournalSize, int writeBufferSize, int storageSyncMethod, boolean preAllocateFiles) throws JournalException {
        this.baseName = baseName;
        this.storageSyncMethod = storageSyncMethod;
        this.initNewFile(index, baseName, dataFolder, maxJournalSize, writeBufferSize, preAllocateFiles);
    }

    public JournalFile(int index, String baseName, File dataFolder, File recycledFile, int writeBufferSize, int storageSyncMethod) throws JournalException {
        this.baseName = baseName;
        this.storageSyncMethod = storageSyncMethod;
        this.initFromRecycledFile(index, baseName, dataFolder, recycledFile, writeBufferSize);
    }

    private void initNewFile(int index, String baseName, File dataFolder, long maxJournalSize, int writeBufferSize, boolean preAllocateFiles) throws JournalException {
        this.file = new File(dataFolder, baseName + SUFFIX + "." + index);
        if (this.file.exists()) {
            throw new JournalException("Journal file already exists : " + this.file.getAbsolutePath());
        }
        try {
            this.output = new RandomAccessFile(this.file, "rw");
            this.channel = this.output.getChannel();
            if (preAllocateFiles) {
                this.fillWithZeroes(maxJournalSize, writeBufferSize);
            }
        }
        catch (IOException e) {
            log.error((Object)("[" + baseName + "] Cannot create store journal : " + this.file.getAbsolutePath()), (Throwable)e);
            throw new JournalException("Cannot create store journal : " + this.file.getAbsolutePath(), e);
        }
        this.writeBuffer = new byte[writeBufferSize];
    }

    private void initFromRecycledFile(int index, String baseName, File dataFolder, File recycledFile, int writeBufferSize) throws JournalException {
        this.file = new File(dataFolder, baseName + SUFFIX + "." + index);
        if (this.file.exists()) {
            throw new JournalException("Journal file already exists : " + this.file.getAbsolutePath());
        }
        if (!recycledFile.renameTo(this.file)) {
            throw new JournalException("Cannot rename recycled journal file " + recycledFile.getAbsolutePath() + " to " + this.file.getAbsolutePath());
        }
        try {
            this.output = new RandomAccessFile(this.file, "rw");
            this.channel = this.output.getChannel();
        }
        catch (IOException e) {
            log.error((Object)("[" + baseName + "] Cannot initialize store journal : " + this.file.getAbsolutePath()), (Throwable)e);
            throw new JournalException("Cannot initialize store journal : " + this.file.getAbsolutePath(), e);
        }
        this.writeBuffer = new byte[writeBufferSize];
    }

    private void fillWithZeroes(long maxJournalSize, int writeBufferSize) throws IOException {
        this.output.seek(0L);
        byte[] zeroes = new byte[writeBufferSize];
        long total = 0L;
        while (total + (long)writeBufferSize < maxJournalSize) {
            this.output.write(zeroes);
            total += (long)writeBufferSize;
        }
        if (total < maxJournalSize) {
            this.output.write(zeroes, 0, (int)(maxJournalSize - total));
        }
        this.output.seek(0L);
    }

    protected void sync() throws JournalException {
        try {
            this.flushBuffer();
            switch (this.storageSyncMethod) {
                case 1: {
                    this.output.getFD().sync();
                    break;
                }
                case 2: {
                    this.channel.force(false);
                    break;
                }
                default: {
                    throw new JournalException("Unsupported sync method : " + this.storageSyncMethod);
                }
            }
        }
        catch (IOException e) {
            log.error((Object)("[" + this.baseName + "] Cannot create sync journal file : " + this.file.getAbsolutePath()), (Throwable)e);
            throw new JournalException("Could not sync journal file : " + this.file.getAbsolutePath(), e);
        }
    }

    protected void complete() throws JournalException {
        this.sync();
        this.complete = true;
    }

    private void flushBuffer() throws IOException {
        if (this.usedBufferSize > 0) {
            this.output.write(this.writeBuffer, 0, this.usedBufferSize);
            this.usedBufferSize = 0;
        }
    }

    public void write(byte[] data) throws JournalException {
        try {
            int len = data.length;
            if (len > this.writeBuffer.length) {
                this.flushBuffer();
                this.output.write(data);
            } else {
                if (len > this.writeBuffer.length - this.usedBufferSize) {
                    this.flushBuffer();
                }
                System.arraycopy(data, 0, this.writeBuffer, this.usedBufferSize, len);
                this.usedBufferSize += len;
            }
            this.size += (long)data.length;
        }
        catch (IOException e) {
            log.error((Object)("[" + this.baseName + "] Cannot write to store journal : " + this.file.getAbsolutePath()), (Throwable)e);
            throw new JournalException("Cannot write to store journal : " + this.file.getAbsolutePath(), e);
        }
    }

    public void writeByte(int v) throws JournalException {
        try {
            if (this.usedBufferSize == this.writeBuffer.length) {
                this.flushBuffer();
            }
            this.writeBuffer[this.usedBufferSize++] = (byte)v;
            ++this.size;
        }
        catch (IOException e) {
            log.error((Object)("[" + this.baseName + "] Cannot write to store journal : " + this.file.getAbsolutePath()), (Throwable)e);
            throw new JournalException("Cannot write to store journal : " + this.file.getAbsolutePath(), e);
        }
    }

    public void writeInt(int v) throws JournalException {
        try {
            if (this.writeBuffer.length - this.usedBufferSize < 4) {
                this.flushBuffer();
            }
            this.writeBuffer[this.usedBufferSize++] = (byte)(v >>> 24 & 0xFF);
            this.writeBuffer[this.usedBufferSize++] = (byte)(v >>> 16 & 0xFF);
            this.writeBuffer[this.usedBufferSize++] = (byte)(v >>> 8 & 0xFF);
            this.writeBuffer[this.usedBufferSize++] = (byte)(v >>> 0 & 0xFF);
            this.size += 4L;
        }
        catch (IOException e) {
            log.error((Object)("[" + this.baseName + "] Cannot write to store journal : " + this.file.getAbsolutePath()), (Throwable)e);
            throw new JournalException("Cannot write to store journal : " + this.file.getAbsolutePath(), e);
        }
    }

    public void writeLong(long v) throws JournalException {
        try {
            if (this.writeBuffer.length - this.usedBufferSize < 8) {
                this.flushBuffer();
            }
            this.writeBuffer[this.usedBufferSize++] = (byte)(v >>> 56 & 0xFFL);
            this.writeBuffer[this.usedBufferSize++] = (byte)(v >>> 48 & 0xFFL);
            this.writeBuffer[this.usedBufferSize++] = (byte)(v >>> 40 & 0xFFL);
            this.writeBuffer[this.usedBufferSize++] = (byte)(v >>> 32 & 0xFFL);
            this.writeBuffer[this.usedBufferSize++] = (byte)(v >>> 24 & 0xFFL);
            this.writeBuffer[this.usedBufferSize++] = (byte)(v >>> 16 & 0xFFL);
            this.writeBuffer[this.usedBufferSize++] = (byte)(v >>> 8 & 0xFFL);
            this.writeBuffer[this.usedBufferSize++] = (byte)(v >>> 0 & 0xFFL);
            this.size += 8L;
        }
        catch (IOException e) {
            log.error((Object)("[" + this.baseName + "] Cannot write to store journal : " + this.file.getAbsolutePath()), (Throwable)e);
            throw new JournalException("Cannot write to store journal : " + this.file.getAbsolutePath(), e);
        }
    }

    public long size() {
        return this.size;
    }

    public boolean isComplete() {
        return this.complete;
    }

    public void setLastTransactionId(long lastTransactionId) {
        this.lastTransactionId = lastTransactionId;
    }

    public long getLastTransactionId() {
        return this.lastTransactionId;
    }

    public void closeAndDelete() throws JournalException {
        this.close();
        if (!this.file.delete() && this.file.exists()) {
            throw new JournalException("Cannot delete journal file : " + this.file.getAbsolutePath());
        }
    }

    public File closeAndRecycle() throws JournalException {
        File recycledFile = new File(this.file.getAbsolutePath() + RECYCLED_SUFFIX);
        if (!this.file.renameTo(recycledFile)) {
            throw new JournalException("Cannot rename journal file " + this.file.getAbsolutePath() + " to " + recycledFile.getAbsolutePath());
        }
        try {
            this.fillWithZeroes(recycledFile.length(), this.writeBuffer.length);
        }
        catch (IOException e) {
            throw new JournalException("Cannot clear journal file : " + recycledFile.getAbsolutePath(), e);
        }
        this.sync();
        this.close();
        return recycledFile;
    }

    public void close() throws JournalException {
        try {
            this.output.close();
            this.channel.close();
        }
        catch (IOException e) {
            log.error((Object)("[" + this.baseName + "] Cannot close journal file : " + this.file.getAbsolutePath()), (Throwable)e);
            throw new JournalException("Cannot close journal file", e);
        }
    }

    public String toString() {
        return this.file != null ? this.file.getAbsolutePath() : "?";
    }
}

