package io.vertx.mutiny.core;

import java.util.Map;
import java.util.stream.Collectors;
import io.smallrye.mutiny.Multi;
import io.smallrye.mutiny.Uni;
import java.util.function.Consumer;
import io.smallrye.mutiny.vertx.TypeArg;
import io.vertx.codegen.annotations.Fluent;
import io.smallrye.common.annotation.CheckReturnValue;
import java.util.List;
import io.vertx.core.json.JsonObject;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;
import io.vertx.core.Future;

/**
 * The execution context of a {@link io.vertx.core.Handler} execution.
 * <p>
 * When Vert.x provides an event to a handler or calls the start or stop methods of a {@link io.vertx.core.Verticle},
 * the execution is associated with a <code>Context</code>.
 * <p>
 * Usually a context is an *event-loop context* and is tied to a specific event loop thread. So executions for that
 * context always occur on that exact same event loop thread.
 * <p>
 * In the case of worker verticles and running inline blocking code a worker context will be associated with the execution
 * which will use a thread from the worker thread pool.
 * <p>
 * When a handler is set by a thread associated with a specific context, the Vert.x will guarantee that when that handler
 * is executed, that execution will be associated with the same context.
 * <p>
 * If a handler is set by a thread not associated with a context (i.e. a non Vert.x thread). Then a new context will
 * be created for that handler.
 * <p>
 * In other words, a context is propagated.
 * <p>
 * This means that when a verticle is deployed, any handlers it sets will be associated with the same context - the context
 * of the verticle.
 * <p>
 * This means (in the case of a standard verticle) that the verticle code will always be executed with the exact same
 * thread, so you don't have to worry about multi-threaded acccess to the verticle state and you can code your application
 * as single threaded.
 * <p>
 * This class also allows arbitrary data to be {@link io.vertx.mutiny.core.Context#put} and {@link io.vertx.mutiny.core.Context#get} on the context so it can be shared easily
 * amongst different handlers of, for example, a verticle instance.
 * <p>
 * This class also provides {@link io.vertx.mutiny.core.Context#runOnContext} which allows an action to be executed asynchronously using the same context.
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.Context original} non Mutiny-ified interface using Vert.x codegen.
 */

@io.smallrye.mutiny.vertx.MutinyGen(io.vertx.core.Context.class)
public class Context {

  public static final io.smallrye.mutiny.vertx.TypeArg<Context> __TYPE_ARG = new io.smallrye.mutiny.vertx.TypeArg<>(    obj -> new Context((io.vertx.core.Context) obj),
    Context::getDelegate
  );

  private final io.vertx.core.Context delegate;
  
  public Context(io.vertx.core.Context delegate) {
    this.delegate = delegate;
  }

  public Context(Object delegate) {
    this.delegate = (io.vertx.core.Context)delegate;
  }

  /**
   * Empty constructor used by CDI, do not use this constructor directly.
   **/
  Context() {
    this.delegate = null;
  }

  public io.vertx.core.Context getDelegate() {
    return delegate;
  }

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    Context that = (Context) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  /**
   * @return true if current thread is a worker thread, false otherwise
   */
  public static boolean isOnWorkerThread() { 
    boolean ret = io.vertx.core.Context.isOnWorkerThread();
    return ret;
  }

  /**
   * @return true if current thread is an event thread, false otherwise
   */
  public static boolean isOnEventLoopThread() { 
    boolean ret = io.vertx.core.Context.isOnEventLoopThread();
    return ret;
  }

  /**
   * @return true if current thread is a Vert.x thread, false otherwise
   */
  public static boolean isOnVertxThread() { 
    boolean ret = io.vertx.core.Context.isOnVertxThread();
    return ret;
  }

  /**
   * @param action the action to run
   */
  private void __runOnContext(Handler<Void> action) { 
    delegate.runOnContext(action);
  }

  /**
   * @param action the action to run
   */
  public void runOnContext(java.lang.Runnable action) {
    __runOnContext(ignored -> action.run()
);
  }

  /**
   * Safely execute some blocking code.
   * <p>
   * Executes the blocking code in the handler <code>blockingCodeHandler</code> using a thread from the worker pool.
   * <p>
   * When the code is complete the handler <code>resultHandler</code> will be called with the result on the original context
   * (e.g. on the original event loop of the caller).
   * <p>
   * A <code>Future</code> instance is passed into <code>blockingCodeHandler</code>. When the blocking code successfully completes,
   * the handler should call the {@link io.vertx.mutiny.core.Promise#complete} or {@link io.vertx.mutiny.core.Promise#complete} method, or the {@link io.vertx.mutiny.core.Promise#fail}
   * method if it failed.
   * <p>
   * The blocking code should block for a reasonable amount of time (i.e no more than a few seconds). Long blocking operations
   * or polling operations (i.e a thread that spin in a loop polling events in a blocking fashion) are precluded.
   * <p>
   * When the blocking operation lasts more than the 10 seconds, a message will be printed on the console by the
   * blocked thread checker.
   * <p>
   * Long blocking operations should use a dedicated thread managed by the application, which can interact with
   * verticles using the event-bus or {@link io.vertx.mutiny.core.Context#runOnContext}
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param blockingCodeHandler handler representing the blocking code to run
   * @param ordered if true then if executeBlocking is called several times on the same context, the executions for that context will be executed serially, not in parallel. if false then they will be no ordering guarantees
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public <T> io.smallrye.mutiny.Uni<T> executeBlocking(io.smallrye.mutiny.Uni<T> blockingCodeHandler, boolean ordered) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(resultHandler -> {
        delegate.executeBlocking(new Handler<io.vertx.core.Promise<T>>() {
          public void handle(io.vertx.core.Promise<T> event) {
            blockingCodeHandler.subscribe().with(it -> event.complete(it), failure -> event.fail(failure));
          }
      }, ordered, resultHandler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.Context#executeBlocking(Consumer,boolean)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param blockingCodeHandler handler representing the blocking code to run
   * @param ordered if true then if executeBlocking is called several times on the same context, the executions for that context will be executed serially, not in parallel. if false then they will be no ordering guarantees
   * @return the T instance produced by the operation.
   */
  public <T> T executeBlockingAndAwait(io.smallrye.mutiny.Uni<T> blockingCodeHandler, boolean ordered) { 
    return (T) executeBlocking(blockingCodeHandler, ordered).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.Context#executeBlocking(Consumer,boolean)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.Context#executeBlocking(Consumer,boolean)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.Context#executeBlocking(Consumer,boolean)} but you don't need to compose it with other operations.
   * @param blockingCodeHandler handler representing the blocking code to run
   * @param ordered if true then if executeBlocking is called several times on the same context, the executions for that context will be executed serially, not in parallel. if false then they will be no ordering guarantees
   */
  public <T> void executeBlockingAndForget(io.smallrye.mutiny.Uni<T> blockingCodeHandler, boolean ordered) { 
    executeBlocking(blockingCodeHandler, ordered).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * Invoke {@link io.vertx.mutiny.core.Context#executeBlocking} with order = true.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param blockingCodeHandler handler representing the blocking code to run
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public <T> io.smallrye.mutiny.Uni<T> executeBlocking(io.smallrye.mutiny.Uni<T> blockingCodeHandler) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(resultHandler -> {
        delegate.executeBlocking(new Handler<io.vertx.core.Promise<T>>() {
          public void handle(io.vertx.core.Promise<T> event) {
            blockingCodeHandler.subscribe().with(it -> event.complete(it), failure -> event.fail(failure));
          }
      }, resultHandler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.Context#executeBlocking(Consumer)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param blockingCodeHandler handler representing the blocking code to run
   * @return the T instance produced by the operation.
   */
  public <T> T executeBlockingAndAwait(io.smallrye.mutiny.Uni<T> blockingCodeHandler) { 
    return (T) executeBlocking(blockingCodeHandler).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.Context#executeBlocking(Consumer)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.Context#executeBlocking(Consumer)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.Context#executeBlocking(Consumer)} but you don't need to compose it with other operations.
   * @param blockingCodeHandler handler representing the blocking code to run
   */
  public <T> void executeBlockingAndForget(io.smallrye.mutiny.Uni<T> blockingCodeHandler) { 
    executeBlocking(blockingCodeHandler).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * @return the deployment ID of the deployment or null if not a Verticle deployment
   */
  public String deploymentID() { 
    String ret = delegate.deploymentID();
    return ret;
  }

  /**
   * @return the configuration of the deployment or null if not a Verticle deployment
   */
  public JsonObject config() { 
    JsonObject ret = delegate.config();
    return ret;
  }

  /**
   * @return 
   */
  public List<String> processArgs() { 
    List<String> ret = delegate.processArgs();
    return ret;
  }

  /**
   * @return true if false otherwise
   */
  public boolean isEventLoopContext() { 
    boolean ret = delegate.isEventLoopContext();
    return ret;
  }

  /**
   * @return true if the current context is a worker context, false otherwise
   */
  public boolean isWorkerContext() { 
    boolean ret = delegate.isWorkerContext();
    return ret;
  }

  /**
   * @param key the key of the data
   * @return the data
   */
  public <T> T get(java.lang.Object key) { 
    T ret = (T) delegate.get(key);
    return ret;
  }

  /**
   * @param key the key of the data
   * @param value the data
   */
  public void put(java.lang.Object key, java.lang.Object value) { 
    delegate.put(key, value);
  }

  /**
   * @param key the key to remove
   * @return true if removed successfully, false otherwise
   */
  public boolean remove(java.lang.Object key) { 
    boolean ret = delegate.remove(key);
    return ret;
  }

  /**
   * @param key the key of the data
   * @return the data
   */
  public <T> T getLocal(java.lang.Object key) { 
    T ret = (T) delegate.getLocal(key);
    return ret;
  }

  /**
   * @param key the key of the data
   * @param value the data
   */
  public void putLocal(java.lang.Object key, java.lang.Object value) { 
    delegate.putLocal(key, value);
  }

  /**
   * @param key the key to remove
   * @return true if removed successfully, false otherwise
   */
  public boolean removeLocal(java.lang.Object key) { 
    boolean ret = delegate.removeLocal(key);
    return ret;
  }

  /**
   * @return The Vertx instance that created the context
   */
  public io.vertx.mutiny.core.Vertx owner() { 
    io.vertx.mutiny.core.Vertx ret = io.vertx.mutiny.core.Vertx.newInstance((io.vertx.core.Vertx)delegate.owner());
    return ret;
  }

  /**
   * @return the number of instances of the verticle that were deployed in the deployment (if any) related to this context
   */
  public int getInstanceCount() { 
    int ret = delegate.getInstanceCount();
    return ret;
  }

  /**
   * @param handler the exception handler
   * @return a reference to this, so the API can be used fluently
   */
  @Fluent
  private io.vertx.mutiny.core.Context __exceptionHandler(Handler<java.lang.Throwable> handler) { 
    delegate.exceptionHandler(handler);
    return this;
  }

  /**
   * @param handler the exception handler
   * @return 
   */
  public io.vertx.mutiny.core.Context exceptionHandler(java.util.function.Consumer<java.lang.Throwable> handler) {
    return __exceptionHandler(handler != null ? handler::accept : null);
  }

  public static  Context newInstance(io.vertx.core.Context arg) {
    return arg != null ? new Context(arg) : null;
  }

}
