/*
 * #%L
 * JMexico :: Swing Editor
 * 
 * $Id: InputDesignTableModel.java 45 2012-02-07 14:47:08Z sletellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/jmexico/tags/jmexico-0.3/jmexico-editor/src/main/java/fr/reseaumexico/editor/InputDesignTableModel.java $
 * %%
 * Copyright (C) 2011 Réseau Mexico, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package fr.reseaumexico.editor;

import fr.reseaumexico.editor.factorValue.FactorValueEditorFactory;
import fr.reseaumexico.model.Factor;
import fr.reseaumexico.model.InputDesign;
import fr.reseaumexico.model.MexicoUtil;
import fr.reseaumexico.model.Scenario;
import fr.reseaumexico.model.ValueType;
import fr.reseaumexico.model.event.InputDesignScenarioEvent;
import fr.reseaumexico.model.event.InputDesignScenarioListener;
import jaxx.runtime.JAXXContext;
import org.apache.commons.lang3.ObjectUtils;

import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableModel;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import static org.nuiton.i18n.I18n._;

/**
 * @author sletellier <letellier@codelutin.com>
 * @since 0.1
 */
public class InputDesignTableModel extends AbstractTableModel implements InputDesignScenarioListener {

    private static final long serialVersionUID = 1L;

    protected TableModel delegate;

    protected InputDesign inputDesign;

    protected List<Factor> factors;

    protected FactorValueEditorFactory factory;

    public InputDesignTableModel(JAXXContext context) {
        // search specific factory in context
        factory = context.getContextValue(FactorValueEditorFactory.class);

        delegate = new DefaultTableModel();
    }

    public void setInputDesign(InputDesign inputDesign) {
        this.inputDesign = inputDesign;
        inputDesign.addInputDesignScenarioListener(this);

        // extract factors
        factors = new LinkedList<Factor>(inputDesign.getExperimentDesign().getFactor());

        // sort on id
//        Collections.sort(factors, new Comparator<Factor>() {
//
//            @Override
//            public int compare(Factor o1, Factor o2) {
//                return o1.getName().compareTo(o2.getName());
//            }
//        });

        factory.clearCache();

        fireTableStructureChanged();
    }

    @Override
    public void scenarioAdded(InputDesignScenarioEvent event) {
        fireTableStructureChanged();
    }

    @Override
    public void scenarioRemoved(InputDesignScenarioEvent event) {
        fireTableStructureChanged();
    }

    public InputDesign getInputDesign() {
        return inputDesign;
    }

    public Collection<Scenario> getScenarios() {
        return inputDesign.getScenario();
    }

    public Scenario getScenario(int i) {
        return inputDesign.getScenario(i);
    }

    public Factor getFactor(int i) {
        return factors.get(i);
    }

    @Override
    public int getRowCount() {
        if (inputDesign == null) {
            return delegate.getRowCount();
        }
        return factors == null ? 0 : factors.size();
    }

    @Override
    public int getColumnCount() {
        if (inputDesign == null) {
            return delegate.getColumnCount();
        }
        Collection<Scenario> scenarios = getScenarios();
        if (scenarios == null) {
            return 1;
        }
        return scenarios.size() + 1;
    }

    @Override
    public String getColumnName(int rowIndex) {
        if (inputDesign == null) {
            return delegate.getColumnName(rowIndex);
        }
        if (rowIndex == 0) {
            return _("jmexico.factor.name");
        }
        Scenario scenario = getScenario(rowIndex - 1);
        return scenario == null ? "" : scenario.getName();
    }

    @Override
    public Class<?> getColumnClass(int rowIndex) {
        if (inputDesign == null) {
            return delegate.getColumnClass(rowIndex);
        }
        if (rowIndex == 0) {
            return Factor.class;
        }
        return Object.class;
    }

    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        if (inputDesign == null) {
            return delegate.isCellEditable(rowIndex, columnIndex);
        }
        return columnIndex > 0;
    }

    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        if (inputDesign == null) {
            return delegate.getValueAt(rowIndex, columnIndex);
        }

        // get key on rowIndex
        Factor factor = getFactor(rowIndex);

        // first column is factors
        if (columnIndex == 0) {
            return factor;
        }

        // others are scenarios
        columnIndex = columnIndex - 1;
        Scenario scenario = getScenario(columnIndex);
        if (scenario == null) {
            return null;
        }
        Map<Factor, Object> factorValues = scenario.getFactorValues();

        Object result = factorValues.get(factor);
        return result;
    }

    @Override
    public void setValueAt(Object o, int rowIndex, int columnIndex) {
        if (inputDesign == null) {
            delegate.setValueAt(o, rowIndex, columnIndex);
            return;
        }
        if (columnIndex == 0) {

            // must never append
            return;
        }
        int scenarioIndex = columnIndex - 1;
        Scenario scenario = getScenario(scenarioIndex);

        // get key if rowIndex
        Factor factor = getFactor(rowIndex);

        // take care of type
        ValueType valueType = factor.getDomain().getValueType();

        String value = ObjectUtils.toString(o);
        scenario.setFactorValue(factor, MexicoUtil.getTypedValue(valueType, value));

        fireTableCellUpdated(rowIndex, columnIndex);
    }
}
