/*
 * #%L
 * MS-Access Importer
 * 
 * $Id: SexagecimalPosition.java 1302 2011-02-15 08:47:40Z chemit $
 * $HeadURL: https://svn.mpl.ird.fr/osiris/observe/msaccess-importer/tags/msaccess-importer-1.4.1/src/main/java/fr/ird/type/SexagecimalPosition.java $
 * %%
 * Copyright (C) 2010 - 2011 IRD, Codelutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package fr.ird.type;

import java.io.Serializable;

/**
 * Le modèle d'une position au format sexagecimal (degre - minute - seconde).
 *
 * @author chemit <chemit@codelutin.com>
 * @since 1.2
 */
public class SexagecimalPosition implements Serializable {

    private static final long serialVersionUID = 1L;

    protected Integer degre;

    protected Integer minute;

    protected Integer seconde;

    /**
     * Methode statique de fabrique de position a partir d'une valeur du format
     * decimal.
     * <p/>
     * Note : Si la valeur (au format decimal) vaut <code>null</code>, alors on
     * reinitialise les composants de la position a <code>null</code> et la
     * methode {@link #isNull()} vaudra alors {@code true}.
     *
     * @param decimal la valeur au format decimal
     * @return une nouvelle instance de position convertie
     */
    public static SexagecimalPosition valueOf(Float decimal) {
        SexagecimalPosition r = new SexagecimalPosition();
        r.update(decimal);
        return r;
    }

    /**
     * Methode statique de fabrique de position a partir d'une valeur du format
     * degre-minute-seconde.
     *
     * @param d la valeur des degres
     * @param m la valeur des minutes
     * @param s la valeur des secondes
     * @return une nouvelle instance de position convertie
     */
    public static SexagecimalPosition valueOf(int d, int m, int s) {
        SexagecimalPosition r = new SexagecimalPosition();
        r.setDegre(d);
        r.setMinute(m);
        r.setSeconde(s);

        return r;
    }

    /**
     * @return {@code true} si aucune composante n'est renseigné,
     *         {@code false} autrement.
     */
    public boolean isNull() {
        return degre == null && minute == null && seconde == null;
    }

    /**
     * Mets a jour les composants de la position a partir d'une valeur decimal.
     * <p/>
     * Note : Si la valeur (au format decimal) vaut <code>null</code>, alors on
     * reinitialise les composants de la position a <code>null</code> et la
     * methode {@link #isNull()} vaudra alors {@code true}.
     *
     * @param decimal la valeur decimale a convertir (qui peut etre nulle).
     */
    public void update(Float decimal) {
        Integer d = null;
        Integer m = null;
        Integer s = null;
        if (decimal != null) {
            int remain = 0;

            d = (int) (Math.round(decimal + 0.5) - 1);
            m = 0;
            s = 0;
            decimal = 60 * (decimal - d);
            if (decimal > 0) {
                m = (int) (Math.round(decimal + 0.5) - 1);
                decimal = 60 * (decimal - m);
                if (decimal > 0) {
                    s = (int) (Math.round(decimal + 0.5) - 1);
                    remain = (int) (10 * (decimal - s));
                }
            }
            if (remain > 9) {
                s++;
                remain = 0;
            }
            if (s == 60) {
                m++;
                s = 0;
            }
            if (m == 60) {
                d++;
                m = 0;
            }
        }
        degre = d;
        minute = m;
        seconde = s;
    }

    public Float toDecimal() {
        if (isNull()) {
            return null;
        }
        Integer d = degre == null ? 0 : degre;
        Integer m = minute == null ? 0 : minute;
        Integer s = seconde == null ? 0 : seconde;
        Float result = Float.valueOf(d);
        if (m > 0) {
            result += (float) m / 60;
            if (s == 0) {
                result += 0.5f / 3600;
            }
        }
        if (s > 0) {
            result += ((float) s + 0.5f) / 3600;
        }
        return result;
    }

    public Integer getDegre() {
        return degre;
    }

    public Integer getMinute() {
        return minute;
    }

    public Integer getSeconde() {
        return seconde;
    }

    public void setDegre(Integer degre) {
        this.degre = degre;
    }

    public void setMinute(Integer minute) {
        this.minute = minute;
    }

    public void setSeconde(Integer seconde) {
        this.seconde = seconde;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        SexagecimalPosition other = (SexagecimalPosition) obj;
        if (degre != other.degre && (degre == null || !degre.equals(other.degre))) {
            return false;
        }
        if (minute != other.minute && (minute == null || !minute.equals(other.minute))) {
            return false;
        }
        if (seconde != other.seconde && (seconde == null || !seconde.equals(other.seconde))) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        return super.toString() + "<" + degre + "° " + minute + "' " + seconde + "''>";
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 97 * hash + (degre != null ? degre.hashCode() : 0);
        hash = 97 * hash + (minute != null ? minute.hashCode() : 0);
        hash = 97 * hash + (seconde != null ? seconde.hashCode() : 0);
        return hash;
    }

    protected SexagecimalPosition() {
        // contructeur non publique, on prefere l'utilisation de la methode
        // #valueOf()
    }
}
