package fr.ifremer.shared.application.swing.table;

/*
 * #%L
 * Ifremer shared :: Application Swing
 * $Id: AbstractApplicationTableModel.java 1379 2013-11-24 22:44:33Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/application/tags/ifremer-shared-1.0/application-swing/src/main/java/fr/ifremer/shared/application/swing/table/AbstractApplicationTableModel.java $
 * %%
 * Copyright (C) 2013 Ifremer, CodeLutin, Tony CHEMIT
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import jaxx.runtime.SwingUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.table.TableColumnModelExt;

import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;
import java.io.Serializable;
import java.util.List;
import java.util.Set;

/**
 * Abstract model of a table.
 *
 * @param <R> type of a row.
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public abstract class AbstractApplicationTableModel<R extends Serializable> extends AbstractTableModel {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AbstractApplicationTableModel.class);

    /**
     * Data in the model.
     *
     * @since 0.2
     */
    protected List<R> rows;

    /**
     * Set of non editable columns.
     *
     * @since 0.2
     */
    protected Set<ColumnIdentifier<?>> noneEditableCols;

    /**
     * Creates a new row when moving to next editable cell / row ?
     *
     * @since 0.3
     */
    protected final boolean createNewRow;

    /**
     * Creates a first empty row when setting a null or empty list of rows ?
     *
     * @since 0.3
     */
    protected final boolean createEmptyRowIsEmpty;

    /**
     * Identifiers of columns (in initial order).
     *
     * @since 1.1
     */
    protected final List<ColumnIdentifier<R>> identifiers;

    public abstract R createNewRow();

    protected AbstractApplicationTableModel(TableColumnModelExt columnModel,
                                            boolean createNewRow,
                                            boolean createEmptyRowIsEmpty) {
        this.identifiers = Lists.newArrayListWithCapacity(columnModel.getColumnCount());
        for (TableColumn tc : columnModel.getColumns(true)) {
            this.identifiers.add((ColumnIdentifier<R>) tc.getIdentifier());
        }
        this.createNewRow = createNewRow;
        this.createEmptyRowIsEmpty = createEmptyRowIsEmpty;
    }

    public final List<R> getRows() {
        return rows;
    }

    public final void setRows(List<R> data) {

        // can't accept a empty data list
        Preconditions.checkNotNull(data, "Data list can not be null.");

        this.rows = null;
        if (createEmptyRowIsEmpty && data.isEmpty()) {

            if (log.isDebugEnabled()) {
                log.debug("Creates a first empty row on tableModel " + this);
            }
            // add a first edit line
            data.add(createNewRow());
        }
        if (log.isDebugEnabled()) {
            log.debug("Set " + data.size() + " row(s) in table model " + this);
        }
        this.rows = data;
        onRowsChanged(data);
        fireTableDataChanged();
    }

    public boolean isCreateNewRow() {
        return createNewRow;
    }

    public boolean isCreateEmptyRowIsEmpty() {
        return createEmptyRowIsEmpty;
    }

    public final void addNewRow() {
        R newValue = createNewRow();
        addNewRow(newValue);
    }

    public final void addNewRow(R newValue) {

        addNewRow(getRowCount(), newValue);
    }

    public final void addNewRow(int rowIndex, R newValue) {

        Preconditions.checkNotNull(newValue, "Row can not be null.");

        List<R> data = getRows();
        Preconditions.checkNotNull(data, "Data list can not be null.");

        data.add(rowIndex, newValue);

        onRowAdded(rowIndex, newValue);
        fireTableRowsInserted(rowIndex, rowIndex);
    }

    public final void fireTableRowsInserted(R newValue) {

        Preconditions.checkNotNull(newValue, "Row can not be null.");

        int rowIndex = getRowIndex(newValue);
        fireTableRowsInserted(rowIndex, rowIndex);
    }

    public final int updateRow(R row) {
        Preconditions.checkNotNull(row, "Row can not be null.");

        List<R> data = getRows();
        Preconditions.checkNotNull(data, "Data list can not be null.");

        int rowIndex = data.indexOf(row);

        fireTableRowsUpdated(rowIndex, rowIndex);
        return rowIndex;
    }

    public final R removeRow(int rowIndex) {
        SwingUtil.ensureRowIndex(this, rowIndex);

        List<R> data = getRows();

        R result = data.remove(rowIndex);

        fireTableRowsDeleted(rowIndex, rowIndex);
        return result;
    }

    protected void onRowsChanged(List<R> data) {
        // by default do nothing
    }

    protected void onRowAdded(int rowIndex, R newValue) {
        // by default do nothing
    }

    public final int getRowIndex(R row) {
        int result = rows == null ? -1 : rows.indexOf(row);
        return result;
    }

    public final R getEntry(int rowIndex) {
        SwingUtil.ensureRowIndex(this, rowIndex);
        List<R> data = getRows();
        R result = data == null ? null : data.get(rowIndex);
        return result;
    }

    public final void setNoneEditableCols(ColumnIdentifier<?>... noneEditableCols) {
        this.noneEditableCols = Sets.newHashSet(noneEditableCols);
    }

    @Override
    public final int getRowCount() {
        return rows == null ? 0 : rows.size();
    }

    @Override
    public final int getColumnCount() {
        return identifiers.size();
    }

    @Override
    public final Object getValueAt(int rowIndex, int columnIndex) {
        R entry = getEntry(rowIndex);
        ColumnIdentifier<R> identifier = getIdentifier(columnIndex);
        if (log.isDebugEnabled()) {
            log.debug("columnIndex: " + columnIndex + " :: " + identifier.getPropertyName());
        }
        Object result = identifier.getValue(entry);
        return result;
    }

    @Override
    public final void setValueAt(Object aValue, int rowIndex, int columnIndex) {
        if (log.isDebugEnabled()) {
            log.debug("setValueAt " + aValue);
        }
        R entry = getEntry(rowIndex);
        ColumnIdentifier<R> identifier = getIdentifier(columnIndex);
        setValueAt(aValue, rowIndex, columnIndex, identifier, entry);
    }

    @Override
    public final boolean isCellEditable(int rowIndex, int columnIndex) {
        ColumnIdentifier<R> identifier = getIdentifier(columnIndex);
        boolean result = isCellEditable(rowIndex, columnIndex, identifier);
        return result;
    }

    protected void setValueAt(Object aValue,
                              int rowIndex,
                              int columnIndex,
                              ColumnIdentifier<R> propertyName,
                              R entry) {
        if (log.isDebugEnabled()) {
            log.debug("setValueAt " + aValue);
        }
        propertyName.setValue(entry, aValue);
    }

    protected boolean isCellEditable(int rowIndex,
                                     int columnIndex,
                                     ColumnIdentifier<R> propertyName) {
        boolean result = !noneEditableCols.contains(propertyName);
        return result;
    }

    public final void fireTableCellUpdated(int rowIndex,
                                           ColumnIdentifier<R>... identifiers) {
        for (ColumnIdentifier<R> identifier : identifiers) {
            int columnIndex = this.identifiers.indexOf(identifier);
            fireTableCellUpdated(rowIndex, columnIndex);
        }
    }

    public final void fireTableRowUpdatedShell(Set<R> shell) {

        int minRowIndex1 = getColumnCount();
        int maxRowIndex1 = 0;

        for (R r : shell) {
            int rowIndex1 = getRowIndex(r);
            minRowIndex1 = Math.min(minRowIndex1, rowIndex1);
            maxRowIndex1 = Math.max(maxRowIndex1, rowIndex1);
        }
        fireTableRowsUpdated(minRowIndex1, maxRowIndex1);
    }

    protected void collectShell(R row, Set<R> collectedRows) {

        // by default just add the incoming row
        collectedRows.add(row);
    }

    protected ColumnIdentifier<R> getIdentifier(int columnIndex) {
        ColumnIdentifier<R> identifier = identifiers.get(columnIndex);
        return identifier;
    }
}
