package fr.ifremer.adagio.synchro.service.referential;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.*;
import fr.ifremer.adagio.core.AdagioTechnicalException;
import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.dao.technical.hibernate.TemporaryDataHelper;
import fr.ifremer.adagio.synchro.dao.administration.user.PersonSessionSynchroJdbcDao;
import fr.ifremer.adagio.synchro.dao.administration.user.PersonSessionSynchroJdbcDaoImpl;
import fr.ifremer.adagio.synchro.intercept.data.ObjectTypeHelper;
import fr.ifremer.adagio.synchro.meta.DatabaseColumns;
import fr.ifremer.adagio.synchro.meta.referential.ReferentialSynchroTables;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.config.SynchroConfiguration;
import fr.ifremer.common.synchro.dao.DaoFactory;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.dao.SynchroBaseDao;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.meta.*;
import fr.ifremer.common.synchro.service.SynchroContext;
import fr.ifremer.common.synchro.service.SynchroResult;
import fr.ifremer.common.synchro.service.SynchroTableOperation;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import javax.annotation.Nullable;
import javax.annotation.Resource;
import javax.sql.DataSource;
import java.io.File;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.*;

import static org.nuiton.i18n.I18n.t;

/**
 * A class defined as a Spring Bean
 * 
 * @author Benoit
 */
@Service("referentialSynchroService")
@Lazy
public class ReferentialSynchroServiceImpl
		extends fr.ifremer.common.synchro.service.SynchroServiceImpl
		implements ReferentialSynchroService {

	private static final Log log =
			LogFactory.getLog(ReferentialSynchroServiceImpl.class);

	private static boolean DISABLE_INTEGRITY_CONSTRAINTS = true;
	private static boolean ALLOW_MISSING_OPTIONAL_COLUMN = true;
	private static boolean ALLOW_ADDITIONAL_MANDATORY_COLUMN_IN_SOURCE_SCHEMA = true;
	private static boolean KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS = true;

	private static final String TABLE_DELETED_ITEM_HISTORY = ReferentialSynchroTables.DELETED_ITEM_HISTORY.name();

	/**
	 * Constants need for insertion into table TEMP_QUERY_PARAMETER, for delete by comparison
	 */
	private static final String TQP_DELETE_BY_COMPARISON_PREFIX = "DELETE#";
	private static final int TQP_DEFAULT_PERSON_ID = -1;

	// do not use a too big cache size, for referential tables
	// because one table should be processed only once
	private static int DAO_CACHE_SIZE = 2;

	@Resource
	private PersonSessionSynchroJdbcDao personSessionSynchroJdbcDao = null;

	/**
	 * <p>
	 * Constructor for ReferentialSynchroServiceImpl.
	 * </p>
	 * 
	 * @param dataSource
	 *            a {@link javax.sql.DataSource} object.
	 * @param config
	 *            a {@link fr.ifremer.common.synchro.config.SynchroConfiguration} object.
	 */
	@Autowired
	public ReferentialSynchroServiceImpl(DataSource dataSource, SynchroConfiguration config) {
		super(dataSource,
				config,
				DISABLE_INTEGRITY_CONSTRAINTS,
				ALLOW_MISSING_OPTIONAL_COLUMN,
				ALLOW_ADDITIONAL_MANDATORY_COLUMN_IN_SOURCE_SCHEMA,
				KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS);
		setDaoCacheSize(DAO_CACHE_SIZE);
	}

	/**
	 * <p>
	 * Constructor for ReferentialSynchroServiceImpl.
	 * </p>
	 */
	public ReferentialSynchroServiceImpl() {
		super(DISABLE_INTEGRITY_CONSTRAINTS,
				ALLOW_MISSING_OPTIONAL_COLUMN,
				ALLOW_ADDITIONAL_MANDATORY_COLUMN_IN_SOURCE_SCHEMA,
				KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS);
		setDaoCacheSize(DAO_CACHE_SIZE);
		personSessionSynchroJdbcDao = new PersonSessionSynchroJdbcDaoImpl();
	}

	/** {@inheritDoc} */
	@Override
	public ReferentialSynchroContext createSynchroContext(File sourceDbDirectory,
			SynchroDirection direction,
			int userId,
			Timestamp lastSynchronizationDate,
			boolean enableDelete,
			boolean enableInsertUpdate) {
		SynchroContext delegate = super.createSynchroContext(sourceDbDirectory,
				ReferentialSynchroTables.getImportTablesIncludes());

		// Make sure q2 connection properties are always used
		delegate.getTarget().putAllProperties(AdagioConfiguration.getInstance().getConnectionProperties());

		ReferentialSynchroContext result = new ReferentialSynchroContext(delegate, direction, userId);
		result.setLastSynchronizationDate(lastSynchronizationDate);
		result.setEnableDelete(enableDelete);
		result.setEnableInsertOrUpdate(enableInsertUpdate);
		initContext(result);

		return result;
	}

	/** {@inheritDoc} */
	@Override
	public ReferentialSynchroContext createSynchroContext(Properties sourceConnectionProperties,
			SynchroDirection direction,
			int userId,
			Timestamp lastSynchronizationDate,
			boolean enableDelete,
			boolean enableInsertUpdate) {
		SynchroContext delegate = super.createSynchroContext(sourceConnectionProperties,
				ReferentialSynchroTables.getImportTablesIncludes());

		// Make sure q2 connection properties are always used
		delegate.getTarget().putAllProperties(AdagioConfiguration.getInstance().getConnectionProperties());

		ReferentialSynchroContext result = new ReferentialSynchroContext(delegate, direction, userId);
		result.setLastSynchronizationDate(lastSynchronizationDate);
		result.setEnableDelete(enableDelete);
		result.setEnableInsertOrUpdate(enableInsertUpdate);
		initContext(result);

		return result;
	}

	/** {@inheritDoc} */
	@Override
	public void prepare(SynchroContext synchroContext) {
		Preconditions.checkArgument(synchroContext != null && synchroContext instanceof ReferentialSynchroContext,
				String.format("The context must be a instance of %s", ReferentialSynchroContext.class.getName()));

		ReferentialSynchroContext referentialSynchroContext = (ReferentialSynchroContext) synchroContext;
		SynchroDirection direction = referentialSynchroContext.getDirection();
		SynchroResult result = referentialSynchroContext.getResult();

		ReferentialSynchroDatabaseConfiguration target = (ReferentialSynchroDatabaseConfiguration) synchroContext.getTarget();
		ReferentialSynchroDatabaseConfiguration source = (ReferentialSynchroDatabaseConfiguration) synchroContext.getSource();

		source.excludeUnusedColumns();
		target.excludeUnusedColumns();

		// If Server DB -> Temp DB
		if (direction == SynchroDirection.IMPORT_SERVER2TEMP) {
			// Make sure personSessionId is filled
			if (referentialSynchroContext.getPersonSessionId() == null) {
				fillPersonSessionId(referentialSynchroContext, result);

				if (!result.isSuccess()) {
					return;
				}
			}
		}

		super.prepare(synchroContext);
	}

	/** {@inheritDoc} */
	@Override
	protected void prepareRootTable(
			DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory,
			SynchroTableMetadata table,
			SynchroContext context,
			SynchroResult result) throws SQLException {

		ReferentialSynchroContext referentialContext = (ReferentialSynchroContext) context;

		// Prepare from super class (insert and update)
		// (skip table DELETED_ITEM_HISTORY if deletes disable)
		if (referentialContext.isEnableInsertOrUpdate()
				&& (referentialContext.isEnableDelete()
				|| !TABLE_DELETED_ITEM_HISTORY.equalsIgnoreCase(table.getName()))) {

			super.prepareRootTable(sourceDaoFactory,
					targetDaoFactory,
					table,
					context,
					result);
		}

		// Count deleted rows for the current table, add add this count to result
		// (skip table DELETED_ITEM_HISTORY: could not delete itself !)
		if (referentialContext.isEnableDelete()
				&& !TABLE_DELETED_ITEM_HISTORY.equalsIgnoreCase(table.getName())) {
			prepareRootTableDeletes(sourceDaoFactory,
					targetDaoFactory,
					table,
					context,
					result);
		}
	}

	/** {@inheritDoc} */
	@Override
	public void synchronize(SynchroContext context) {
		super.synchronize(context);

		// Log if reject exists
		SynchroResult result = context.getResult();
		if (result.isSuccess() && !result.getRejectedRows().isEmpty()) {
			log.warn(I18n.t("adagio.synchro.synchronizeReferential.rejects", result.getRejectedRows().toString()));
		}
	}

	/** {@inheritDoc} */
	@Override
	public boolean hasChangesOnVesselTables(SynchroResult result) {
		Preconditions.checkNotNull(result);

		boolean hasVesselUpdate = result.getNbRows(ReferentialSynchroTables.VESSEL.name()) > 0
				|| result.getNbRows(ReferentialSynchroTables.VESSEL_FEATURES.name()) > 0
				|| result.getNbRows(ReferentialSynchroTables.VESSEL_OWNER.name()) > 0
				|| result.getNbRows(ReferentialSynchroTables.PERSON_SESSION_VESSEL.name()) > 0
				|| result.getNbRows(ReferentialSynchroTables.PERSON_SESSION_ITEM.name()) > 0;

		if (hasVesselUpdate && log.isDebugEnabled()) {
			log.debug("Vessels data has been updated");
		}

		return hasVesselUpdate;
	}

	/* -- Internal methods -- */

	/**
	 * <p>
	 * initContext.
	 * </p>
	 * 
	 * @param context
	 *            a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	protected void initContext(SynchroContext context) {

		// Set update date
		context.getTarget().setColumnUpdateDate(DatabaseColumns.UPDATE_DATE.name().toLowerCase());
		context.getSource().setColumnUpdateDate(DatabaseColumns.UPDATE_DATE.name().toLowerCase());
	}

	/**
	 * Count number of row to delete, for the given table
	 * 
	 * @param sourceDaoFactory
	 *            a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory
	 *            a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param context
	 *            a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param result
	 *            a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void prepareRootTableDeletes(
			DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory,
			SynchroTableMetadata table,
			SynchroContext context,
			SynchroResult result) throws SQLException {

		String tableName = table.getName();
		Set<String> objectTypeFks = ObjectTypeHelper.getObjectTypeFromTableName(tableName, tableName);

		if (CollectionUtils.isEmpty(objectTypeFks)) {
			return;
		}

		SynchroTableDao dihSourceDao = sourceDaoFactory.getSourceDao(TABLE_DELETED_ITEM_HISTORY);

		List<List<Object>> columnValues = Lists.newArrayListWithCapacity(objectTypeFks.size());
		for (String objectTypeFk : objectTypeFks) {
			columnValues.add(ImmutableList.<Object> of(objectTypeFk));
		}

		// Read rows of DELETED_ITEM_HISTORY (from the temp DB)
		Map<String, Object> bindings = createSelectBindingsForTable(context, TABLE_DELETED_ITEM_HISTORY);
		long count = dihSourceDao.countDataByFks(
				ImmutableSet.of(DatabaseColumns.OBJECT_TYPE_FK.name()),
				columnValues,
				bindings
				);
		if (count > 0) {
			result.addRows(tableName, (int) count);
		}
	}

	/** {@inheritDoc} */
	@Override
	protected List<SynchroTableOperation> getRootOperations(
			DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			SynchroContext context) throws SQLException {
		List<SynchroTableOperation> result = Lists.newArrayList();

		ReferentialSynchroContext referentialContext = (ReferentialSynchroContext) context;

		// Add default operations (insert and update)
		if (referentialContext.isEnableInsertOrUpdate()) {
			Collection<SynchroTableOperation> defaultOperations = super.getRootOperations(sourceDaoFactory, targetDaoFactory, dbMeta, context);
			result.addAll(defaultOperations);
		}

		// Add delete items history operation
		if (referentialContext.isEnableDelete()) {
			Collection<SynchroTableOperation> deletedItemOperations = getRootDeleteOperations(sourceDaoFactory, targetDaoFactory, dbMeta, context);
			result.addAll(deletedItemOperations);
		}
		return result;
	}

	private Collection<SynchroTableOperation> getRootDeleteOperations(
			DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			SynchroContext context) throws SQLException {
		Preconditions.checkArgument(dbMeta.getConfiguration().isFullMetadataEnable());

		Deque<SynchroTableOperation> result = Queues.newArrayDeque();

		SynchroTableDao dihSourceDao = sourceDaoFactory.getSourceDao(TABLE_DELETED_ITEM_HISTORY);
		Set<String> includedReferentialTables = context.getTableNames();

		// Read rows of DELETED_ITEM_HISTORY (from the temp DB)
		Map<String, Object> bindings = createSelectBindingsForTable(context, TABLE_DELETED_ITEM_HISTORY);

		ResultSet dihResultSet = null;
		List<Object> dihIdsToRemove = Lists.newArrayList();
		Set<String> tableNamesWithDelete = Sets.newHashSet();
		boolean doDelete = !context.getTarget().isMirrorDatabase();
		SynchroTableOperation previousDeletedOperation = null;
		SynchroTableOperation previousDeletedChildrenOperation = null;

		try {
			log.debug(I18n.t("adagio.synchro.synchronizeReferential.deletedItems"));

			dihResultSet = dihSourceDao.getData(bindings);

			while (dihResultSet.next()) {

				String objectType = dihResultSet.getString(DatabaseColumns.OBJECT_TYPE_FK.name());
				String tableName = ObjectTypeHelper.getTableNameFromObjectType(objectType);

				boolean isReferentialTable = StringUtils.isNotBlank(tableName)
						&& includedReferentialTables.contains(tableName.toUpperCase());

				if (isReferentialTable) {
					SynchroTableDao targetDao = targetDaoFactory.getSourceDao(tableName);
					SynchroTableMetadata table = targetDao.getTable();

					String objectCode = dihResultSet.getString(DatabaseColumns.OBJECT_CODE.name());
					String objectId = dihResultSet.getString(DatabaseColumns.OBJECT_ID.name());
					String vesselCode = dihResultSet.getString(DatabaseColumns.VESSEL_FK.name());

					// [Special case] VESSEL can use VESSEL_FK as an OBJECT_CODE (see mantis #24315)
					if (ReferentialSynchroTables.VESSEL.name().equalsIgnoreCase(tableName)
							&& StringUtils.isBlank(objectCode)
							&& StringUtils.isNotBlank(vesselCode)) {
						objectCode = vesselCode;
					}

					// Delete by a PK (id or code)
					if (StringUtils.isNotBlank(objectCode) || StringUtils.isNotBlank(objectId)) {

						if (doDelete) {
							List<Object> pk;
							// If composite key: deserialize
							if (StringUtils.isNotBlank(objectCode) && !table.isSimpleKey()) {
								pk = SynchroTableMetadata.fromPkStr(objectCode);
							}
							else {
								pk = StringUtils.isNotBlank(objectCode)
										? ImmutableList.<Object> of(objectCode)
										: ImmutableList.<Object> of(objectId);
							}

							boolean hasChildTables = table.hasChildJoins();

							// FIRST add children before parent deletion - before calling 'result.add(operation)'
							if (hasChildTables) {

								SynchroTableOperation childrenOperation;
								// group only consecutive rows (mantis #31722)
								if (previousDeletedChildrenOperation == null
										|| !previousDeletedChildrenOperation.getTableName().equalsIgnoreCase(tableName)) {
									if (previousDeletedChildrenOperation != null) {
										// Store previous children operation, before create a new one
										result.add(previousDeletedChildrenOperation);
									}
									childrenOperation = new SynchroTableOperation(tableName, context);
									previousDeletedChildrenOperation = childrenOperation; // became the current children
																							// operation
								}
								else {
									childrenOperation = previousDeletedChildrenOperation;
								}

								addChildrenToDelete(table, childrenOperation, ImmutableList.of(pk), result, context);
							}

							// group deletion by table (mantis #23535)
							// ONLY if consecutive table (mantis #31722)
							SynchroTableOperation operation;
							if (previousDeletedOperation == null || !previousDeletedOperation.getTableName().equalsIgnoreCase(tableName)) {
								if (previousDeletedOperation != null) {
									// Store previous operation, before create a new one
									result.add(previousDeletedOperation);
								}
								operation = new SynchroTableOperation(tableName, context);
								operation.setEnableProgress(true);
								previousDeletedOperation = operation; // became the current operation
							}
							else {
								operation = previousDeletedOperation;
							}

							operation.addMissingDelete(pk);
						}
					}

					// No id or code:
					// Should delete after a comparison between local's and remote's PKs
					else {
						tableNamesWithDelete.add(tableName);
					}
				}
			}

			// Make sure all operation has been added to result
			if (previousDeletedChildrenOperation != null) {
				result.add(previousDeletedChildrenOperation);
			}
			if (previousDeletedOperation != null) {
				result.add(previousDeletedOperation);
			}

		} finally {
			Daos.closeSilently(dihResultSet);
		}

		if (CollectionUtils.isNotEmpty(dihIdsToRemove)) {
			SynchroTableOperation operation = new SynchroTableOperation(TABLE_DELETED_ITEM_HISTORY, context);
			operation.addChildrenToDeleteFromOneColumn(TABLE_DELETED_ITEM_HISTORY, DatabaseColumns.REMOTE_ID.name().toLowerCase(), dihIdsToRemove);
			result.add(operation);
		}
		// If some deletion with no id nor code (only a table name)
		if (CollectionUtils.isNotEmpty(tableNamesWithDelete)) {

			// If target is a a temp DB (e.g. Server DB -> Temp DB)
			if (!doDelete) {
				saveTablesWithDelete(tableNamesWithDelete,
						sourceDaoFactory,
						targetDaoFactory,
						dbMeta,
						context);
			}

			// If source is a temp DB (e.g. Temp DB -> Local DB)
			else if (doDelete && context.getSource().isMirrorDatabase()) {
				addDeletedItemsFromTables(
						tableNamesWithDelete,
						result,
						sourceDaoFactory,
						targetDaoFactory,
						dbMeta,
						context);
			}

			// Else (could be Server DB -> Local DB)
			// (e.g. for unit test, or when direct connection to server DB)
			else if (doDelete) {
				saveTablesWithDelete(tableNamesWithDelete,
						sourceDaoFactory,
						targetDaoFactory,
						dbMeta,
						context);

				addDeletedItemsFromTables(
						tableNamesWithDelete,
						result,
						targetDaoFactory, // workaround, to read existing PK from target and not source DB
						targetDaoFactory,
						dbMeta,
						context);
			}
		}

		return result;
	}

	/**
	 * <p>
	 * addChildrenToDelete.
	 * </p>
	 * 
	 * @param parentTable
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param childrenOperation
	 *            a {@link fr.ifremer.common.synchro.service.SynchroTableOperation} object.
	 * @param parentPks
	 *            a {@link java.util.List} object.
	 * @param pendingOperations
	 *            a {@link java.util.Deque} object.
	 * @param context
	 *            a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	protected final void addChildrenToDelete(
			SynchroTableMetadata parentTable,
			SynchroTableOperation childrenOperation,
			List<List<Object>> parentPks,
			Deque<SynchroTableOperation> pendingOperations,
			SynchroContext context) {

		Set<String> pkNames = parentTable.getPkNames();

		// More than one PK: not implemented yet
		if (pkNames.size() > 1) {
			throw new UnsupportedOperationException("Not sure of this implementation: please check before comment out this exception !");
		}

		// First, add retrieve children joins and ADD into pendingOperations
		for (SynchroJoinMetadata join : parentTable.getChildJoins()) {
			SynchroTableMetadata childTable = join.getTargetTable();
			SynchroColumnMetadata childTableColumn = join.getTargetColumn();

			// Add child to delete into operation
			childrenOperation.addChildrenToDeleteFromManyColumns(childTable.getName(), ImmutableSet.of(childTableColumn.getName()), parentPks);
		}
	}

	/**
	 * <p>
	 * saveTablesWithDelete.
	 * </p>
	 * 
	 * @param tableNames
	 *            a {@link java.util.Set} object.
	 * @param sourceDaoFactory
	 *            a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory
	 *            a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param dbMeta
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param context
	 *            a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void saveTablesWithDelete(
			Set<String> tableNames,
			DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			SynchroContext context) throws SQLException {

		SynchroBaseDao targetBaseDao = targetDaoFactory.getDao();

		// Delete previous PKs for delete by comparison
		targetBaseDao.executeDeleteTempQueryParameter(TQP_DELETE_BY_COMPARISON_PREFIX + "%", true, TQP_DEFAULT_PERSON_ID);

		for (String tableName : tableNames) {

			SynchroTableDao sourceDao = sourceDaoFactory.getSourceDao(tableName);
			Set<String> pkStrs = sourceDao.getPksStr();

			targetBaseDao.executeInsertIntoTempQueryParameter(
					ImmutableList.<Object> copyOf(pkStrs),
					TQP_DELETE_BY_COMPARISON_PREFIX + tableName,
					TQP_DEFAULT_PERSON_ID
					);
		}
	}

	/**
	 * This method will create then add deleted operation, in the pending operations queue.<br>
	 * This need a TEMP_QUERY_PARAMETER filled with ID of deleted table
	 * 
	 * @param tableNames
	 *            a {@link java.util.Set} object.
	 * @param operations
	 *            a {@link java.util.Deque} object.
	 * @param sourceDaoFactory
	 *            a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory
	 *            a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param dbMeta
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param context
	 *            a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void addDeletedItemsFromTables(
			Set<String> tableNames,
			Deque<SynchroTableOperation> operations,
			DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			SynchroContext context) throws SQLException {

		// Create a DAO on TQP (TEMP_QUERY_PARAMETER) table
		SynchroTableDao tqpDao = sourceDaoFactory.getSourceDao(SynchroBaseDao.TEMP_QUERY_PARAMETER_TABLE);
		int tqpValueColumnIndex = tqpDao.getTable().getColumnIndex(SynchroBaseDao.TEMP_QUERY_PARAMETER_VALUE_COLUMN);
		Preconditions.checkArgument(tqpValueColumnIndex != -1);

		// Prepare some variables need to read rows on TQP
		Map<String, Object> emptyBinding = Maps.newHashMap();
		Set<String> fkNames = ImmutableSet.of(SynchroBaseDao.TEMP_QUERY_PARAMETER_PARAM_COLUMN);

		// For each row that has deletion
		for (String tableName : tableNames) {
			SynchroTableMetadata table = dbMeta.getTable(tableName);
			boolean hasChildTables = table.hasChildJoins();
			Set<String> tablePkNames = table.getPkNames();
			int pkCount = tablePkNames.size();

			// Retrieve PK Str (stored by method 'saveTablesWithDelete()')
			List<Object> fkValue = ImmutableList.<Object> of(TQP_DELETE_BY_COMPARISON_PREFIX + tableName);
			ResultSet rs = tqpDao.getDataByFks(
					fkNames,
					ImmutableList.of(fkValue),
					emptyBinding);
			List<List<Object>> pks = Lists.newArrayList();
			while (rs.next()) {
				String pkStr = rs.getString(tqpValueColumnIndex + 1);
				List<Object> pk = SynchroTableMetadata.fromPkStr(pkStr);

				// Make sure the PK str was well formed
				if (pkCount != pk.size()) {
					String expectedPkStrExample = Joiner.on(String.format(">%s<", SynchroTableMetadata.PK_SEPARATOR)).join(tablePkNames);
					throw new SynchroTechnicalException(String.format(
							"Unable to import delete on %s: invalid PK found in the source database (in %s). Should have %s column (e.g. %s).",
							tableName,
							SynchroBaseDao.TEMP_QUERY_PARAMETER_TABLE,
							pkCount,
							expectedPkStrExample));
				}
				pks.add(pk);
			}
			rs.close();

			// Check TQP has been correctly filled
			if (pks.size() == 0) {
				throw new SynchroTechnicalException(String.format(
						"Unable to import delete on %s: No PK found in the source database (in %s). Unable to compare and find PKs to delete.",
						tableName,
						SynchroBaseDao.TEMP_QUERY_PARAMETER_TABLE));
			}

			SynchroTableOperation operation = new SynchroTableOperation(tableName, context);
			operation.setEnableProgress(true);
			SynchroTableDao targetTableDao = targetDaoFactory.getTargetDao(tableName, null, operation);

			List<List<Object>> pksToDelete = targetTableDao.getPksByNotFoundFks(
					tablePkNames,
					pks,
					emptyBinding);

			// Excluded temporary rows (keep temporary rows)
			pksToDelete = filterExcludeTemporary(table, pksToDelete);

			// If some rows need to be deleted
			if (CollectionUtils.isNotEmpty(pksToDelete)) {
				// Fill the operation as a 'delete operation'
				operation.addAllMissingDelete(pksToDelete);

				// If has children, add child deletion to result
				if (hasChildTables) {
					addDeleteChildrenToDeque(table, pksToDelete, operations, context);
				}

				// Add operation to the result list
				operations.add(operation);
			}

		}

		// Clean processed row from TempQueryParameter
		targetDaoFactory.getDao().executeDeleteTempQueryParameter(TQP_DELETE_BY_COMPARISON_PREFIX + "%", true, TQP_DEFAULT_PERSON_ID);
	}

	/**
	 * <p>
	 * filterExcludeTemporary.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param pks
	 *            a {@link java.util.List} object.
	 * @return a {@link java.util.List} object.
	 */
	protected List<List<Object>> filterExcludeTemporary(
			SynchroTableMetadata table,
			List<List<Object>> pks) {
		Set<String> tablePkNames = table.getPkNames();

		if (tablePkNames.size() > 1) {
			return pks;
		}

		SynchroColumnMetadata pkColumn = table.getColumn(tablePkNames.iterator().next());
		Collection<List<Object>> result;

		// If pk is a numeric (e.g. an ID column)
		if (SynchroMetadataUtils.isNumericType(pkColumn)) {
			result = Collections2.filter(pks,
					new Predicate<List<Object>>() {
						@Override
						public boolean apply(@Nullable List<Object> input) {
							long pk = Long.parseLong(input.get(0).toString());
							return !TemporaryDataHelper.isTemporaryId(pk);
						}
					});
		}

		// If pk is a string (e.g. a CODE column)
		else {

			result = Collections2.filter(pks,
					new Predicate<List<Object>>() {
						@Override
						public boolean apply(@Nullable List<Object> input) {
							String pk = input.get(0).toString();
							return !TemporaryDataHelper.isTemporaryCode(pk);
						}
					});
		}

		return ImmutableList.copyOf(result);
	}

	/**
	 * <p>
	 * fillPersonSessionId.
	 * </p>
	 * 
	 * @param referentialSynchroContext
	 *            a {@link fr.ifremer.adagio.synchro.service.referential.ReferentialSynchroContext} object.
	 * @param result
	 *            a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 */
	protected void fillPersonSessionId(ReferentialSynchroContext referentialSynchroContext, SynchroResult result) {
		Preconditions.checkNotNull(referentialSynchroContext.getPersonId(),
				"One of 'personId' or 'personSessionId' must be set in the synchro context");

		result.getProgressionModel().setMessage(t("adagio.synchro.synchronizeData.initPersonSession"));
		if (log.isInfoEnabled()) {
			log.info(t("adagio.synchro.synchronizeData.initPersonSession.log", referentialSynchroContext.getPersonId()));
		}

		try {
			int personSessionId = personSessionSynchroJdbcDao.initPersonSession(
					referentialSynchroContext.getSource().getConnectionProperties(),
					referentialSynchroContext.getPersonId());
			if (log.isDebugEnabled()) {
				log.debug(String.format("Session initialized: id=%s", personSessionId));
			}

			referentialSynchroContext.setPersonSessionId(personSessionId);
		} catch (AdagioTechnicalException e) {
			result.setError(e);
		}

	}

	/** {@inheritDoc} */
	@Override
	protected Map<String, Object> createSelectBindingsForTable(SynchroContext context, String tableName) {

		Map<String, Object> result = super.createSelectBindingsForTable(context, tableName);

		// If table synchronization is forced (e.g. person_session could be forced, when user has no rights yet)
		ReferentialSynchroContext referentialSynchroContext = (ReferentialSynchroContext) context;
		if (CollectionUtils.isNotEmpty(referentialSynchroContext.getTableNamesForced())
				&& referentialSynchroContext.getTableNamesForced().contains(tableName.toUpperCase())) {
			log.debug(String.format("[%s] Forced synchronization (last synchronization date ignored)", tableName));
			result.remove(SynchroTableMetadata.UPDATE_DATE_BINDPARAM);
		}

		return result;
	}

}
