package fr.ifremer.adagio.synchro.service.referential;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import fr.ifremer.common.synchro.service.SynchroContext;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.common.synchro.service.SynchroResult;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.io.File;
import java.sql.Timestamp;
import java.util.Properties;

/**
 * <p>
 * ReferentialSynchroService interface.
 * </p>
 * 
 */
@Transactional(propagation = Propagation.SUPPORTS)
public interface ReferentialSynchroService {

	/**
	 * Create a synchro context, from a source database directory (HSQLDB only), and with a last synchronization date.
	 * This use local connection properties as source properties, but change JDBC URL to use the given directory
	 * 
	 * @param sourceDbDirectory
	 *            a directory with a HSQLDB directory.
	 * @param direction
	 *            synchronisation direction (e.g. TEMP DB to SERVER DB)
	 * @param userId
	 *            user identifier
	 * @param lastSynchronizationDate
	 *            the last synchro date
	 * @param enableDelete
	 *            need to synchronize delete ?
	 * @param enableInsertUpdate
	 *            need to synchronize insert/update ?
	 * @return a new context for synchronization
	 */
	ReferentialSynchroContext createSynchroContext(File sourceDbDirectory,
			SynchroDirection direction,
			int userId,
			Timestamp lastSynchronizationDate,
			boolean enableDelete,
			boolean enableInsertUpdate);

	/**
	 * Create a synchro context, using configuration properties for target and source properties, and with a last
	 * synchronization date.
	 * 
	 * @param sourceConnectionProperties
	 *            Properties to use as source connection (could be Oracle, HSQLDB...)
	 * @param direction
	 *            synchronisation direction (e.g. TEMP DB to SERVER DB)
	 * @param userId
	 *            user identifier
	 * @param lastSynchronizationDate
	 *            the last synchro date
	 * @param enableDelete
	 *            need to synchronize delete ?
	 * @param enableInsertUpdate
	 *            need to synchronize insert/update ?
	 * @return a new context for synchronization
	 */
	ReferentialSynchroContext createSynchroContext(Properties sourceConnectionProperties,
			SynchroDirection direction,
			int userId,
			Timestamp lastSynchronizationDate,
			boolean enableDelete,
			boolean enableInsertUpdate);

	/**
	 * Prepare the synchronize operation from the target data database supported
	 * by this service, says just compute nb rows to update for each table and
	 * update the result model.
	 * 
	 * @param synchroContext
	 *            Context of synchronization
	 */
	@Transactional(readOnly = false, propagation = Propagation.REQUIRES_NEW)
	void prepare(SynchroContext synchroContext);

	/**
	 * Launch the synchronize operation from the target data database supported
	 * by this service.
	 * 
	 * @param synchroContext
	 *            Context of synchronization
	 */
	@Transactional(readOnly = false, propagation = Propagation.REQUIRED)
	void synchronize(SynchroContext synchroContext);

	/**
	 * Retrieve the max(update_date) in the source database
	 * 
	 * @param synchroContext
	 *            a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @return a {@link java.sql.Timestamp} object.
	 */
	@Transactional(readOnly = true, propagation = Propagation.REQUIRED)
	Timestamp getSourceLastUpdateDate(SynchroContext synchroContext);

	/**
	 * Return if tables relative to Vessels has been changes <br>
	 * e.g. Vessel, VesselFeatures, VesselOwner...
	 * 
	 * @param result
	 *            a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @return a boolean.
	 */
	boolean hasChangesOnVesselTables(SynchroResult result);
}
