package fr.ifremer.adagio.synchro.service.doc;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2017 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.ifremer.adagio.core.dao.technical.gson.GsonUtils;
import fr.ifremer.adagio.core.vo.synchro.SynchroTableMetaVO;
import fr.ifremer.adagio.core.vo.synchro.SynchroTableRelationsVO;
import fr.ifremer.adagio.synchro.meta.DatabaseColumns;
import fr.ifremer.adagio.synchro.meta.data.DataSynchroTables;
import fr.ifremer.adagio.synchro.meta.referential.ReferentialSynchroTables;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.adagio.synchro.service.data.DataSynchroContext;
import fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration;
import fr.ifremer.adagio.synchro.service.referential.ReferentialSynchroContext;
import fr.ifremer.adagio.synchro.service.referential.ReferentialSynchroDatabaseConfiguration;
import fr.ifremer.common.synchro.config.SynchroConfiguration;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.SynchroJoinMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import java.io.File;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.*;

/**
 * Created by Ludovic on 07/12/2016.
 */
@Service("docSynchroService")
@Lazy
public class DocSynchroServiceImpl implements DocSynchroService {

	private static final Log log = LogFactory.getLog(DocSynchroServiceImpl.class);
	private static final SynchroDirection DEFAULT_SYNCHRO_DIRECTION = SynchroDirection.IMPORT_SERVER2TEMP;

	@Autowired
	public DocSynchroServiceImpl(SynchroConfiguration config) {
	}

	@Override
	public void createTableRelationsFile(Properties connectionProperties, File tableRelationFileCache) throws SQLException {
		createTableRelationsFile(connectionProperties, tableRelationFileCache, DEFAULT_SYNCHRO_DIRECTION);
	}

	@Override
	public void createTableRelationsFile(Properties connectionProperties, File tableRelationFileCache, SynchroDirection synchroDirection)
			throws SQLException {

		Preconditions.checkArgument(Daos.isValidConnectionProperties(connectionProperties));
		Preconditions.checkNotNull(tableRelationFileCache);
		Preconditions.checkNotNull(synchroDirection);

		log.info(I18n.t("adagio.synchro.doc.createTableRelations", tableRelationFileCache.getPath()));

		Connection connection = Daos.createConnection(connectionProperties);

		// table names to read
		Set<String> tableNames = Sets.newHashSet(ReferentialSynchroTables.getImportTablesIncludes());

		// REFERENTIAL
		ReferentialSynchroDatabaseConfiguration referentialConfiguration = new ReferentialSynchroDatabaseConfiguration(
				new ReferentialSynchroContext(null, synchroDirection, -1, -1),
				connectionProperties, false);
		referentialConfiguration.setFullMetadataEnable(true);
		referentialConfiguration.setColumnUpdateDate(DatabaseColumns.UPDATE_DATE.name().toLowerCase());

		// exclude unused columns todo depend de direction
		referentialConfiguration.excludeUnusedColumns();

		// read metadata
		SynchroDatabaseMetadata referentialMetadata = new SynchroDatabaseMetadata(connection, referentialConfiguration);
		referentialMetadata.prepare(tableNames);
		referentialMetadata.close();

		// DATA
		DataSynchroDatabaseConfiguration dataConfiguration = new DataSynchroDatabaseConfiguration(
				new DataSynchroContext(null, synchroDirection, -1, -1),
				connectionProperties, false);
		dataConfiguration.setFullMetadataEnable(true);
		dataConfiguration.setColumnRemoteId(DatabaseColumns.REMOTE_ID.name().toLowerCase());
		dataConfiguration.setColumnSynchronizationStatus(DatabaseColumns.SYNCHRONIZATION_STATUS.name().toLowerCase());

		// exclude unused columns todo depend de direction
		dataConfiguration.excludeUnusedColumns();

		// read metadata for referential + data tables to get all relations
		tableNames.addAll(DataSynchroTables.getImportTablesIncludes());
		SynchroDatabaseMetadata dataMetadata = new SynchroDatabaseMetadata(connection, dataConfiguration);
		dataMetadata.prepare(tableNames);
		dataMetadata.close();

		// close connection
		Daos.closeSilently(connection);

		// Build tableRelations
		SynchroTableRelationsVO tableRelations = new SynchroTableRelationsVO();

		for (String tableName : referentialMetadata.getLoadedTableNames()) {
			populateTableRelations(tableRelations, synchroDirection, referentialMetadata.getLoadedTable(tableName), false);
		}

		for (String tableName : dataMetadata.getLoadedTableNames()) {
			populateTableRelations(tableRelations, synchroDirection, dataMetadata.getLoadedTable(tableName), true);
		}

		// add custom relations
		addMoreRelations(tableRelations, synchroDirection);

		// write file
		GsonUtils.serializeToFile(tableRelations, tableRelationFileCache);
	}

	protected void addMoreRelations(SynchroTableRelationsVO tableRelations, SynchroDirection synchroDirection) {
		// do nothing by default
	}

	protected boolean isJoinValid(SynchroJoinMetadata join, SynchroDirection synchroDirection) {
		return join.isValid();
	}

	private void populateTableRelations(SynchroTableRelationsVO tableRelations, SynchroDirection synchroDirection, SynchroTableMetadata table,
			boolean isData) {
		populateTableRelations(tableRelations, synchroDirection, table, isData, Collections.<String> emptyList());
	}

	private void populateTableRelations(SynchroTableRelationsVO tableRelations, SynchroDirection synchroDirection, SynchroTableMetadata table,
			boolean isData, Collection<String> tablesNamesToIgnore) {

		SynchroTableMetaVO tableMeta = tableRelations.getOrCreateTable(table.getName(), table.isRoot(), table.isWithUpdateDateColumn(), isData);

		if (log.isDebugEnabled()) {
			log.debug(String.format("populateTableRelations in %s for %s", isData ? "DATA" : "REFERENTIAL", tableMeta.toExtendedString()));
		}

		if (tablesNamesToIgnore.contains(table.getName())) {
			if (log.isDebugEnabled()) {
				log.debug(String.format("table %s has been ignored", tableMeta.toExtendedString()));
			}
			return;
		}

		for (SynchroJoinMetadata join : table.getChildJoins()) {
			SynchroTableMetadata joinTable = join.getTargetTable();

			boolean joinValid = isJoinValid(join, synchroDirection);
			if (!joinValid
					|| tableRelations.exists(joinTable.getName(), !isData)
					|| table.getName().equalsIgnoreCase(joinTable.getName())
					|| tablesNamesToIgnore.contains(joinTable.getName())) {
				if (log.isDebugEnabled()) {
					if (!joinValid) {
						log.debug(String.format("join from %s to %s is not valid", tableMeta.toExtendedString(), joinTable.getName()));
					}
					if (tableRelations.exists(joinTable.getName(), !isData)) {
						log.debug(String.format("joined table %s already exists in referential", joinTable.getName()));
					}
					if (table.getName().equalsIgnoreCase(joinTable.getName())) {
						log.debug(String.format("join from %s to %s is same table", tableMeta.toExtendedString(), joinTable.getName()));
					}
					if (tablesNamesToIgnore.contains(joinTable.getName())) {
						log.debug(String.format("joined table %s has been ignored", joinTable.getName()));
					}
				}
				continue;
			}

			SynchroTableMetaVO joinTableMeta = tableRelations.getOrCreateTable(joinTable.getName(), joinTable.isRoot(),
					joinTable.isWithUpdateDateColumn(), isData);
			tableRelations.addChildTable(tableMeta, joinTableMeta);
			if (log.isDebugEnabled()) {
				log.debug(String.format("table %s is child of %s", joinTableMeta.toExtendedString(), tableMeta.toExtendedString()));
			}
		}
	}

	@Override
	public List<SynchroTableMetaVO> getAffectedTablesForUpdateHierarchy(SynchroTableRelationsVO tableRelations, SynchroTableMetaVO table) {
		List<SynchroTableMetaVO> result = Lists.newArrayList();
		SynchroTableMetaVO tableCopy = new SynchroTableMetaVO(table);

		for (SynchroTableMetaVO rootTable : getFirstRootParentTables(tableRelations, tableCopy)) {
			result.add(populateRelationTables(tableRelations, rootTable, tableCopy).setToUpdate(true));
		}

		return result;
	}

	@Override
	public List<SynchroTableMetaVO> getAffectedTablesForDeleteHierarchy(SynchroTableRelationsVO tableRelations, SynchroTableMetaVO table) {
		List<SynchroTableMetaVO> result = Lists.newArrayList();
		SynchroTableMetaVO tableCopy = new SynchroTableMetaVO(table);

		// process parents
		for (SynchroTableMetaVO rootTable : getFirstRootParentTables(tableRelations, tableCopy)) {
			result.add(populateRelationTables(tableRelations, rootTable, tableCopy).setToUpdate(true));
		}

		// process children on root table
		if (table.isRoot()) {

			// first pass on root children
			for (SynchroTableMetaVO child : tableRelations.getChildTables(table)) {
				if (child.isRoot() && (table.isData() != child.isData())) {
					SynchroTableMetaVO childCopy = new SynchroTableMetaVO(child);
					childCopy.addChildren(tableCopy);
					result.add(childCopy.setToUpdate(true));
				}
			}

			// second pass on non-root children
			for (SynchroTableMetaVO child : tableRelations.getChildTables(table)) {
				if (!child.isRoot() && (table.isData() != child.isData())) {
					for (SynchroTableMetaVO firstRootParent : getFirstRootParentTables(tableRelations, child, false)) {
						if (!result.contains(firstRootParent) && !table.equals(firstRootParent)) {
							result.add(populateRelationTables(tableRelations, firstRootParent, tableCopy).setToUpdate(true));
						}
					}
				}
			}

		} else {

			// process children on non root table
			for (SynchroTableMetaVO itsParent : getFirstRootParentTables(tableRelations, table, false)) {
				if (!result.contains(itsParent) && !table.equals(itsParent)) {
					result.add(populateRelationTables(tableRelations, itsParent, tableCopy).setToUpdate(true));
				}
			}

		}

		return result;
	}

	protected List<SynchroTableMetaVO> getFirstRootParentTables(SynchroTableRelationsVO tableRelations, SynchroTableMetaVO table) {
		return getFirstRootParentTables(tableRelations, table, false);
	}

	protected List<SynchroTableMetaVO> getFirstRootParentTables(SynchroTableRelationsVO tableRelations, SynchroTableMetaVO table,
			boolean dataTableOnly) {
		if (table.isRoot() && !table.isData()) {
			return Collections.emptyList();
		}
		Set<SynchroTableMetaVO> parents = Sets.newHashSet();
		for (SynchroTableMetaVO parentTable : findParentContainsChild(tableRelations, table)) {
			boolean dataOnly = dataTableOnly || table.isData();

			if (parentTable.isRoot() && parentTable.isData() == dataOnly) {
				parents.add(parentTable);
			}
			if (dataOnly) {
				parents.addAll(getFirstRootParentTables(tableRelations, parentTable, true));
			}
		}
		List<SynchroTableMetaVO> result = Lists.newArrayList(parents);
		Collections.sort(result);
		return ImmutableList.copyOf(parents);
	}

	private List<SynchroTableMetaVO> findParentContainsChild(SynchroTableRelationsVO tableRelations, SynchroTableMetaVO targetChildTable) {
		List<SynchroTableMetaVO> result = Lists.newArrayList();
		for (SynchroTableMetaVO parentTable : tableRelations.getChildTableMap().keySet()) {
			if (tableRelations.getChildTables(parentTable).contains(targetChildTable)) {
				result.add(parentTable);
			}
		}
		return ImmutableList.copyOf(result);
	}

	private SynchroTableMetaVO populateRelationTables(SynchroTableRelationsVO tableRelations, SynchroTableMetaVO base, SynchroTableMetaVO target) {
		SynchroTableMetaVO result = new SynchroTableMetaVO(base);
		List<SynchroTableMetaVO> nextChildTables = getNextChildTable(tableRelations, base, target);
		if (!nextChildTables.isEmpty()) {
			for (SynchroTableMetaVO nextChild : nextChildTables) {
				if (nextChild.equals(target)) {
					result.addChildren(new SynchroTableMetaVO(nextChild));
				} else {
					result.addChildren(populateRelationTables(tableRelations, nextChild, target));
				}
			}
		} else {
			// get first parent
			for (SynchroTableMetaVO nextParent : getNextParentTable(tableRelations, base, target)) {
				if (nextParent.equals(target)) {
					result.addChildren(new SynchroTableMetaVO(nextParent));
				}
			}
		}
		return result;
	}

	private List<SynchroTableMetaVO> getNextChildTable(SynchroTableRelationsVO tableRelations, SynchroTableMetaVO base, SynchroTableMetaVO target) {
		List<SynchroTableMetaVO> result = Lists.newArrayList();
		// get children containing target
		for (SynchroTableMetaVO child : tableRelations.getChildTables(base)) {
			if (child.equals(target)) {
				result.add(child);
			} else if (findTableInChildren(tableRelations, child, target)) {
				// child relation
				result.add(child);
			} else if (findTableInParents(tableRelations, child, target)) {
				result.add(child);
			}
		}
		Collections.sort(result);
		return ImmutableList.copyOf(result);
	}

	private List<SynchroTableMetaVO> getNextParentTable(SynchroTableRelationsVO tableRelations, SynchroTableMetaVO base, SynchroTableMetaVO target) {
		List<SynchroTableMetaVO> result = Lists.newArrayList();
		// get parents containing target
		for (SynchroTableMetaVO parent : findParentContainsChild(tableRelations, base)) {
			if (parent.equals(target)) {
				result.add(parent);
			}
		}
		Collections.sort(result);
		return ImmutableList.copyOf(result);
	}

	private boolean findTableInChildren(SynchroTableRelationsVO tableRelations, SynchroTableMetaVO table, SynchroTableMetaVO target) {
		for (SynchroTableMetaVO child : tableRelations.getChildTables(table)) {
			if (child.equals(target)) {
				return true;
			}
		}
		for (SynchroTableMetaVO child : tableRelations.getChildTables(table)) {
			if (findTableInChildren(tableRelations, child, target)) {
				return true;
			}
			if (findTableInParents(tableRelations, child, target)) {
				return true;
			}
		}
		return false;
	}

	private boolean findTableInParents(SynchroTableRelationsVO tableRelations, SynchroTableMetaVO table, SynchroTableMetaVO target) {
		for (SynchroTableMetaVO parent : findParentContainsChild(tableRelations, table)) {
			if (parent.equals(target)) {
				return true;
			}
		}
		return false;
	}

	protected boolean notContains(List<SynchroTableMetaVO> tables, SynchroTableMetaVO tableToFind) {
		if (tables.contains(tableToFind)) {
			return false;
		}

		for (SynchroTableMetaVO table : tables) {
			boolean notFound = notContains(table.getChildren(), tableToFind);
			if (!notFound) {
				return false;
			}
		}

		return true;
	}

}
