package fr.ifremer.adagio.synchro.service.data;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.*;
import fr.ifremer.adagio.core.AdagioTechnicalException;
import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.dao.technical.DaoUtils;
import fr.ifremer.adagio.core.dao.technical.DateUtils;
import fr.ifremer.adagio.core.dao.technical.hibernate.ConfigurationHelper;
import fr.ifremer.adagio.core.dao.technical.synchronization.SynchronizationStatus;
import fr.ifremer.adagio.synchro.dao.administration.user.PersonSessionSynchroJdbcDao;
import fr.ifremer.adagio.synchro.dao.administration.user.PersonSessionSynchroJdbcDaoImpl;
import fr.ifremer.adagio.synchro.intercept.data.ObjectTypeHelper;
import fr.ifremer.adagio.synchro.meta.DatabaseColumns;
import fr.ifremer.adagio.synchro.meta.data.DataSynchroTables;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.config.SynchroConfiguration;
import fr.ifremer.common.synchro.dao.DaoFactory;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.service.*;
import fr.ifremer.common.synchro.type.ProgressionModel;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.dialect.Dialect;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.sql.DataSource;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.*;

import static org.nuiton.i18n.I18n.t;

/**
 * <p>
 * DataSynchroServiceImpl class.
 * </p>
 */
@Service("dataSynchroService")
@Lazy
public class DataSynchroServiceImpl
		extends fr.ifremer.common.synchro.service.SynchroServiceImpl
		implements DataSynchroService {

	private static final Log log = LogFactory.getLog(DataSynchroServiceImpl.class);

	private static boolean DISABLE_INTEGRITY_CONSTRAINTS = false;
	private static boolean ALLOW_MISSING_OPTIONAL_COLUMN = true;
	private static boolean ALLOW_ADDITIONAL_MANDATORY_COLUMN_IN_SOURCE_SCHEMA = false;
	private static boolean KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS = false;

	private static final String TABLE_DELETED_ITEM_HISTORY = DataSynchroTables.DELETED_ITEM_HISTORY.name();
	private static final String COLUMN_SYNCHRONIZATION_STATUS = DatabaseColumns.SYNCHRONIZATION_STATUS.name().toLowerCase();
	private static final String COLUMN_REMOTE_ID = DatabaseColumns.REMOTE_ID.name().toLowerCase();

	@Resource
	private PersonSessionSynchroJdbcDao personSessionSynchroJdbcDao = null;

	@Resource
	private AdagioConfiguration adagioConfig = null;

	private final int exportUpdateDateDelayInSecond;

	/**
	 * <p>
	 * Constructor for DataSynchroServiceImpl.
	 * </p>
	 * 
	 * @param dataSource
	 *            a {@link javax.sql.DataSource} object.
	 * @param config
	 *            a {@link fr.ifremer.common.synchro.config.SynchroConfiguration} object.
	 */
	@Autowired
	public DataSynchroServiceImpl(DataSource dataSource, SynchroConfiguration config) {
		super(dataSource, config,
				DISABLE_INTEGRITY_CONSTRAINTS,
				ALLOW_MISSING_OPTIONAL_COLUMN,
				ALLOW_ADDITIONAL_MANDATORY_COLUMN_IN_SOURCE_SCHEMA,
				KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS);
		exportUpdateDateDelayInSecond = AdagioConfiguration.getInstance().getExportDataUpdateDateDelayInSecond();
	}

	/**
	 * <p>
	 * Constructor for DataSynchroServiceImpl.
	 * </p>
	 */
	public DataSynchroServiceImpl() {
		super(DISABLE_INTEGRITY_CONSTRAINTS,
				ALLOW_MISSING_OPTIONAL_COLUMN,
				ALLOW_ADDITIONAL_MANDATORY_COLUMN_IN_SOURCE_SCHEMA,
				KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS);
		adagioConfig = AdagioConfiguration.getInstance();
		exportUpdateDateDelayInSecond = adagioConfig.getExportDataUpdateDateDelayInSecond();
		personSessionSynchroJdbcDao = new PersonSessionSynchroJdbcDaoImpl();
	}

	/** {@inheritDoc} */
	@Override
	public DataSynchroContext createSynchroContext(File sourceDbDirectory, SynchroDirection direction, int userId) {
		Preconditions.checkNotNull(sourceDbDirectory);
		Preconditions.checkArgument(sourceDbDirectory.exists() && sourceDbDirectory.isDirectory());
		return createSynchroContext(sourceDbDirectory, direction, userId, null, true, true);
	}

	/** {@inheritDoc} */
	@Override
	public DataSynchroContext createSynchroContext(File sourceDbDirectory, SynchroDirection direction, int userId,
			Timestamp lastSynchronizationDate) {
		return createSynchroContext(sourceDbDirectory, direction, userId, lastSynchronizationDate, true, true);
	}

	/** {@inheritDoc} */
	@Override
	public DataSynchroContext createSynchroContext(File sourceDbDirectory, SynchroDirection direction, int userId,
			Timestamp lastSynchronizationDate, boolean enableDelete, boolean enableInsertUpdate) {
		SynchroContext delegate = super.createSynchroContext(sourceDbDirectory, DataSynchroTables.getImportTablesIncludes());

		// Make sure q2 connection properties are always used
		delegate.getTarget().putAllProperties(AdagioConfiguration.getInstance().getConnectionProperties());

		DataSynchroContext result = new DataSynchroContext(delegate, direction, userId);
		result.setLastSynchronizationDate(lastSynchronizationDate);
		result.setEnableDelete(enableDelete);
		result.setEnableInsertOrUpdate(enableInsertUpdate);
		initContext(result);
		return result;
	}

	/** {@inheritDoc} */
	@Override
	public DataSynchroContext createSynchroContext(Properties sourceConnectionProperties, SynchroDirection direction, int userId) {
		return createSynchroContext(sourceConnectionProperties, direction, userId, null, true, true);
	}

	/** {@inheritDoc} */
	@Override
	public DataSynchroContext createSynchroContext(Properties sourceConnectionProperties, SynchroDirection direction, int userId,
			Timestamp lastSynchronizationDate) {
		return createSynchroContext(sourceConnectionProperties, direction, userId, lastSynchronizationDate, true, true);
	}

	/** {@inheritDoc} */
	@Override
	public DataSynchroContext createSynchroContext(Properties sourceConnectionProperties, SynchroDirection direction, int userId,
			Timestamp lastSynchronizationDate, boolean enableDelete, boolean enableInsertUpdate) {
		Preconditions.checkNotNull(sourceConnectionProperties);
		SynchroContext delegate = super.createSynchroContext(sourceConnectionProperties, DataSynchroTables.getImportTablesIncludes());

		// Make sure q2 connection properties are always used
		delegate.getTarget().putAllProperties(AdagioConfiguration.getInstance().getConnectionProperties());

		DataSynchroContext result = new DataSynchroContext(delegate, direction, userId);
		result.setLastSynchronizationDate(lastSynchronizationDate);
		result.setEnableDelete(enableDelete);
		result.setEnableInsertOrUpdate(enableInsertUpdate);
		initContext(result);
		return result;
	}

	/** {@inheritDoc} */
	@Override
	public void prepare(SynchroContext synchroContext) {
		Preconditions.checkArgument(synchroContext != null && synchroContext instanceof DataSynchroContext,
				String.format("The context must be a instance of %s", DataSynchroContext.class.getName()));

		DataSynchroContext dataSynchroContext = (DataSynchroContext) synchroContext;
		SynchroResult result = dataSynchroContext.getResult();
		SynchroDirection direction = dataSynchroContext.getDirection();

		DataSynchroDatabaseConfiguration target = (DataSynchroDatabaseConfiguration) dataSynchroContext.getTarget();
		DataSynchroDatabaseConfiguration source = (DataSynchroDatabaseConfiguration) dataSynchroContext.getSource();

		// If server DB -> Temp DB
		if (direction == SynchroDirection.IMPORT_SERVER2TEMP) {

			// Set target database as mirror AND temporary
			target.setIsMirrorDatabase(true);

			// Ignore some not used columns
			target.excludeUnusedColumns();

			// Ignore some tables for schema checking (mantis #31457)
			target.addCheckSchemaTableExclude(DataSynchroTables.QUANTIFICATION_MEASUREMENT.name());
			source.addCheckSchemaTableExclude(DataSynchroTables.QUANTIFICATION_MEASUREMENT.name());
			target.addCheckSchemaTableExclude(DataSynchroTables.SORTING_MEASUREMENT.name());
			source.addCheckSchemaTableExclude(DataSynchroTables.SORTING_MEASUREMENT.name());

			// Disable integrity constraints
			{
				disableIntegrityConstraints(target.getConnectionProperties(), result);
				if (!result.isSuccess()) {
					return;
				}

				// No need to check unicity of input rows (server rows are safe)
				source.setCheckUniqueConstraintBetweenInputRows(false);
				target.setCheckUniqueConstraintBetweenInputRows(false);
			}

			// Make sure personSessionId is filled
			if (dataSynchroContext.getPersonSessionId() == null) {
				fillPersonSessionId(dataSynchroContext, result);

				if (!result.isSuccess()) {
					return;
				}
			}

			// Compute the default period, if need
			if (dataSynchroContext.getDataStartDate() == null
					|| dataSynchroContext.getDataEndDate() == null) {
				fillDefaultDataPeriod(dataSynchroContext);
			}

			// Compute the default PK restriction, if need
			if (dataSynchroContext.getPkIncludes() == null) {
				fillDefaultPkIncludes(dataSynchroContext);
			}
		}

		// If Temp DB -> Local DB
		else if (direction == SynchroDirection.IMPORT_TEMP2LOCAL) {

			// Set target database as not a Mirror
			target.setIsMirrorDatabase(false);

			// No need to check unicity of input rows (rows come from server and are safe)
			source.setCheckUniqueConstraintBetweenInputRows(false);
			target.setCheckUniqueConstraintBetweenInputRows(false);
		}

		// If local -> temporary database
		else if (direction == SynchroDirection.EXPORT_LOCAL2TEMP) {

			// Set target database as mirror AND temporary
			target.setIsMirrorDatabase(true);

			// Disable integrity constraints
			{
				disableIntegrityConstraints(target.getConnectionProperties(), result);
				if (!result.isSuccess()) {
					return;
				}
			}

			// No need to check unicity of input rows (will be done on temp->server)
			source.setCheckUniqueConstraintBetweenInputRows(false);
			target.setCheckUniqueConstraintBetweenInputRows(false);
		}

		// If Temp DB -> server DB
		else if (direction == SynchroDirection.EXPORT_TEMP2SERVER) {

			// Ignore some not used columns
			source.excludeUnusedColumns();
			source.setFullMetadataEnable(true);

			// Ignore some tables for schema checking (mantis #31457)
			target.addCheckSchemaTableExclude(DataSynchroTables.QUANTIFICATION_MEASUREMENT.name());
			source.addCheckSchemaTableExclude(DataSynchroTables.QUANTIFICATION_MEASUREMENT.name());
			target.addCheckSchemaTableExclude(DataSynchroTables.SORTING_MEASUREMENT.name());
			source.addCheckSchemaTableExclude(DataSynchroTables.SORTING_MEASUREMENT.name());

			// Set target database as not a Mirror
			target.setIsMirrorDatabase(false);
			target.setFullMetadataEnable(false);

			// Fill system timestamp into context, and copy it to source
			fillSystemTimestampWithDelay(result, target);
			source.setSystemTimestamp(target.getSystemTimestamp());
		}

		super.prepare(synchroContext);
	}

	/** {@inheritDoc} */
	@Override
	public void synchronize(SynchroContext synchroContext) {
		Preconditions.checkArgument(synchroContext != null && synchroContext instanceof DataSynchroContext,
				String.format("The context must be a instance of %s", DataSynchroContext.class.getName()));

		DataSynchroContext dataSynchroContext = (DataSynchroContext) synchroContext;
		SynchroResult result = dataSynchroContext.getResult();
		List<Multimap<String, String>> pkIncludesListForBatch = null;

		DataSynchroDatabaseConfiguration target = dataSynchroContext.getTarget();
		DataSynchroDatabaseConfiguration source = dataSynchroContext.getSource();

		// If server DB -> temp DB
		if (dataSynchroContext.getDirection() == SynchroDirection.IMPORT_SERVER2TEMP) {

			// Enable protected columns
			target.removeColumnExclude(COLUMN_SYNCHRONIZATION_STATUS);
			target.removeColumnExclude(COLUMN_REMOTE_ID);

			// Disable integrity constraints
			{
				disableIntegrityConstraints(dataSynchroContext.getTarget().getConnectionProperties(), result);
				if (!result.isSuccess()) {
					return;
				}
			}
		}

		// If Temp DB -> server DB
		else if (dataSynchroContext.getDirection() == SynchroDirection.EXPORT_TEMP2SERVER) {

			// Enable protected columns
			source.removeColumnExclude(COLUMN_SYNCHRONIZATION_STATUS);
			source.removeColumnExclude(COLUMN_REMOTE_ID);
		}

		// If Temp DB -> Local DB
		else if (dataSynchroContext.getDirection() == SynchroDirection.IMPORT_TEMP2LOCAL) {

			// Use batch (split import - see mantis #27275)
			// only if:
			// - insert/update are enable
			// - pk filter not already exists
			if (dataSynchroContext.isEnableInsertOrUpdate()
					&& (dataSynchroContext.getPkIncludes() == null || dataSynchroContext.getPkIncludes().isEmpty())) {
				pkIncludesListForBatch = computePkIncludesListForBatch(result, dataSynchroContext);
			}
		}

		boolean synchronizeUsingBatch = CollectionUtils.isNotEmpty(pkIncludesListForBatch);

		// Synchronize all
		if (!synchronizeUsingBatch
				|| !dataSynchroContext.isEnableInsertOrUpdate()) {
			super.synchronize(synchroContext);
		}

		// Synchronize using batch (see mantis #27275)
		else {
			synchronizeUsingBatch(synchroContext, pkIncludesListForBatch);
		}

	}

	@Override
	protected void cleanSynchroResult(SynchroContext context, SynchroResult result) {
		super.cleanSynchroResult(context, result);

		// Remove missing updates on MEASUREMENT_FILE on UPDATE_DATE witch are not in REMOTE_ID updates
		// Because the UPDATE_DATE value of each MEASUREMENT_FILE is maintained on the server side (see
		// MeasurementFileInterceptor)
		Map<String, Map<String, Object>> measurementFileMissingUpdates = MapUtils.isNotEmpty(result.getSourceMissingUpdates())
				? result.getSourceMissingUpdates().get(DataSynchroTables.MEASUREMENT_FILE.name())
				: null;
		if (MapUtils.isNotEmpty(measurementFileMissingUpdates)) {

			// find both remote_id and update_date updates
			Map<String, Object> remoteIdUpdates = measurementFileMissingUpdates.get(DatabaseColumns.REMOTE_ID.name().toLowerCase());
			Map<String, Object> updateDateUpdates = measurementFileMissingUpdates.get(DatabaseColumns.UPDATE_DATE.name().toLowerCase());
			if (MapUtils.isNotEmpty(updateDateUpdates)) {

				if (MapUtils.isNotEmpty(remoteIdUpdates)) {

					// remove the update_date updates of lines without a remote_id updates
					Set<String> validKeys = remoteIdUpdates.keySet();
					Set<String> updateDateUpdatesToRemove = Sets.newHashSet(updateDateUpdates.keySet());
					updateDateUpdatesToRemove.removeAll(validKeys);
					int nbRemoves = 0;
					for (String updateDateUpdateToRemove : updateDateUpdatesToRemove) {
						updateDateUpdates.remove(updateDateUpdateToRemove);
						nbRemoves++;
					}
					if (log.isTraceEnabled()) {
						log.trace(String.format("[%s] %s updates on %s have been removed from result",
								DataSynchroTables.MEASUREMENT_FILE.name(),
								nbRemoves,
								DatabaseColumns.UPDATE_DATE.name()));
					}

				} else {

					// No new MEASUREMENT_FILE, so remove all update_date updates
					measurementFileMissingUpdates.remove(DatabaseColumns.UPDATE_DATE.name().toLowerCase());
					if (log.isTraceEnabled()) {
						log.trace(String.format("[%s] all updates on %s have been removed from result",
								DataSynchroTables.MEASUREMENT_FILE.name(),
								DatabaseColumns.UPDATE_DATE.name()));
					}
				}
			}
		}
	}

	/* -- Internal methods -- */
	/**
	 * <p>
	 * initContext.
	 * </p>
	 * 
	 * @param context
	 *            a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 */
	protected void initContext(SynchroContext context) {

		// Set update date
		context.getSource().setColumnUpdateDate(DatabaseColumns.UPDATE_DATE.name().toLowerCase());
		context.getTarget().setColumnUpdateDate(DatabaseColumns.UPDATE_DATE.name().toLowerCase());

	}

	/** {@inheritDoc} */
	@Override
	protected void prepareRootTable(
			DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory,
			SynchroTableMetadata table,
			SynchroContext context,
			SynchroResult result) throws SQLException {

		DataSynchroContext dataContext = (DataSynchroContext) context;

		// Prepare from super class (insert and update)
		// (skip table DELETED_ITEM_HISTORY if deletes disable)
		if (dataContext.isEnableInsertOrUpdate()
				&& (dataContext.isEnableDelete()
				|| !TABLE_DELETED_ITEM_HISTORY.equalsIgnoreCase(table.getName()))) {
			super.prepareRootTable(sourceDaoFactory,
					targetDaoFactory,
					table,
					context,
					result);
		}

		// Count deleted rows for the current table, add add this count to result
		// (skip table DELETED_ITEM_HISTORY: could not delete itself !)
		if (dataContext.isEnableDelete()
				&& !TABLE_DELETED_ITEM_HISTORY.equalsIgnoreCase(table.getName())) {
			prepareRootTableDeletes(sourceDaoFactory,
					targetDaoFactory,
					table,
					context,
					result);
		}
	}

	/**
	 * Count number of row to delete, for the given table
	 * 
	 * @param sourceDaoFactory
	 *            a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param targetDaoFactory
	 *            a {@link fr.ifremer.common.synchro.dao.DaoFactory} object.
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param context
	 *            a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param result
	 *            a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void prepareRootTableDeletes(
			DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory,
			SynchroTableMetadata table,
			SynchroContext context,
			SynchroResult result) throws SQLException {

		String tableName = table.getName();
		Set<String> objectTypeFks = ObjectTypeHelper.getObjectTypeFromTableName(tableName);

		if (CollectionUtils.isEmpty(objectTypeFks)) {
			return;
		}

		SynchroTableDao dihSourceDao = sourceDaoFactory.getSourceDao(TABLE_DELETED_ITEM_HISTORY);

		List<List<Object>> columnValues = Lists.newArrayListWithCapacity(objectTypeFks.size());
		for (String objectTypeFk : objectTypeFks) {
			columnValues.add(ImmutableList.<Object> of(objectTypeFk));
		}

		// Read rows of DELETED_ITEM_HISTORY (from the temp DB)
		Map<String, Object> bindings = createSelectBindingsForTable(context, TABLE_DELETED_ITEM_HISTORY);
		long count = dihSourceDao.countDataByFks(
				ImmutableSet.of(DatabaseColumns.OBJECT_TYPE_FK.name()),
				columnValues,
				bindings
				);
		if (count > 0) {
			result.addRows(tableName, (int) count);
		}
	}

	/** {@inheritDoc} */
	@Override
	protected Map<String, Object> createDefaultSelectBindings(SynchroContext context) {

		Map<String, Object> bindings = super.createDefaultSelectBindings(context);

		if (context instanceof DataSynchroContext) {
			DataSynchroContext dataContext = (DataSynchroContext) context;

			// Fill with the current person_fk (need for Dao and insert into TempQueryParemeter)
			bindings.put("userId", dataContext.getPersonId());

			// Fill the period (using in interceptor queries - see ObservedLocationInterceptor)
			if (dataContext.getDataStartDate() != null) {
				bindings.put("startDate", new Timestamp(dataContext.getDataStartDate().getTime()));
				bindings.put("endDate", new Timestamp(dataContext.getDataEndDate().getTime()));
			}
		}

		return bindings;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * Override the default method, to disable this binding: <br>
	 * <code>bindings.put(SynchroTableMetadata.UPDATE_DATE_BINDPARAM, context.getResult().getUpdateDate(tableName))</code>
	 */
	@Override
	protected Map<String, Object> createSelectBindingsForTable(SynchroContext context, String tableName) {
		// Get defaults binding from context
		Map<String, Object> bindings = Maps.newHashMap(getSelectBindings(context));

		Timestamp lastSynchronizationDate = context.getLastSynchronizationDate();
		boolean notEmptyTargetTable = (context.getResult().getUpdateDate(tableName) != null);
		boolean enableUpdateDateFilter = (lastSynchronizationDate != null)
				&& (context.getTarget().isMirrorDatabase() || notEmptyTargetTable);

		if (enableUpdateDateFilter) {
			bindings.put(SynchroTableMetadata.UPDATE_DATE_BINDPARAM, lastSynchronizationDate);
		}

		return bindings;
	}

	/**
	 * <p>
	 * fillSystemTimestampWithDelay.
	 * </p>
	 * 
	 * @param result
	 *            a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param databaseConfiguration
	 *            a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 */
	protected void fillSystemTimestampWithDelay(SynchroResult result, SynchroDatabaseConfiguration databaseConfiguration) {

		result.getProgressionModel().setMessage(t("adagio.synchro.synchronizeData.initSystemTimestamp"));
		if (log.isInfoEnabled()) {
			log.info(t("adagio.synchro.synchronizeData.initSystemTimestamp"));
		}

		Connection connection = null;
		try {
			connection = createConnection(databaseConfiguration);
			Dialect dialect = databaseConfiguration.getDialect();
			Timestamp currentTimestamp = DaoUtils.getCurrentTimestamp(connection, dialect);

			// Add a delay, to allow concurrent importation to reimport this data later
			DateUtils.addSeconds(currentTimestamp, exportUpdateDateDelayInSecond);

			if (log.isDebugEnabled()) {
				log.debug(String.format("Current timestamp: %s", currentTimestamp));
			}

			databaseConfiguration.setSystemTimestamp(currentTimestamp);
		} catch (AdagioTechnicalException e) {
			result.setError(e);
		} catch (SQLException e) {
			result.setError(e);
		} finally {
			closeSilently(connection);
		}
	}

	/**
	 * <p>
	 * fillDefaultDataPeriod.
	 * </p>
	 * 
	 * @param dataSynchroContext
	 *            a {@link fr.ifremer.adagio.synchro.service.data.DataSynchroContext} object.
	 */
	protected void fillDefaultDataPeriod(DataSynchroContext dataSynchroContext) {

		dataSynchroContext.setDataStartDate(DataSynchroUtils.getDefaultStartDate());
		dataSynchroContext.setDataEndDate(DataSynchroUtils.getDefaultEndDate());

		if (log.isDebugEnabled()) {
			log.debug(String.format("Context has no start date for data import: will use default period, from [%s] to [%s]",
					dataSynchroContext.getDataStartDate(), dataSynchroContext.getDataEndDate()));
		}
	}

	/**
	 * <p>
	 * fillDefaultPkIncludes.
	 * </p>
	 * 
	 * @param dataSynchroContext
	 *            a {@link fr.ifremer.adagio.synchro.service.data.DataSynchroContext} object.
	 */
	protected void fillDefaultPkIncludes(DataSynchroContext dataSynchroContext) {
		String confProperty = adagioConfig.getImportDataPkIncludes();
		Multimap<String, String> pkIncludes = ConfigurationHelper.getMultimap(confProperty);
		dataSynchroContext.setPkIncludes(pkIncludes);
	}

	/** {@inheritDoc} */
	@Override
	protected List<SynchroTableOperation> getRootOperations(
			DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			SynchroContext context) throws SQLException {
		DataSynchroContext dataContext = (DataSynchroContext) context;
		List<SynchroTableOperation> result = Lists.newArrayList();

		// Add delete items history operation - if enable in context
		// NOTE : deletion should be done BEFORE insertion (to avoid duplication error) - mantis ObsDeb #27022
		if (dataContext.isEnableDelete()
				&& (dataContext.getDirection() == SynchroDirection.EXPORT_TEMP2SERVER
				|| dataContext.getDirection() == SynchroDirection.IMPORT_TEMP2LOCAL)) {

			Collection<SynchroTableOperation> deletedItemOperations = getDeleteOperations(sourceDaoFactory, targetDaoFactory, dbMeta, dataContext);
			result.addAll(deletedItemOperations);
		}

		// Add default operations (insert/update) - if enable in context
		if (dataContext.isEnableInsertOrUpdate()) {
			Collection<SynchroTableOperation> defaultOperations = super.getRootOperations(sourceDaoFactory, targetDaoFactory, dbMeta, context);
			result.addAll(defaultOperations);
		}

		return result;
	}

	private Collection<SynchroTableOperation> getDeleteOperations(
			DaoFactory sourceDaoFactory,
			DaoFactory targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			DataSynchroContext context) throws SQLException {
		Preconditions.checkArgument(dbMeta.getConfiguration().isFullMetadataEnable());

		Deque<SynchroTableOperation> result = Queues.newArrayDeque();

		SynchroTableDao dihSourceDao = sourceDaoFactory.getSourceDao(TABLE_DELETED_ITEM_HISTORY);
		Set<String> includedDataTables = DataSynchroTables.getImportTablesIncludes();

		// Read rows of DELETED_ITEM_HISTORY (from the temp DB)
		Map<String, Object> bindings = createSelectBindingsForTable(context, TABLE_DELETED_ITEM_HISTORY);

		ResultSet dihResultSet = null;
		List<Object> dihIdsToRemove = Lists.newArrayList();

		try {
			dihResultSet = dihSourceDao.getData(bindings);

			while (dihResultSet.next()) {

				long dihId = dihResultSet.getLong(DatabaseColumns.ID.name());
				String objectType = dihResultSet.getString(DatabaseColumns.OBJECT_TYPE_FK.name());
				String tableName = ObjectTypeHelper.getTableNameFromObjectType(objectType);

				boolean isDataTable = StringUtils.isNotBlank(tableName)
						&& includedDataTables.contains(tableName.toUpperCase());

				if (isDataTable) {
					SynchroTableDao targetDao = targetDaoFactory.getSourceDao(tableName);
					SynchroTableMetadata table = targetDao.getTable();
					boolean shouldDeleteUsingRemoteId = (context.getDirection() == SynchroDirection.IMPORT_TEMP2LOCAL);

					String objectCode = dihResultSet.getString(DatabaseColumns.OBJECT_CODE.name());
					String objectId = dihResultSet.getString(DatabaseColumns.OBJECT_ID.name());

					// Delete by a PK (id or code)
					if (StringUtils.isNotBlank(objectCode) || StringUtils.isNotBlank(objectId)) {
						List<Object> pk = StringUtils.isNotBlank(objectCode)
								? ImmutableList.<Object> of(objectCode)
								: ImmutableList.<Object> of(objectId);

						boolean hasChildTables = targetDao.getTable().hasChildJoins();

						SynchroTableOperation operation = new SynchroTableOperation(tableName, context);
						operation.setEnableProgress(true);
						result.add(operation);

						// If import to local DB : object_id = table's remote_id
						if (shouldDeleteUsingRemoteId) {
							if (table.getColumnIndex(COLUMN_REMOTE_ID) != -1) {
								operation.addChildrenToDeleteFromOneColumn(tableName, COLUMN_REMOTE_ID, pk);

								// Make sure DIH is deleted, if exported from this local DB
								dihIdsToRemove.add(dihId);
							} else if (log.isWarnEnabled()) {
								// This should never happen !
								log.warn(String
										.format("[%s] Found deleted items on table [%s] (pk: [%s]) but no column [%s] exists on this table. Skipping this deleted items.",
												TABLE_DELETED_ITEM_HISTORY,
												tableName,
												SynchroTableMetadata.toPkStr(pk),
												COLUMN_REMOTE_ID
										));
							}

							// Children table should NOT be add here - mantis #23799
							// This is done later, inside method SynchroServiceImpl.synchronizeChildrenToDeletes()
						}

						// If export to server DB : object_id = table's id
						else {
							operation.addMissingDelete(pk);

							// If has children, add child deletion to result
							if (hasChildTables) {
								addDeleteChildrenToDeque(table, ImmutableList.of(pk), result, context);
							}
						}
					}
				}
			}
		} finally {
			Daos.closeSilently(dihResultSet);
		}

		if (CollectionUtils.isNotEmpty(dihIdsToRemove)) {
			SynchroTableOperation operation = new SynchroTableOperation(TABLE_DELETED_ITEM_HISTORY, context);
			operation.addChildrenToDeleteFromOneColumn(TABLE_DELETED_ITEM_HISTORY, COLUMN_REMOTE_ID, dihIdsToRemove);
			result.add(operation);
		}

		return result;
	}

	/**
	 * <p>
	 * fillPersonSessionId.
	 * </p>
	 * 
	 * @param dataSynchroContext
	 *            a {@link fr.ifremer.adagio.synchro.service.data.DataSynchroContext} object.
	 * @param result
	 *            a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 */
	protected void fillPersonSessionId(DataSynchroContext dataSynchroContext, SynchroResult result) {

		try {
			int personSessionId = personSessionSynchroJdbcDao.getPersonSessionIdByPerson(
					dataSynchroContext.getSource().getConnectionProperties(),
					dataSynchroContext.getPersonId());
			if (log.isDebugEnabled()) {
				log.debug(String.format("Person session id: %s", personSessionId));
			}

			dataSynchroContext.setPersonSessionId(personSessionId);
		} catch (AdagioTechnicalException e) {
			result.setError(e);
		}
	}

	/** {@inheritDoc} */
	@Override
	protected void resolveTableRow(SynchroTableMetadata table,
			RejectedRow reject,
			RejectedRow.ResolveStrategy rejectStrategy,
			DaoFactory daoFactory,
			SynchroTableOperation operation,
			SynchroContext context,
			SynchroResult result) {

		// Execute default implementation
		super.resolveTableRow(
				table,
				reject,
				rejectStrategy,
				daoFactory,
				operation,
				context,
				result
				);

		boolean hasSynchronizationStatus = table.getColumnNames().contains(COLUMN_SYNCHRONIZATION_STATUS);
		DataSynchroContext dataSynchroContext = (DataSynchroContext) context;

		// Add a special case for Exportation :
		// when keep local data : make sure all entity will be re-export,
		// by changing synchronizaton_status to READ_TO_SYNC
		if (hasSynchronizationStatus
				&& dataSynchroContext.getDirection().isExport()
				&& reject.cause == RejectedRow.Cause.BAD_UPDATE_DATE
				&& rejectStrategy == RejectedRow.ResolveStrategy.KEEP_LOCAL) {

			operation.addMissingColumnUpdate(
					COLUMN_SYNCHRONIZATION_STATUS,
					reject.targetPkStr,
					SynchronizationStatus.READY_TO_SYNCHRONIZE.getValue());
		}
	}

	/** {@inheritDoc} */
	@Override
	protected void detachRows(
			SynchroTableDao targetDao,
			List<List<Object>> pks,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations)
			throws SQLException {

		// Call default implementation
		super.detachRows(targetDao, pks, context, result, pendingOperations);

		// Then override to reset remote_id:

		SynchroTableMetadata table = targetDao.getTable();
		boolean hasSynchronizationStatus = table.getColumnNames().contains(COLUMN_SYNCHRONIZATION_STATUS);
		boolean hasRemoteId = table.getColumnNames().contains(COLUMN_REMOTE_ID);
		if (!hasSynchronizationStatus && !hasRemoteId) {
			return;
		}

		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);
		String readyToSynchronize = SynchronizationStatus.READY_TO_SYNCHRONIZE.getValue();
		int countR = 0;

		// For each pk to detach: reset remote_id
		if (hasRemoteId) {
			for (List<Object> pk : pks) {
				targetDao.executeUpdateColumn(COLUMN_REMOTE_ID, pk, null);
				countR++;
				reportProgress(result, targetDao, countR, tablePrefix);
			}
		}

		// For each pk to detach: set synchronization_status to READY_TO_SYNC
		// (in case the status was SYNC)
		if (hasSynchronizationStatus) {
			for (List<Object> pk : pks) {
				targetDao.executeUpdateColumn(COLUMN_SYNCHRONIZATION_STATUS, pk, readyToSynchronize);
				countR++;
				reportProgress(result, targetDao, countR, tablePrefix);
			}
		}

		if (countR == 0) {
			return;
		}

		targetDao.flush();

		result.addTableName(tableName);
		result.addUpdates(tableName, countR);

		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (detachment: %s)", tablePrefix, countR, countR));
		}

		if (targetDao.getCurrentOperation().isEnableProgress()) {
			result.getProgressionModel().increments(countR % batchSize);
		}

	}

	/**
	 * Check if a split is need, for import (see mantis #27275)
	 * 
	 * @param resultAfterPrepare
	 *            a {@link fr.ifremer.common.synchro.service.SynchroResult} object.
	 * @param context
	 *            a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @return a {@link java.util.List} object.
	 */
	protected List<Multimap<String, String>> computePkIncludesListForBatch(SynchroResult resultAfterPrepare,
			SynchroContext context) {
		int maxRootRowCount = adagioConfig.getImportDataMaxRootRowCount();
		if (MapUtils.isEmpty(resultAfterPrepare.getUpdateDateHits())
				|| maxRootRowCount <= 0 /* Skip if disable */
				|| resultAfterPrepare.getTotalRows() <= maxRootRowCount) {
			return null;
		}

		Preconditions.checkNotNull(context);
		if (log.isDebugEnabled()) {
			log.debug(String.format("Compute PKs to import, by batch of %s rows", maxRootRowCount));
		}

		SynchroDatabaseConfiguration source = context.getSource();
		Preconditions.checkNotNull(source);

		Connection sourceConnection = null;
		DaoFactory sourceDaoFactory = null;
		SynchroDatabaseMetadata dbMeta = null;
		ResultSet dataToUpdate = null;

		try {

			Set<String> tableNames = resultAfterPrepare.getUpdateDateHits().keySet();
			List<Multimap<String, String>> result = Lists.newArrayList();

			// create source Connection
			sourceConnection = createConnection(source);

			log.debug("Loading source database metadata...");
			boolean isFullMetadataEnable = source.isFullMetadataEnable();
			source.setFullMetadataEnable(true);
			dbMeta = loadDatabaseMetadata(
					sourceConnection,
					source,
					tableNames);
			source.setFullMetadataEnable(isFullMetadataEnable);

			// Create DAO factories
			sourceDaoFactory = newDaoFactory(sourceConnection, source, dbMeta);

			// For each table found during preparation phase
			Multimap<String, String> currentBatch = ArrayListMultimap.create();
			int currentBatchSize = 0;
			for (String tableName : tableNames) {

				SynchroTableDao sourceDao = sourceDaoFactory.getSourceDao(tableName);

				Map<String, Object> bindings = createSelectBindingsForTable(context, tableName);
				dataToUpdate = sourceDao.getData(bindings);

				while (dataToUpdate.next()) {

					String pkStr = SynchroTableMetadata.toPkStr(sourceDao.getPk(dataToUpdate));

					// If current batch is full: add it to result list
					if (currentBatchSize == maxRootRowCount) {
						result.add(currentBatch);
						currentBatch = ArrayListMultimap.create();
						currentBatchSize = 0;
					}

					currentBatch.put(tableName, pkStr);
					currentBatchSize++;
				}

				// Close the result set
				Daos.closeSilently(dataToUpdate);
				dataToUpdate = null;
			}

			// Add last batch to result
			if (currentBatchSize > 0) {
				result.add(currentBatch);
			}

			return result.size() > 1 ? result : null; // Do NOT return result, if only one batch (null=default import)

		} catch (Exception e) {
			if (log.isDebugEnabled()) {
				log.debug(e);
			}
			throw new SynchroTechnicalException(e);
		} finally {
			Daos.closeSilently(dataToUpdate);
			IOUtils.closeQuietly(sourceDaoFactory);
			closeSilently(dbMeta);
			closeSilently(sourceConnection);
		}
	}

	/**
	 * Remap batch progression model to the main progression model
	 * 
	 * @param mainProgressionModel
	 *            a {@link fr.ifremer.common.synchro.type.ProgressionModel} object.
	 * @param batchProgressionModel
	 *            a {@link fr.ifremer.common.synchro.type.ProgressionModel} object
	 * @param batchProgressionModelOffset
	 *            a int.
	 */
	protected void addProgressionListeners(
			final ProgressionModel mainProgressionModel,
			final ProgressionModel batchProgressionModel,
			final int batchProgressionModelOffset) {
		// Listen 'current' attribute changes
		batchProgressionModel.addPropertyChangeListener(fr.ifremer.common.synchro.type.ProgressionModel.PROPERTY_CURRENT,
				new PropertyChangeListener() {
					@Override
					public void propertyChange(PropertyChangeEvent evt) {
						Integer current = (Integer) evt.getNewValue();
						mainProgressionModel.setCurrent(batchProgressionModelOffset + current);
					}
				});

		// Listen message changes
		batchProgressionModel.addPropertyChangeListener(fr.ifremer.common.synchro.type.ProgressionModel.PROPERTY_MESSAGE,
				new PropertyChangeListener() {
					@Override
					public void propertyChange(PropertyChangeEvent evt) {
						String message = (String) evt.getNewValue();
						mainProgressionModel.setMessage(message);
					}
				});
	}

	/**
	 * THis method will do an intermediate commit on current connection.
	 * If HsqlDB, will compact the database.
	 * At the end, GC is calling
	 * 
	 * @param dbConfig
	 *            a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void commitAndCompact(SynchroDatabaseConfiguration dbConfig) throws SQLException {
		// Do an intermediate commit
		Connection connection = null;
		try {
			connection = createConnection(dbConfig);
			connection.commit();
			DaoUtils.compactDatabase(connection);
			System.gc();
		} finally {
			closeSilently(connection);
		}
	}

	/**
	 * <p>
	 * synchronizeUsingBatch.
	 * </p>
	 * 
	 * @param synchroContext
	 *            a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param pkIncludesListForBatch
	 *            a {@link java.util.List} object.
	 */
	protected void synchronizeUsingBatch(SynchroContext synchroContext,
			List<Multimap<String, String>> pkIncludesListForBatch) {

		DataSynchroContext dataSynchroContext = (DataSynchroContext) synchroContext;
		SynchroResult result = dataSynchroContext.getResult();

		boolean savedEnableDelete = dataSynchroContext.isEnableDelete();
		boolean currentBatchEnableDelete = savedEnableDelete;

		// Init the progression model (only current+total, because message will be delegate to a child progression
		// model)
		ProgressionModel progressionModel = result.getProgressionModel();
		progressionModel.setCurrent(0);
		progressionModel.setTotal(result.getTotalRows() + pkIncludesListForBatch.size());

		// Clear result (will be updated after each batch, by calling 'addAll()')
		result.clear();

		for (Multimap<String, String> currentBatchPkIncludes : pkIncludesListForBatch) {

			// Update the context to process only PK included in the current batch
			SynchroResult currentBatchResult = new SynchroResult(result.getLocalUrl(), result.getRemoteUrl());
			for (String tableName : currentBatchPkIncludes.keySet()) {
				currentBatchResult.setUpdateDate(tableName, null);
				currentBatchResult.addRows(tableName, currentBatchPkIncludes.get(tableName).size());
			}
			dataSynchroContext.setResult(currentBatchResult);
			dataSynchroContext.setPkIncludes(currentBatchPkIncludes);
			dataSynchroContext.setEnableDelete(currentBatchEnableDelete);

			int batchProgessionModelOffset = progressionModel.getCurrent();
			addProgressionListeners(progressionModel,
					currentBatchResult.getProgressionModel(),
					batchProgessionModelOffset);

			// Call synchronize
			super.synchronize(synchroContext);

			// Add the current result to the final result
			result.addAll(currentBatchResult);

			// Update progression model
			// progressionModel.setCurrent(batchProgessionModelOffset + currentBatchResult.getTotalRows());

			// If error, then store this error into final result, and stop iteration
			if (!currentBatchResult.isSuccess()) {
				result.setError(currentBatchResult.getError());
				break;
			}

			// Disable deletions import, for the next iteration
			// because all deletions processed during the first iteration
			currentBatchEnableDelete = false;

			// Do an intermediate commit (+compact)
			try {
				commitAndCompact(synchroContext.getTarget());
			} catch (SQLException e) {
				result.setError(e);
				break;
			}
		}

		// Update progression model
		progressionModel.setCurrent(progressionModel.getTotal());

		// Restore the context: result and configuration (pkIncludes and enableDelete)
		dataSynchroContext.setResult(result);
		dataSynchroContext.setPkIncludes(null);
		dataSynchroContext.setEnableDelete(savedEnableDelete);
	}
}
