package fr.ifremer.adagio.synchro.service.data;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.Multimap;
import fr.ifremer.adagio.synchro.service.AbstractSynchroContext;
import fr.ifremer.common.synchro.service.SynchroContext;
import fr.ifremer.adagio.synchro.service.SynchroDirection;

import java.util.Date;
import java.util.Set;

/**
 * <p>
 * DataSynchroContext class.
 * </p>
 * 
 */
public class DataSynchroContext extends AbstractSynchroContext<DataSynchroDatabaseConfiguration> {

	protected Date dataStartDate;

	protected Date dataEndDate;

	protected Multimap<String, String> pkIncludes;

	/**
	 * Could an edited row (syanchronizationStatus != SYNC) be override silently ? <br>
	 * Default is <code>false</code>. Set to <code>true</code> to avoid rejected rows, when importing by Pk (with a
	 * pkIncludes) (See class ImportEditedRowInterceptor for usage)
	 */
	protected boolean forceEditedRowOverride = false;

	/**
	 * List of program codes
	 */
	protected Set<String> programCodes;

	/**
	 * User profil Id for referential administrator
	 */
	protected int userProfilIdReferentailAdministrator = 1; // default value (= for Ifremer SIH database)

	/**
	 * Enable attachment files operation (only for import)
	 */
	protected boolean enableAttachmentFiles;

	/**
	 * <p>
	 * Constructor for DataSynchroContext.
	 * </p>
	 */
	public DataSynchroContext() {
		super();
	}

	/**
	 * <p>
	 * Constructor for DataSynchroContext.
	 * </p>
	 * 
	 * @param otherBean
	 *            a {@link fr.ifremer.adagio.synchro.service.data.DataSynchroContext} object.
	 */
	public DataSynchroContext(DataSynchroContext otherBean) {
		copy(otherBean);
	}

	/**
	 * <p>
	 * Constructor for DataSynchroContext.
	 * </p>
	 * 
	 * @param direction
	 *            a {@link fr.ifremer.adagio.synchro.service.SynchroDirection} object.
	 * @param personId
	 *            a {@link java.lang.Integer} object.
	 */
	public DataSynchroContext(SynchroDirection direction, Integer personId) {
		super();
		setDirection(direction);
		setPersonId(personId);
	}

	/**
	 * <p>
	 * Constructor for DataSynchroContext.
	 * </p>
	 * 
	 * @param delegate
	 *            a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param direction
	 *            a {@link fr.ifremer.adagio.synchro.service.SynchroDirection} object.
	 * @param personId
	 *            a {@link java.lang.Integer} object.
	 */
	public DataSynchroContext(SynchroContext delegate, SynchroDirection direction, Integer personId) {
		super(delegate, direction, personId);
	}

	/**
	 * <p>
	 * Constructor for DataSynchroContext.
	 * </p>
	 * 
	 * @param delegate
	 *            a {@link fr.ifremer.common.synchro.service.SynchroContext} object.
	 * @param direction
	 *            a {@link fr.ifremer.adagio.synchro.service.SynchroDirection} object.
	 * @param personId
	 *            a {@link java.lang.Integer} object.
	 * @param personSessionId
	 *            a {@link java.lang.Integer} object.
	 */
	public DataSynchroContext(SynchroContext delegate, SynchroDirection direction, Integer personId, Integer personSessionId) {
		super(delegate, direction, personId);
		setPersonSessionId(personSessionId);
	}

	/**
	 * <p>
	 * Getter for the field <code>dataStartDate</code>.
	 * </p>
	 * 
	 * @return a {@link java.util.Date} object.
	 */
	public Date getDataStartDate() {
		return dataStartDate;
	}

	/**
	 * <p>
	 * Setter for the field <code>dataStartDate</code>.
	 * </p>
	 * 
	 * @param dataStartDate
	 *            a {@link java.util.Date} object.
	 */
	public void setDataStartDate(Date dataStartDate) {
		this.dataStartDate = dataStartDate;
	}

	/**
	 * <p>
	 * Getter for the field <code>dataEndDate</code>.
	 * </p>
	 * 
	 * @return a {@link java.util.Date} object.
	 */
	public Date getDataEndDate() {
		return dataEndDate;
	}

	/**
	 * <p>
	 * Setter for the field <code>dataEndDate</code>.
	 * </p>
	 * 
	 * @param dataEndDate
	 *            a {@link java.util.Date} object.
	 */
	public void setDataEndDate(Date dataEndDate) {
		this.dataEndDate = dataEndDate;
	}

	/**
	 * <p>
	 * Setter for the field <code>pkIncludes</code>.
	 * </p>
	 * 
	 * @param pkIncludes
	 *            a {@link com.google.common.collect.Multimap} object.
	 */
	public void setPkIncludes(Multimap<String, String> pkIncludes) {
		this.pkIncludes = pkIncludes;
	}

	/**
	 * <p>
	 * Getter for the field <code>pkIncludes</code>.
	 * </p>
	 * 
	 * @return a {@link com.google.common.collect.Multimap} object.
	 */
	public Multimap<String, String> getPkIncludes() {
		return this.pkIncludes;
	}

	/**
	 * <p>
	 * Getter for the field <code>programCodes</code>.
	 * </p>
	 * 
	 * @return a {@link java.util.Set} object.
	 */
	public Set<String> getProgramCodes() {
		return programCodes;
	}

	/** {@inheritDoc} */
	public void setProgramCodes(Set<String> programCodes) {
		this.programCodes = programCodes;
	}

	/**
	 * <p>
	 * Getter for the field <code>userProfilIdReferentailAdministrator</code>.
	 * </p>
	 * 
	 * @return an integer
	 */
	public int getUserProfilIdReferentailAdministrator() {
		return userProfilIdReferentailAdministrator;
	}

	/**
	 * <p>
	 * Setter for the field <code>userProfilIdReferentailAdministrator</code>.
	 * </p>
	 * 
	 * @param userProfilIdReferentailAdministrator
	 *            the value to set
	 */
	public void setUserProfilIdReferentailAdministrator(int userProfilIdReferentailAdministrator) {
		this.userProfilIdReferentailAdministrator = userProfilIdReferentailAdministrator;
	}

	/**
	 * <p>
	 * isForceEditedRowOverride.
	 * </p>
	 * 
	 * @return a boolean.
	 */
	public boolean isForceEditedRowOverride() {
		return forceEditedRowOverride;
	}

	/**
	 * <p>
	 * Setter for the field <code>forceEditedRowOverride</code>.
	 * </p>
	 * 
	 * @param forceEditedRowOverride
	 *            a boolean.
	 */
	public void setForceEditedRowOverride(boolean forceEditedRowOverride) {
		this.forceEditedRowOverride = forceEditedRowOverride;
	}

	/**
	 * The attachment files should be enable on import if wanted, and always enable on export
	 * 
	 * @return a boolean
	 */
	public boolean isEnableAttachmentFiles() {
		return (enableAttachmentFiles && getDirection().isImport()) || getDirection().isExport();
	}

	public void setEnableAttachmentFiles(boolean enableAttachmentFiles) {
		this.enableAttachmentFiles = enableAttachmentFiles;
	}

	/** {@inheritDoc} */
	@Override
	public void copy(SynchroContext<? extends DataSynchroDatabaseConfiguration> otherBean) {
		super.copy(otherBean);

		if (otherBean != null) {
			// Change database configuration class
			if (otherBean.getTarget() != null) {
				setTarget(new DataSynchroDatabaseConfiguration(this, otherBean.getTarget()));
			}
			if (otherBean.getSource() != null) {
				setSource(new DataSynchroDatabaseConfiguration(this, otherBean.getSource()));
			}

			if (otherBean instanceof DataSynchroContext) {
				this.dataStartDate = ((DataSynchroContext) otherBean).dataStartDate;
				this.dataEndDate = ((DataSynchroContext) otherBean).dataEndDate;
				this.pkIncludes = ((DataSynchroContext) otherBean).pkIncludes;
				this.programCodes = ((DataSynchroContext) otherBean).programCodes;
				this.forceEditedRowOverride = ((DataSynchroContext) otherBean).forceEditedRowOverride;
			}
		}
	}

	/** {@inheritDoc} */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder(super.toString());
		sb.append("\n  override edited row silently: ").append(forceEditedRowOverride);

		if (getDataStartDate() != null) {
			sb.append("\n  data start date: ").append(getDataStartDate());
			sb.append("\n         end date: ").append(getDataEndDate());
		}
		if (getProgramCodes() != null) {
			sb.append("\n     programs: ").append(getProgramCodes());
		}
		if (getPkIncludes() != null) {
			sb.append("\n  PK includes: ").append(getPkIncludes().toString());
		}

		return sb.toString();
	}

}
