package fr.ifremer.adagio.synchro.intercept.referential;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.ImmutableSet;
import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryOperator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Set;

/**
 * Filter to keep on rows on the default SIH program, and avoid to get all rows
 * 
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.7.0
 */
public class VesselTablesQueriesInterceptor extends AbstractReferentialInterceptor {

	private static final Log log = LogFactory.getLog(VesselTablesQueriesInterceptor.class);

	private static final Set<String> VESSEL_TABLES = ImmutableSet.of("VESSEL", "VESSEL_FEATURES", "VESSEL_REGISTRATION_PERIOD",
			"VESSEL_OWNER_PERIOD", "VESSEL_OWNER");

	/** Constant <code>COLUMN_PROGRAM_FK="program_fk"</code> */
	public static final String COLUMN_PROGRAM_FK = "program_fk";
	/** Constant <code>COLUMN_VESSEL_FK="vessel_fk"</code> */
	public static final String COLUMN_VESSEL_FK = "vessel_fk";

	/**
	 * Apply this interceptor only for import from server (see Mantis #32814)
	 */
	public VesselTablesQueriesInterceptor() {
		super(VESSEL_TABLES, SynchroDirection.IMPORT_SERVER2TEMP, SynchroDirection.IMPORT_NO_TEMP);
	}

	/**
	 * <p>
	 * handleCreateQuery.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.CreateQueryEvent} object.
	 */
	@Subscribe
	public void handleCreateQuery(CreateQueryEvent e) {
		switch (e.queryName) {
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
		case selectMaxUpdateDate:
			// Add restriction on vessels
			e.sql = addVesselRestriction(e.source, e.sql);
			break;

		default:
			break;
		}
	}

	/**
	 * <p>
	 * addVesselRestriction.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param sql
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String addVesselRestriction(SynchroTableMetadata table, String sql) {

		boolean hasVesselColumn = hasColumns(table, COLUMN_VESSEL_FK);
		boolean hasProgramColumn = hasColumns(table, COLUMN_PROGRAM_FK);

		SynchroQueryBuilder query = SynchroQueryBuilder.newBuilder(sql);

		String programCode = AdagioConfiguration.getInstance().getVesselRegistryProgramCode();

		// from
		if (hasVesselColumn) {
			// Do not use a INNER JOIN, because of VESSEL_FEATURES that could be imported
			// into Temp DB without rows in VESSEL (see mantis #25100)
			query.addJoin(" LEFT OUTER JOIN VESSEL v ON v.code=t." + COLUMN_VESSEL_FK);
			query.addWhere(SynchroQueryOperator.AND,
					String.format("(v.code IS NULL OR v.%s='%s')",
							COLUMN_PROGRAM_FK,
							programCode
							));
		}

		// where
		if (!hasVesselColumn && hasProgramColumn) {

			query.addWhere(SynchroQueryOperator.AND,
					String.format(" t.%s='%s'",
							COLUMN_PROGRAM_FK,
							programCode
							));
		}

		String newSql = query.build();
		if (log.isTraceEnabled()) {
			log.trace("Replace default SQL with: " + newSql);
		}

		return newSql;
	}
}
