package fr.ifremer.adagio.synchro.intercept.referential;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import fr.ifremer.adagio.synchro.meta.DatabaseColumns;
import fr.ifremer.adagio.synchro.meta.referential.ReferentialSynchroTables;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.common.synchro.dao.SynchroBaseDao;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

/**
 * This interceptor will detect deletion on VESSEL, and delete VESSEL_FEATURES on VESSEL_FK.
 * (see mantis #24089)
 * 
 * It also deletes all references to a deleted vessel in PERSON_SESSION_VESSEL (Mantis #36534)
 * 
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.8.7
 */
public class VesselInterceptor extends AbstractReferentialInterceptor {

	private PreparedStatement deleteVesselFeaturesStatement = null;
	private PreparedStatement deletePersonSessionVesselStatement = null;

	/**
	 * <p>
	 * Constructor for VesselInterceptor.
	 * </p>
	 */
	public VesselInterceptor() {
		super(SynchroDirection.IMPORT_TEMP2LOCAL, SynchroDirection.IMPORT_NO_TEMP);
		setEnableOnWrite(true);
	}

	/** {@inheritDoc} */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return ReferentialSynchroTables.VESSEL.name().equalsIgnoreCase(table.getName());
	}

	/** {@inheritDoc} */
	protected void doOnDelete(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {

		// Get the vessel code
		Object pkObj = pk.get(0);

		// delete VESSEL_FEATURES
		deleteVesselFeatures(targetDao, pkObj);

		// delete PERSON_SESSION_VESSEL (Mantis #36534)
		deletePersonSessionVessel(targetDao, pkObj);

	}

	/* -- Internal methods -- */

	/**
	 * <p>
	 * deleteVesselFeatures.
	 * </p>
	 * 
	 * @param targetDao
	 *            a {@link fr.ifremer.common.synchro.dao.SynchroBaseDao} object.
	 * @param vesselPk
	 *            a {@link java.lang.Object} object.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void deleteVesselFeatures(SynchroBaseDao targetDao, Object vesselPk) throws SQLException {
		if (deleteVesselFeaturesStatement == null || deleteVesselFeaturesStatement.isClosed()) {

			deleteVesselFeaturesStatement = targetDao.getPreparedStatement(getDeleteVesselFeaturesQuery());
		}

		deleteVesselFeaturesStatement.setObject(1, vesselPk);
		deleteVesselFeaturesStatement.executeUpdate();
	}

	protected void deletePersonSessionVessel(SynchroBaseDao targetDao, Object vesselPk) throws SQLException {
		if (deletePersonSessionVesselStatement == null || deletePersonSessionVesselStatement.isClosed()) {

			deletePersonSessionVesselStatement = targetDao.getPreparedStatement(getDeletePersonSessionVesselQuery());
		}

		deletePersonSessionVesselStatement.setObject(1, vesselPk);
		deletePersonSessionVesselStatement.executeUpdate();
	}

	/**
	 * <p>
	 * getDeleteVesselFeaturesQuery.
	 * </p>
	 * 
	 * @return a {@link java.lang.String} object.
	 */
	protected String getDeleteVesselFeaturesQuery() {
		return String.format("DELETE FROM %s WHERE %s = ?",
				ReferentialSynchroTables.VESSEL_FEATURES.name(),
				DatabaseColumns.VESSEL_FK.name()
				);
	}

	protected String getDeletePersonSessionVesselQuery() {
		return String.format("DELETE FROM %s WHERE %s = ?",
				ReferentialSynchroTables.PERSON_SESSION_VESSEL.name(),
				DatabaseColumns.VESSEL_FK.name()
				);
	}
}
