package fr.ifremer.adagio.synchro.intercept.referential;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryOperator;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.tool.hbm2ddl.TableMetadata;

/**
 * <p>
 * TranscribingItemInterceptor class.
 * </p>
 */
public class TranscribingItemInterceptor extends AbstractReferentialInterceptor {

	/**
	 * Constant <code>COLUMN_TRANSCRIBING_ITEM_TYPE_FK="transcribing_item_type_fk"</code>
	 */
	public static final String COLUMN_TRANSCRIBING_ITEM_TYPE_FK = "transcribing_item_type_fk";

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return "TRANSCRIBING_ITEM".equalsIgnoreCase(table.getName());
	}

	/**
	 * <p>
	 * handleCreateQuery.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.CreateQueryEvent} object.
	 */
	@Subscribe
	public void handleCreateQuery(CreateQueryEvent e) {

		String transcribingTypeIds = AdagioConfiguration.getInstance().getImportTranscribingItemTypeIds();
		if (StringUtils.isBlank(transcribingTypeIds)) {
			transcribingTypeIds = "-1";
		}

		switch (e.queryName) {
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
			// Add restriction transcribing type
			e.sql = SynchroQueryBuilder.newBuilder(e.sql)
					.addWhere(SynchroQueryOperator.AND, String.format("%s in (%s)",
							COLUMN_TRANSCRIBING_ITEM_TYPE_FK,
							transcribingTypeIds))
					.build();
			break;

		case selectMaxUpdateDate:
			// Add specific query to determine the max update date.
			// It return the min update date against the max update date of each type, or 1970-01-01 if an item has not
			// been found
			e.sql = "SELECT MIN(COALESCE(UPDATE_DT, TO_TIMESTAMP('1970-01-01', 'YYYY-MM-DD'))) FROM (\n" +
					"\tSELECT TIT.ID, MAX(TI.UPDATE_DATE) AS UPDATE_DT FROM TRANSCRIBING_ITEM_TYPE TIT \n" +
					"\tLEFT OUTER JOIN TRANSCRIBING_ITEM TI ON TI.TRANSCRIBING_ITEM_TYPE_FK=TIT.ID\n" +
					"\tWHERE TIT.ID IN (" + transcribingTypeIds + ")\n" +
					"\tGROUP BY TIT.ID\n" +
					")";
			break;

		default:
			break;
		}
	}
}
