package fr.ifremer.adagio.synchro.intercept.referential;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.eventbus.Subscribe;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.common.synchro.meta.SynchroJoinMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.meta.event.LoadJoinEvent;
import fr.ifremer.common.synchro.meta.event.LoadTableEvent;
import fr.ifremer.adagio.synchro.meta.referential.ReferentialSynchroTables;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.sql.Timestamp;
import java.util.Collection;

/**
 * Manage only data table with columns 'id' AND 'remote_id' :
 * <ul>
 * <li>Set as root only if the table as an object_type (OBJECT_TYPE is need for join with PERSON_SESION_VESSEL)</li>
 * <li>when exporting to server, override 'update_date' column with a systimestamp</li>
 * </ul>
 * 
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.6.3
 */
public class ReferentialTableInterceptor extends AbstractReferentialInterceptor {

	private static final Log log = LogFactory.getLog(ReferentialTableInterceptor.class);

	/** Constant <code>childJoinIncludes</code> */
	protected static Multimap<String, String> childJoinIncludes;
	/** Constant <code>childJoinExcludes</code> */
	protected static Multimap<String, String> childJoinExcludes;
	static {
		childJoinIncludes = initChildJoinIncludes();
		childJoinExcludes = initChildJoinExcludes();
	}

	private Timestamp systimestamp = null;

	/**
	 * <p>
	 * Constructor for ReferentialTableInterceptor.
	 * </p>
	 */
	public ReferentialTableInterceptor() {
		super(ReferentialSynchroTables.getImportTablesIncludes());
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {
		ReferentialTableInterceptor result = (ReferentialTableInterceptor) super.clone();
		result.systimestamp = this.systimestamp;
		return result;
	}

	/**
	 * <p>
	 * handleTableLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadTableEvent} object.
	 */
	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {

		SynchroTableMetadata table = e.table;
		String tableName = table.getName();

		// Set table as root if synchronization status column exists
		boolean hasUpdateDate = hasColumns(table, getConfig().getColumnUpdateDate());

		boolean ignoreUpdateDateColumn = "PERSON_SESSION_VESSEL".equals(tableName) || "PERSON_SESSION_ITEM".equals(tableName);

		// Set isRoot :
		// - if not already done - e.g. DeletedItemHistoryInterceptor already set as true,
		// so do not override this to false !
		// - is has an 'update_date' column
		// - Not a PERSON_SESSION_xxx tables (imported as child of PERSON_SESSION
		boolean isRoot = hasUpdateDate && !ignoreUpdateDateColumn;
		if (!table.isRoot() && isRoot) {
			table.setRoot(true);
			return;
		}

		// Set STATUS as root even if no update_date
		if (ReferentialSynchroTables.STATUS.name().equalsIgnoreCase(table.getName())) {
			table.setRoot(true);
		}
	}

	/**
	 * <p>
	 * handleJoinLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadJoinEvent} object.
	 */
	@Subscribe
	public void handleJoinLoad(LoadJoinEvent e) {
		SynchroJoinMetadata join = e.join;

		String fkTableName = join.getFkTable().getName().toUpperCase();
		String fkColumnName = join.getFkColumn().getName().toUpperCase();

		// Special case for PERSON_SESSION children
		boolean ignoreFkTableUpdateDateColumn = "PERSON_SESSION_VESSEL".equals(fkTableName)
				|| "PERSON_SESSION_ITEM".equals(fkTableName);

		/*
		 * if (join.isChild()
		 * && "PERSON_SESSION".equals(join.getPkTable().getName().toUpperCase())
		 * && hasColumns(join.getFkTable(), "PERSON_SESSION_FK")) {
		 * // Keep as valid child join
		 * return;
		 * }
		 */

		// Disable all join to a root table
		if (join.isChild() && hasColumns(join.getFkTable(), getConfig().getColumnUpdateDate()) && !ignoreFkTableUpdateDateColumn) {
			if (log.isDebugEnabled()) {
				log.debug(String.format("Disabling %s: child table is a root table", join.toString()));
			}
			join.setIsValid(false);
			return;
		}

		// Disable link to itself
		if (join.getFkTable() == join.getPkTable()) {
			if (log.isDebugEnabled()) {
				log.debug(String.format("Disabling %s: recursive link", join.toString()));
			}
			join.setIsValid(false);
			return;
		}

		// Keep a child join if its includes
		if (join.isChild()) {
			// Is join explicitly include ?
			Collection<String> columnIncludes = childJoinIncludes.get(fkTableName);
			// Tables that are NOT listed will have all children joins exclude.
			boolean isColumnInclude = CollectionUtils.isNotEmpty(columnIncludes) && columnIncludes.contains(fkColumnName);

			// Is join explicitly exclude ?
			Collection<String> columnExcludes = childJoinExcludes.get(fkTableName);
			boolean isColumnExclude = CollectionUtils.isNotEmpty(columnExcludes) && columnExcludes.contains(fkColumnName);

			if (!isColumnInclude || isColumnExclude) {
				if (log.isDebugEnabled()) {
					log.debug(String.format("Disabling %s: include=%s, exclude=%s",
							join.toString(),
							isColumnInclude,
							isColumnExclude));
				}
				join.setIsValid(false);
			}
		}
	}

	/* -- Internal methods -- */

	/**
	 * Only this child link will be kept (other are invalidate).<br>
	 * Tables that are NOT listed here will have all children joins <u>exclude</u>.
	 * 
	 * @return a {@link com.google.common.collect.Multimap} object.
	 */
	protected static Multimap<String, String> initChildJoinIncludes() {
		Multimap<String, String> result = ArrayListMultimap.create();

		//
		result.put("APPLIED_STRATEGY", "STRATEGY_FK");
		result.put("APPLIED_PERIOD", "APPLIED_STRATEGY_FK");
		result.put("DEPARTMENT_HIERARCHY", "PARENT_DEPARTMENT_FK");
		result.put("FRACTION2MATRIX", "FRACTION_FK");
		result.put("GROUPING_ITEM", "GROUPING_FK");
		result.put("GROUPING_ITEM_HIERARCHY", "GROUPING_FK");
		result.put("GROUPING_VESSEL_HIERARCHY", "GROUPING_FK");
		result.put("LOCATION_ASSOCIATION", "PARENT_LOCATION_FK");
		result.put("GEAR_ASSOCIATION", "FROM_GEAR_FK");
		result.put("LOCATION_HIERARCHY", "PARENT_LOCATION_FK");
		result.put("LOCATION_HIERARCHY_EXCEPTION", "PARENT_LOCATION_FK");
		result.put("LOCATION_HIERARCHY_OVERRIDE", "PARENT_LOCATION_FK");
		result.put("QUALITATIVE_VALUE", "PARAMETER_FK");
		result.put("PERSON2USER_PROFIL", "PERSON_FK");
		result.put("PROGRAM2DEPARTMENT", "PROGRAM_FK");
		result.put("PROGRAM2LOCATION", "PROGRAM_FK");
		result.put("PROGRAM2LOCATION_CLASSIF", "PROGRAM_FK");
		result.put("PROGRAM2PERSON", "PROGRAM_FK");
		result.put("PROGRAM2PERSON_EXCEPTION", "PROGRAM2PERSON_FK");
		result.put("PMFM_STRATEGY", "STRATEGY_FK");
		result.put("PMFM_APPLIED_STRATEGY", "APPLIED_STRATEGY_FK");
		result.put("PMFM2QUALITATIVE_VALUE", "PMFM_FK");
		result.put("REFERENCE_DOCUMENT2AUTHOR", "REFERENCE_DOCUMENT_FK");
		result.put("REFERENCE_TAXON_STRATEGY", "STRATEGY_FK");
		result.put("STRATEGY2GEAR", "STRATEGY_FK");
		result.put("STRATEGY2MANAGER_PERSON", "STRATEGY_FK");
		result.put("SPATIAL_ITEM2LOCATION", "SPATIAL_ITEM_FK");
		result.put("SPATIAL_ITEM2LOCATION", "LOCATION_FK"); // mantis #23383 (add a link, to enable delete cascade from
															// LOCATION)
		result.put("TAXON_GROUP2TAXON_HIERARCHY", "TAXON_GROUP_FK");
		result.put("TAXON_GROUP_HIERARCHY", "CHILD_TAXON_GROUP_FK");
		result.put("VESSEL_REGISTRATION_PERIOD", "VESSEL_FK");
		result.put("VESSEL_OWNER_PERIOD", "VESSEL_FK");
		result.put("TAXON_INFORMATION_HISTORY", "DOCUMENT_REFERENCE_FK");
		// mantis #23381 (note 68762) - ignoring UPDATE_DATE and use TAXON_GROUP as parent
		result.put("TAXON_GROUP_HISTORICAL_RECORD", "TAXON_GROUP_FK");
		result.put("TAXON_GROUP_INFORMATION", "TAXON_GROUP_FK");
		result.put("GEAR_CLASSIFICATION_ASSOCIATIO", "FROM_GEAR_CLASSIFICATION_FK");

		result.put("PERSON_SESSION_VESSEL", "PERSON_SESSION_FK");
		result.put("PERSON_SESSION_ITEM", "PERSON_SESSION_FK");

		return result;
	}

	/**
	 * <p>
	 * initChildJoinExcludes.
	 * </p>
	 * 
	 * @return a {@link com.google.common.collect.Multimap} object.
	 */
	protected static Multimap<String, String> initChildJoinExcludes() {
		Multimap<String, String> result = ArrayListMultimap.create();

		// this child link will be excluded
		result.put("TAXON_GROUP", "PARENT_TAXON_GROUP_FK");
		// no need to follow recursive links (since integrity constraints are disabled for referentials)
		result.put("DEPARTMENT", "PARENT_DEPARTMENT_FK");

		return result;
	}
}
