package fr.ifremer.adagio.synchro.intercept.referential;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.dao.SynchroBaseDao;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

/**
 * This interceptor will detect deletion on REFERENCE_TAXON, and link data to a new REFERENCE_TAXON
 * according to transcribing in TRANSCRIBING_ITEM.
 * 
 * @author Nolwenn Rannou (nolwenn.rannou@ifremer.fr)
 * @since 4.0.8
 */
public class ReferenceTaxonInterceptor extends AbstractReferentialInterceptor {

	private static final Log log = LogFactory.getLog(ReferenceTaxonInterceptor.class);

	/** Constant <code>TABLE_REFERENCE_TAXON="REFERENCE_TAXON"</code> */
	public static final String TABLE_REFERENCE_TAXON = "REFERENCE_TAXON";
	/** Constant <code>TABLE_BATCH="BATCH"</code> */
	public static final String TABLE_BATCH = "BATCH";
	/** Constant <code>TABLE_SAMPLE="SAMPLE"</code> */
	public static final String TABLE_SAMPLE = "SAMPLE";
	/** Constant <code>COLUMN_REFERENCE_TAXON_FK="REFERENCE_TAXON_FK"</code> */
	public static final String COLUMN_REFERENCE_TAXON_FK = "REFERENCE_TAXON_FK";
	/** Constant <code>TRANSCRIBING_ITEM_COLUMN_OBJECT_ID="OBJECT_ID"</code> */
	public static final String TRANSCRIBING_ITEM_COLUMN_OBJECT_ID = "OBJECT_ID";
	/** Constant <code>TRANSCRIBING_ITEM_COLUMN_EXTERNAL_CODE="EXTERNAL_CODE"</code> */
	public static final String TRANSCRIBING_ITEM_COLUMN_EXTERNAL_CODE = "EXTERNAL_CODE";
	/** Constant <code>TRANSCRIBING_ITEM_TYPE_REFERENCE_HISTORY="TAXINOMIE-COMMUN.REFERENCE_HISTORY"</code> */
	public static final String TRANSCRIBING_ITEM_TYPE_REFERENCE_HISTORY = "TAXINOMIE-COMMUN.REFERENCE_HISTORY";

	private PreparedStatement selectNewReferenceTaxon = null;
	private PreparedStatement updateBatchReferenceTaxon = null;
	private PreparedStatement updateSampleReferenceTaxon = null;

	/**
	 * <p>
	 * Constructor for ReferenceTaxonInterceptor.
	 * </p>
	 */
	public ReferenceTaxonInterceptor() {
		super();
		setEnableOnWrite(true);
	}

	/** {@inheritDoc} */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return TABLE_REFERENCE_TAXON.equalsIgnoreCase(table.getName());
	}

	/** {@inheritDoc} */
	protected void doOnDelete(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
		SynchroDirection direction = getConfig().getDirection();

		if (direction == SynchroDirection.IMPORT_TEMP2LOCAL) {

			// Get the old reference taxon id
			// Warning: use parseInt because ID could a String or an Integer
			Integer oldReferenceTaxonId = Integer.parseInt(pk.get(0).toString());

			// Retrieve the new reference taxon id
			Integer newReferenceTaxonId = getNewReferenceTaxonId(targetDao.getConnection(), oldReferenceTaxonId);

			// Update data if transcribing exists
			if (newReferenceTaxonId != null) {
				if (log.isInfoEnabled()) {
					log.info(String.format("[%s] Replace taxon %s with taxon %s", TABLE_REFERENCE_TAXON, oldReferenceTaxonId,
							newReferenceTaxonId));
				}

				updateBatchReferenceTaxon(targetDao, oldReferenceTaxonId, newReferenceTaxonId);
				updateSampleReferenceTaxon(targetDao, oldReferenceTaxonId, newReferenceTaxonId);
			}
		}
	}

	/* -- Internal methods -- */

	/**
	 * <p>
	 * getNewReferenceTaxonId.
	 * </p>
	 * 
	 * @param connection
	 *            a {@link java.sql.Connection} object.
	 * @param oldReferenceTaxonId
	 *            a int.
	 * @return a {@link java.lang.Integer} object.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected Integer getNewReferenceTaxonId(Connection connection, int oldReferenceTaxonId) throws SQLException {
		if (selectNewReferenceTaxon == null) {
			selectNewReferenceTaxon = connection.prepareStatement(getSelectNewReferenceTaxonQuery());
		}

		selectNewReferenceTaxon.setInt(1, oldReferenceTaxonId);

		ResultSet resultSet = null;
		try {
			resultSet = selectNewReferenceTaxon.executeQuery();

			if (!resultSet.next()) {
				return null;
			}
			if (resultSet.getObject(1) == null) {
				return null;
			}
			return resultSet.getInt(1);
		} finally {
			Daos.closeSilently(resultSet);
		}
	}

	/**
	 * <p>
	 * updateBatchReferenceTaxon.
	 * </p>
	 * 
	 * @param targetDao
	 *            a {@link SynchroBaseDao} object.
	 * @param oldReferenceTaxonId
	 *            a {@link Object} object.
	 * @param newReferenceTaxonId
	 *            a {@link Object} object.
	 * @throws SQLException
	 *             if any.
	 */
	protected void updateBatchReferenceTaxon(SynchroBaseDao targetDao, Object oldReferenceTaxonId, Object newReferenceTaxonId) throws SQLException {
		if (updateBatchReferenceTaxon == null || updateBatchReferenceTaxon.isClosed()) {
			updateBatchReferenceTaxon = targetDao.getPreparedStatement(getUpdateBatchReferenceTaxonQuery());
		}

		updateBatchReferenceTaxon.setObject(1, newReferenceTaxonId);
		updateBatchReferenceTaxon.setObject(2, oldReferenceTaxonId);
		int batchUpdated = updateBatchReferenceTaxon.executeUpdate();

		if (log.isInfoEnabled()) {
			log.info(String.format("[%s] updates: %s", TABLE_BATCH, batchUpdated));
		}
	}

	/**
	 * <p>
	 * updateBatchReferenceTaxon.
	 * </p>
	 * 
	 * @param targetDao
	 *            a {@link SynchroBaseDao} object.
	 * @param oldReferenceTaxonId
	 *            a {@link Object} object.
	 * @param newReferenceTaxonId
	 *            a {@link Object} object.
	 * @throws SQLException
	 *             if any.
	 */
	protected void updateSampleReferenceTaxon(SynchroBaseDao targetDao, Object oldReferenceTaxonId, Object newReferenceTaxonId) throws SQLException {
		if (updateSampleReferenceTaxon == null || updateSampleReferenceTaxon.isClosed()) {
			updateSampleReferenceTaxon = targetDao.getPreparedStatement(getUpdateSampleReferenceTaxonQuery());
		}

		updateSampleReferenceTaxon.setObject(1, newReferenceTaxonId);
		updateSampleReferenceTaxon.setObject(2, oldReferenceTaxonId);
		int sampleUpdated = updateSampleReferenceTaxon.executeUpdate();

		if (log.isInfoEnabled()) {
			log.info(String.format("[%s] updates: %s", TABLE_SAMPLE, sampleUpdated));
		}
	}

	/**
	 * <p>
	 * getSelectNewReferenceTaxonQuery.
	 * </p>
	 * 
	 * @return a {@link java.lang.String} object.
	 */
	protected String getSelectNewReferenceTaxonQuery() {
		return String.format("SELECT %s " +
				"FROM transcribing_item ti " +
				"JOIN transcribing_item_type tit " +
				"ON ti.transcribing_item_type_fk = tit.id " +
				"AND tit.label = '%s' " +
				"WHERE %s = ?",
				TRANSCRIBING_ITEM_COLUMN_OBJECT_ID,
				TRANSCRIBING_ITEM_TYPE_REFERENCE_HISTORY,
				TRANSCRIBING_ITEM_COLUMN_EXTERNAL_CODE
				);
	}

	/**
	 * <p>
	 * getUpdateBatchReferenceTaxonQuery.
	 * </p>
	 * 
	 * @return a {@link String} object.
	 */
	protected String getUpdateBatchReferenceTaxonQuery() {
		return String.format("UPDATE %s SET %s = ? WHERE %s = ?",
				TABLE_BATCH,
				COLUMN_REFERENCE_TAXON_FK,
				COLUMN_REFERENCE_TAXON_FK
				);
	}

	/**
	 * <p>
	 * getUpdateSampleReferenceTaxonQuery.
	 * </p>
	 * 
	 * @return a {@link String} object.
	 */
	protected String getUpdateSampleReferenceTaxonQuery() {
		return String.format("UPDATE %s SET %s = ? WHERE %s = ?",
				TABLE_SAMPLE,
				COLUMN_REFERENCE_TAXON_FK,
				COLUMN_REFERENCE_TAXON_FK
				);
	}

}
