package fr.ifremer.adagio.synchro.intercept.referential;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.ImmutableList;
import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.synchro.meta.DatabaseColumns;
import fr.ifremer.adagio.synchro.meta.referential.ReferentialSynchroTables;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.common.synchro.meta.event.LoadTableEvent;
import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryOperator;

import java.util.List;

/**
 * <p>
 * PersonSessionItemInterceptor class.
 * </p>
 * 
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.5.0
 */
public class PersonSessionItemInterceptor extends AbstractReferentialInterceptor {

	/** Constant <code>NATURAL_ID_COLUMN_NAMES</code> */
	public static final List<String> NATURAL_ID_COLUMN_NAMES = ImmutableList.of(
			DatabaseColumns.PERSON_SESSION_FK.name(),
			DatabaseColumns.PROGRAM_FK.name(),
			DatabaseColumns.OBJECT_TYPE_FK.name(),
			DatabaseColumns.OBJECT_ID.name()
			);

	/**
	 * <p>
	 * Constructor for PersonSessionItemInterceptor.
	 * </p>
	 */
	public PersonSessionItemInterceptor() {
		// Enable only on import (not need for export)
		super(ReferentialSynchroTables.PERSON_SESSION_ITEM.name(),
				SynchroDirection.IMPORT_SERVER2TEMP,
				SynchroDirection.IMPORT_NO_TEMP,
				SynchroDirection.IMPORT_TEMP2LOCAL);
	}

	/**
	 * <p>
	 * handleCreateQuery.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.CreateQueryEvent} object.
	 */
	@Subscribe
	public void handleCreateQuery(CreateQueryEvent e) {
		SynchroDirection direction = getConfig().getDirection();
		switch (e.queryName) {
		// Select queries : remove unsed columns
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
		case selectMaxUpdateDate:
			if (direction == SynchroDirection.IMPORT_SERVER2TEMP
					|| direction == SynchroDirection.IMPORT_NO_TEMP) {
				// Limit to user current session id
				int personSessionId = checkAndGetPersonSessionId();
				e.sql = SynchroQueryBuilder.newBuilder(e.sql)
						.addWhere(SynchroQueryOperator.AND, "t.person_session_fk", String.valueOf(personSessionId))
						.build();
			}
			break;

		default:
			break;
		}
	}

	/**
	 * <p>
	 * handleTableLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadTableEvent} object.
	 */
	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {
		SynchroTableMetadata table = e.table;

		// When import to local DB
		if (getConfig().getDirection() == SynchroDirection.IMPORT_TEMP2LOCAL) {
			// Define natural id
			// This is need because ObsDeb create temporary PERSON_SESSION_ITEM (mantis #28877)
			// THat should be replaced on import to local
			table.addUniqueConstraint("NATURAL_ID_UNIQUE_C", NATURAL_ID_COLUMN_NAMES, SynchroTableMetadata.DuplicateKeyStrategy.REPLACE);
		}
	}

}
