package fr.ifremer.adagio.synchro.intercept.referential;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import fr.ifremer.common.synchro.dao.SynchroBaseDao;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

/**
 * This interceptor will detect deletion on LOCATION, and delete LOCATION_ASSOCIATION and LOCATION_HIERARCHY
 * on CHILD_LOCATION_FK.
 * 
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.8.7
 */
public class LocationInterceptor extends AbstractReferentialInterceptor {

	/** Constant <code>TABLE_LOCATION_ASSOCIATION="LOCATION_ASSOCIATION"</code> */
	public static final String TABLE_LOCATION_ASSOCIATION = "LOCATION_ASSOCIATION";
	/** Constant <code>TABLE_LOCATION_HIERARCHY="LOCATION_HIERARCHY"</code> */
	public static final String TABLE_LOCATION_HIERARCHY = "LOCATION_HIERARCHY";
	/** Constant <code>COLUMN_CHILD_LOCATION_FK="child_location_fk"</code> */
	public static final String COLUMN_CHILD_LOCATION_FK = "child_location_fk";

	private PreparedStatement deleteLocationHierarchty = null;
	private PreparedStatement deleteLocationAssociation = null;

	/**
	 * <p>
	 * Constructor for LocationInterceptor.
	 * </p>
	 */
	public LocationInterceptor() {
		super();
		setEnableOnWrite(true);
	}

	/** {@inheritDoc} */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return "LOCATION".equalsIgnoreCase(table.getName());
	}

	/** {@inheritDoc} */
	protected void doOnDelete(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {

		// Get the vessel code
		Object pkObj = pk.get(0);

		// delete LOCATION_HIERARCHY
		deleteLocationHierarchy(targetDao, pkObj);

		// delete LOCATION_ASSOCIATION
		deleteLocationAssociation(targetDao, pkObj);

	}

	/* -- Internal methods -- */

	/**
	 * <p>
	 * deleteLocationHierarchy.
	 * </p>
	 * 
	 * @param targetDao
	 *            a {@link fr.ifremer.common.synchro.dao.SynchroBaseDao} object.
	 * @param vesselPk
	 *            a {@link java.lang.Object} object.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void deleteLocationHierarchy(SynchroBaseDao targetDao, Object vesselPk) throws SQLException {
		if (deleteLocationHierarchty == null || deleteLocationHierarchty.isClosed()) {

			deleteLocationHierarchty = targetDao.getPreparedStatement(getDeleteLocationHierarchyQuery());
		}

		deleteLocationHierarchty.setObject(1, vesselPk);
		deleteLocationHierarchty.executeUpdate();
	}

	/**
	 * <p>
	 * deleteLocationAssociation.
	 * </p>
	 * 
	 * @param targetDao
	 *            a {@link fr.ifremer.common.synchro.dao.SynchroBaseDao} object.
	 * @param vesselPk
	 *            a {@link java.lang.Object} object.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void deleteLocationAssociation(SynchroBaseDao targetDao, Object vesselPk) throws SQLException {
		if (deleteLocationAssociation == null || deleteLocationAssociation.isClosed()) {

			deleteLocationAssociation = targetDao.getPreparedStatement(getDeleteLocationAssocationQuery());
		}

		deleteLocationAssociation.setObject(1, vesselPk);
		deleteLocationAssociation.executeUpdate();
	}

	/**
	 * <p>
	 * getDeleteLocationHierarchyQuery.
	 * </p>
	 * 
	 * @return a {@link java.lang.String} object.
	 */
	protected String getDeleteLocationHierarchyQuery() {
		return String.format("DELETE FROM %s WHERE %s = ?",
				TABLE_LOCATION_HIERARCHY,
				COLUMN_CHILD_LOCATION_FK
				);
	}

	/**
	 * <p>
	 * getDeleteLocationAssocationQuery.
	 * </p>
	 * 
	 * @return a {@link java.lang.String} object.
	 */
	protected String getDeleteLocationAssocationQuery() {
		return String.format("DELETE FROM %s WHERE %s = ?",
				TABLE_LOCATION_ASSOCIATION,
				COLUMN_CHILD_LOCATION_FK
				);
	}
}
