package fr.ifremer.adagio.synchro.intercept.referential;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Joiner;
import com.google.common.collect.Sets;
import com.google.common.eventbus.Subscribe;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.intercept.data.ObjectTypeHelper;
import fr.ifremer.adagio.synchro.meta.DatabaseColumns;
import fr.ifremer.common.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.adagio.synchro.meta.referential.ReferentialSynchroTables;
import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryOperator;
import fr.ifremer.adagio.synchro.service.referential.ReferentialSynchroDatabaseConfiguration;
import org.apache.commons.collections4.CollectionUtils;

import java.util.Set;

/**
 * Import DELETED_ITEM_HISTORY rows, limited to referential tables
 * 
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.7.0
 */
public class DeletedItemHistoryInterceptor extends AbstractReferentialInterceptor {

	private String whereClauseOnTableIncludes = null;

	/**
	 * <p>
	 * Constructor for DeletedItemHistoryInterceptor.
	 * </p>
	 */
	public DeletedItemHistoryInterceptor() {
		super(ReferentialSynchroTables.DELETED_ITEM_HISTORY.name());
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {
		DeletedItemHistoryInterceptor newBean = (DeletedItemHistoryInterceptor) super.clone();
		newBean.whereClauseOnTableIncludes = this.whereClauseOnTableIncludes;
		return newBean;
	}

	/** {@inheritDoc} */
	@Override
	protected void init(ReferentialSynchroDatabaseConfiguration config) {
		whereClauseOnTableIncludes = createWhereClauseOnTableIncludes();
	}

	/**
	 * <p>
	 * handleCreateQuery.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.CreateQueryEvent} object.
	 */
	@Subscribe
	public void handleCreateQuery(CreateQueryEvent e) {
		switch (e.queryName) {
		case count:
		case countFromUpdateDate:
		case selectMaxUpdateDate:
			e.sql = SynchroQueryBuilder.newBuilder(e.sql)
					.addWhere(SynchroQueryOperator.AND, whereClauseOnTableIncludes)
					.build();
			break;
		case select:
		case selectFromUpdateDate:
			e.sql = SynchroQueryBuilder.newBuilder(e.sql)
					.addWhere(SynchroQueryOperator.AND, whereClauseOnTableIncludes)
					// Keep original delete order (e.g. to import taxon deletion - mantis 28389)
					.addOrderByColumn("t." + DatabaseColumns.UPDATE_DATE, true/* ASC */)
					.build();
			break;

		default:
			break;
		}
	}

	/* -- Internal methods -- */

	private String addRestrictions(String sql) {
		SynchroQueryBuilder query = SynchroQueryBuilder.newBuilder(sql);

		query.addWhere(SynchroQueryOperator.AND, whereClauseOnTableIncludes);

		return query.build();
	}

	private String createWhereClauseOnTableIncludes() {
		Set<String> tableToIncludes = ReferentialSynchroTables.getImportTablesIncludes();

		// If no tables configured : should return no lines
		if (CollectionUtils.isEmpty(tableToIncludes)) {
			return "1=2";
		}

		Set<String> allObjectTypes = Sets.newHashSet();
		for (String tableName : tableToIncludes) {
			Set<String> objectTypes = ObjectTypeHelper.getObjectTypeFromTableName(tableName);
			if (CollectionUtils.isNotEmpty(objectTypes)) {
				allObjectTypes.addAll(objectTypes);
			}
			else {
				// By default, add the table name itself
				allObjectTypes.add(tableName);
			}
		}

		// Remove itself
		allObjectTypes.remove(ReferentialSynchroTables.DELETED_ITEM_HISTORY.name());

		// Remove unsused tables (deletion row detected as child of PERSON_SESSION)
		allObjectTypes.remove(ReferentialSynchroTables.PERSON_SESSION_VESSEL.name());
		allObjectTypes.remove(ReferentialSynchroTables.PERSON_SESSION_ITEM.name());

		return new StringBuilder()
				.append("OBJECT_TYPE_FK IN ('")
				.append(Joiner.on("','").join(allObjectTypes))
				.append("')")
				.toString();
	}

}
