package fr.ifremer.adagio.synchro.intercept.data.measurement;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.ImmutableList;
import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.synchro.intercept.data.AbstractDataInterceptor;
import fr.ifremer.adagio.synchro.meta.data.DataSynchroTables;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy;
import fr.ifremer.common.synchro.meta.event.LoadPkEvent;
import fr.ifremer.common.synchro.meta.event.LoadTableEvent;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.util.Set;

/**
 * <p>
 * ProduceQuantificationMeasurementInterceptor class.
 * </p>
 * 
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.7.0
 */
public class ProduceQuantificationMeasurementInterceptor extends AbstractDataInterceptor {

	/** Constant <code>SEQUENCE_NAME="quantification_measurement_seq"</code> */
	public static final String SEQUENCE_NAME = "quantification_measurement_seq";
	/** Constant <code>COLUMN_PRODUCE_FK="produce_fk"</code> */
	public static final String COLUMN_PRODUCE_FK = "produce_fk";

	/**
	 * <p>
	 * Constructor for ProduceQuantificationMeasurementInterceptor.
	 * </p>
	 */
	public ProduceQuantificationMeasurementInterceptor() {
		super(SynchroDirection.EXPORT_TEMP2SERVER,
				SynchroDirection.IMPORT_TEMP2LOCAL);
	}

	/** {@inheritDoc} */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return DataSynchroTables.QUANTIFICATION_MEASUREMENT_P.name().equalsIgnoreCase(table.getName());
	}

	/**
	 * <p>
	 * handleLoadTable.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadTableEvent} object.
	 */
	@Subscribe
	public void handleLoadTable(LoadTableEvent e) {
		SynchroTableMetadata table = e.table;

		// Hack, because sequence name are different between adagio database and Allegro
		table.setSequenceName(SEQUENCE_NAME);

		// If export (Temp DB -> server DB)
		if (getConfig().getDirection() == SynchroDirection.EXPORT_TEMP2SERVER) {
			String remoteIdColumn = getConfig().getColumnRemoteId();
			String idColumn = getConfig().getColumnId();

			// Override natural ID, to use 'ID' and 'PRODUCE_FK'
			e.table.putUniqueConstraint(remoteIdColumn, ImmutableList.of(idColumn, COLUMN_PRODUCE_FK), DuplicateKeyStrategy.REPLACE);
		}
	}

	/**
	 * <p>
	 * handlePkLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadPkEvent} object.
	 */
	@Subscribe
	public void handlePkLoad(LoadPkEvent e) {
		Set<String> pk = e.pk;

		// If export (Temp DB -> server DB)
		if (getConfig().getDirection() == SynchroDirection.EXPORT_TEMP2SERVER) {

			// Add produce_fk as primary key component (need for performence - mantis #23391)
			if (pk != null && !pk.contains(COLUMN_PRODUCE_FK)) {
				pk.add(COLUMN_PRODUCE_FK);
			}
		}
	}
}
