package fr.ifremer.adagio.synchro.intercept.data.measurement;

/*
 * #%L
 * Reef DB :: Quadrige2 Client Core
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2014 - 2015 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.synchro.dao.data.measure.DbAttachmentFiles;
import fr.ifremer.adagio.synchro.intercept.data.AbstractDataInterceptor;
import fr.ifremer.adagio.synchro.intercept.data.ObjectTypeHelper;
import fr.ifremer.adagio.synchro.meta.DatabaseColumns;
import fr.ifremer.adagio.synchro.meta.data.DataSynchroTables;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration;
import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.dao.SynchroBaseDao;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.common.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.event.LoadTableEvent;
import fr.ifremer.common.synchro.service.SynchroContext;
import fr.ifremer.common.synchro.service.SynchroResult;
import fr.ifremer.common.synchro.util.file.FileOperation;
import fr.ifremer.common.synchro.util.file.FileOperationBuilder;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.io.File;
import java.io.IOException;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * <p>
 * MeasurementFileInterceptor class.
 * </p>
 */
public class MeasurementFileInterceptor extends AbstractDataInterceptor {

	private static final Log log = LogFactory.getLog(MeasurementFileInterceptor.class);

	private static final String COLUMN_PATH = DatabaseColumns.PATH.name();
	private static final String COLUMN_OBJECT_ID = DatabaseColumns.OBJECT_ID.name();
	private static final String COLUMN_OBJECT_TYPE_FK = DatabaseColumns.OBJECT_TYPE_FK.name();

	private File sourceDirectory = null;
	private File targetDirectory = null;
	private int pathColumnIndex;
	private int objectIdColumnIndex;
	private int objectTypeFkColumnIndex;
	private int updateDateColumnIndex;
	private int idColumnIndex;
	private int remoteIdColumnIndex;

	public MeasurementFileInterceptor() {
		super();
		setEnableOnRead(true);
		setEnableOnWrite(true);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return DataSynchroTables.MEASUREMENT_FILE.name().equalsIgnoreCase(table.getName());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void doClose() throws IOException {
		super.doClose();

		sourceDirectory = null;
		targetDirectory = null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public SynchroInterceptorBase clone() {
		MeasurementFileInterceptor newBean = (MeasurementFileInterceptor) super.clone();
		newBean.pathColumnIndex = this.pathColumnIndex;
		newBean.objectIdColumnIndex = this.objectIdColumnIndex;
		newBean.objectTypeFkColumnIndex = this.objectTypeFkColumnIndex;
		newBean.updateDateColumnIndex = this.updateDateColumnIndex;
		newBean.idColumnIndex = this.idColumnIndex;
		newBean.remoteIdColumnIndex = this.remoteIdColumnIndex;
		return newBean;
	}

	/**
	 * @param e
	 *            a {@link LoadTableEvent} object.
	 */
	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {

		pathColumnIndex = e.table.getSelectColumnIndex(COLUMN_PATH);
		objectIdColumnIndex = e.table.getSelectColumnIndex(COLUMN_OBJECT_ID);
		objectTypeFkColumnIndex = e.table.getSelectColumnIndex(COLUMN_OBJECT_TYPE_FK);
		updateDateColumnIndex = e.table.getSelectColumnIndex(getConfig().getColumnUpdateDate());
		idColumnIndex = e.table.getSelectColumnIndex(getConfig().getColumnId());
		remoteIdColumnIndex = e.table.getSelectColumnIndex(getConfig().getColumnRemoteId());
	}

	@Override
	protected void doOnRead(Object[] data, SynchroTableDao sourceDao, SynchroTableDao targetDao) throws SQLException {

		SynchroDirection direction = getConfig().getDirection();

		if (direction == SynchroDirection.EXPORT_TEMP2SERVER) {
			if (data[objectIdColumnIndex] == null)
				return;

			long localObjectId = Long.parseLong(data[objectIdColumnIndex].toString());
			String objectTypeCode = String.valueOf(data[objectTypeFkColumnIndex]);

			Object remoteObjectId = getRemoteIdFromId(sourceDao, objectTypeCode, localObjectId);
			if (remoteObjectId == null) {
				// no match
				throw new SynchroTechnicalException("Unable to find remote id for object type " + objectTypeCode + " and id = " + localObjectId);
			}

			data[objectIdColumnIndex] = remoteObjectId;

		} else if (direction == SynchroDirection.IMPORT_TEMP2LOCAL) {

			if (data[objectIdColumnIndex] == null)
				return;

			long remoteObjectId = Long.parseLong(data[objectIdColumnIndex].toString());
			String objectTypeCode = String.valueOf(data[objectTypeFkColumnIndex]);

			Object localObjectId = getIdFromRemoteId(targetDao, objectTypeCode, remoteObjectId);

			if (localObjectId == null) {
				// no match
				throw new SynchroTechnicalException("Unable to find local id for object type " + objectTypeCode + " and remote id = "
						+ remoteObjectId);
			}

			data[objectIdColumnIndex] = localObjectId;

		}

	}

	@Override
	protected void doOnWrite(Object[] data, List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer,
			boolean insert) throws SQLException {

		if (buffer == null)
			return;

		SynchroDirection direction = getConfig().getDirection();
		SynchroContext<DataSynchroDatabaseConfiguration> context = (SynchroContext<DataSynchroDatabaseConfiguration>) buffer.getSynchroContext();
		SynchroResult result = context.getResult();

		// IMPORT: Server DB -> Temp DB
		if (direction == SynchroDirection.IMPORT_SERVER2TEMP) {

			// Get source update date
			Timestamp updateDate = (Timestamp) data[updateDateColumnIndex];

			// Get last synchronization date
			Timestamp lastSynchronizationDate = context.getLastSynchronizationDate();

			int updateDateCompare = Daos.compareUpdateDates(updateDate, lastSynchronizationDate);
			if (updateDateCompare > 0) {

				// Copy file only if update date is after last synchro date
				String path = String.valueOf(data[pathColumnIndex]);
				addCopyOperation(path, path, context, result, buffer, true);
			}
		}

		// EXPORT: Local DB -> Temp DB
		if (direction == SynchroDirection.EXPORT_LOCAL2TEMP) {

			// Get actual remote id
			Object remoteObjectId = data[remoteIdColumnIndex];
			if (remoteObjectId == null) {

				// Copy file to temp only if not already synchronized
				String path = String.valueOf(data[pathColumnIndex]);
				addCopyOperation(path, path, context, result, buffer, true);
			}
		}

		// IMPORT: Temp DB -> Local DB
		else if (direction == SynchroDirection.IMPORT_TEMP2LOCAL) {
			Number remoteObjectId = (Number) data[objectIdColumnIndex];
			String objectTypeCode = String.valueOf(data[objectTypeFkColumnIndex]);
			String remotePath = String.valueOf(data[pathColumnIndex]);

			// Override OBJECT_ID column
			Number localObjectId = getIdFromRemoteId(targetDao, objectTypeCode, remoteObjectId);
			data[objectIdColumnIndex] = localObjectId;

			// Override PATH column
			String localPath = DbAttachmentFiles.getPath(objectTypeCode, localObjectId, pk.get(0), FilenameUtils.getExtension(remotePath));
			data[pathColumnIndex] = localPath;

			// Copy file
			addCopyOperation(remotePath, localPath, context, result, buffer, false);
		}

		// EXPORT: Temp DB -> Server DB
		else if (direction == SynchroDirection.EXPORT_TEMP2SERVER) {
			Object localObjectId = data[objectIdColumnIndex];
			String objectTypeCode = String.valueOf(data[objectTypeFkColumnIndex]);
			String localPath = String.valueOf(data[pathColumnIndex]);

			// Override OBJECT_ID column
			Object remoteObjectId = getRemoteIdFromId(sourceDao, result, objectTypeCode, localObjectId);
			data[objectIdColumnIndex] = remoteObjectId;

			// Override PATH column
			String remotePath = DbAttachmentFiles.getPath(objectTypeCode, remoteObjectId, pk.get(0), FilenameUtils.getExtension(localPath));
			data[pathColumnIndex] = remotePath;

			// Copy file
			boolean copy = addCopyOperation(localPath, remotePath, context, result, buffer, false);

			if (!copy) {
				// If file is not copied, means that the file is already synchronized, preserve update_date on server
				buffer.addMissingColumnUpdate(getConfig().getColumnUpdateDate(), pk, data[updateDateColumnIndex]);
			}
		}
	}

	@Override
	protected void doOnDelete(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {

		if (buffer == null)
			return;

		SynchroResult result = buffer.getSynchroContext().getResult();
		SynchroContext<DataSynchroDatabaseConfiguration> context = (SynchroContext<DataSynchroDatabaseConfiguration>) buffer.getSynchroContext();

		Object id = pk.get(0);
		String query = initSelectPathFromIdQuery(getConfig());
		String path = targetDao.getUniqueTyped(query, new Object[] { id });

		// Do the file deletion
		addDeleteOperation(path, context, result, buffer);

	}

	/* -- Internal methods -- */

	private String initSelectIdFromRemoteIdQuery(DataSynchroDatabaseConfiguration config, String tableName) {
		return String.format("SELECT %s FROM %s WHERE %s=?",
				config.getColumnId(),
				tableName,
				config.getColumnRemoteId()
				);
	}

	private String initSelectRemoteIdFromIdQuery(DataSynchroDatabaseConfiguration config, String tableName) {
		return String.format("SELECT %s FROM %s WHERE %s=?",
				config.getColumnRemoteId(),
				tableName,
				config.getColumnId()
				);
	}

	private String initSelectPathFromIdQuery(DataSynchroDatabaseConfiguration config) {
		return String.format("SELECT %s FROM %s WHERE %s=?",
				COLUMN_PATH,
				DataSynchroTables.MEASUREMENT_FILE.name(),
				config.getColumnId()
				);
	}

	private File getSourceDirectory(SynchroContext<DataSynchroDatabaseConfiguration> context) {

		if (sourceDirectory == null) {
			sourceDirectory = context.getSource().getDbAttachmentDirectory() != null
					? context.getSource().getDbAttachmentDirectory()
					: AdagioConfiguration.getInstance().getDbAttachmentDirectory();
			if (sourceDirectory == null || !sourceDirectory.exists() || !sourceDirectory.isDirectory()) {
				throw new SynchroTechnicalException("Could not find source meas_file directory: " + sourceDirectory);
			}
		}
		return sourceDirectory;
	}

	private File getTargetDirectory(SynchroContext<DataSynchroDatabaseConfiguration> context) {
		if (targetDirectory == null) {
			targetDirectory = context.getTarget().getDbAttachmentDirectory() != null
					? context.getTarget().getDbAttachmentDirectory()
					: AdagioConfiguration.getInstance().getDbAttachmentDirectory();
			if (targetDirectory == null) {
				targetDirectory = new File(AdagioConfiguration.getInstance().getTempDirectory(), DbAttachmentFiles.DB_ATTACHMENT_DIRECTORY);
				log.warn(String.format(
						"[%s] Target directory for attachment file not defined (in the target DatabaseConfiguration). Using a temporary path [%s]",
						DataSynchroTables.MEASUREMENT_FILE.name(),
						targetDirectory.getPath()
						));
			}
			if (!targetDirectory.exists() || !targetDirectory.isDirectory()) {
				try {
					FileUtils.forceMkdir(targetDirectory);
				} catch (IOException e) {
					throw new SynchroTechnicalException("Could not create directory " + targetDirectory.getPath(), e);
				}
			}
		}
		return targetDirectory;
	}

	private Number getIdFromRemoteId(SynchroBaseDao dao, String objectTypeCode, Object remoteId) throws SQLException {

		String tableName = ObjectTypeHelper.getTableNameFromObjectType(objectTypeCode);
		String selectIdFromRemoteIdQuery = initSelectIdFromRemoteIdQuery(getConfig(), tableName);
		return dao.getUniqueTyped(selectIdFromRemoteIdQuery, new Object[] { remoteId });
	}

	private Object getRemoteIdFromId(SynchroTableDao sourceDao, String objectTypeCode, Object localId) throws SQLException {

		String tableName = ObjectTypeHelper.getTableNameFromObjectType(objectTypeCode);
		String query = initSelectRemoteIdFromIdQuery(getConfig(), tableName);
		return sourceDao.getUniqueTyped(query, new Object[] { localId });
	}

	private Object getRemoteIdFromId(SynchroTableDao sourceDao, SynchroResult result, String objectTypeCode, Object localId) throws SQLException {

		// The following code is for optimization only, it could be removed if it causes problem
		try {
			return getRemoteIdFromSourceMissingUpdates(result, localId, objectTypeCode);
		} catch (SynchroTechnicalException e) {
			// Continue
		}

		return getRemoteIdFromId(sourceDao, objectTypeCode, localId);
	}

	/*
	 * For optimization, try to get the remote id from the sourceMissingUpdates map
	 */
	private Object getRemoteIdFromSourceMissingUpdates(SynchroResult result, Object localId, String objectTypeCode) {

		String tableName = ObjectTypeHelper.getTableNameFromObjectType(objectTypeCode);
		Map<String, Map<String, Object>> cachedTablesForExport = result.getSourceMissingUpdates().get(tableName);
		if (cachedTablesForExport == null) {
			throw new SynchroTechnicalException(String.format("Table [%s] not found in the sourceMissingUpdates map", tableName));
		}
		Map<String, Object> cachedRemoteIdByLocalIdForExport = cachedTablesForExport.get(DatabaseColumns.REMOTE_ID.name().toLowerCase());
		if (cachedRemoteIdByLocalIdForExport == null || !cachedRemoteIdByLocalIdForExport.containsKey(String.valueOf(localId))) {
			throw new SynchroTechnicalException(String.format("No remote_id found in the sourceMissingUpdates map, for table [%s]", tableName));
		}
		return cachedRemoteIdByLocalIdForExport.get(String.valueOf(localId));
	}

	private boolean addCopyOperation(String srcPath, String destPath,
			SynchroContext<DataSynchroDatabaseConfiguration> context,
			SynchroResult result,
			SynchroOperationRepository buffer,
			boolean throwExceptionIfScrNotFound) {

		if (!getConfig().isEnableAttachmentFiles()) {
			// Skip copy operation
			return false;
		}

		// Copy file
		File src = new File(getSourceDirectory(context), srcPath);
		if (!src.exists()) {
			if (throwExceptionIfScrNotFound) {
				throw new SynchroTechnicalException("Unable to locate source file: " + src.getPath());
			}
			return false;
		}

		File dest = new File(getTargetDirectory(context), destPath);
		try {
			if (log.isDebugEnabled() && !Objects.equals(srcPath, destPath)) {
				log.debug(String.format("[%s] preparing copy [%s] -> [%s]",
						DataSynchroTables.MEASUREMENT_FILE.name(),
						srcPath,
						destPath));
			}

			FileOperation copy;
			// If target = TempDB: no lock and undo
			if (context.getTarget().isMirrorDatabase()) {
				copy = FileOperationBuilder.prepareCopy(src, dest)
						.build();
			}
			// If target DB is not TempDB: apply lock and undo
			else {
				copy = FileOperationBuilder.prepareCopy(src, dest)
						.withLock()
						.withUndo()
						.build();
			}
			buffer.addFileOperation(copy);

			// Add to result
			result.addCopiedFiles(DataSynchroTables.MEASUREMENT_FILE.name(), 1);
		} catch (IOException e) {
			throw new SynchroTechnicalException("Could not create operation to copy file into: " + dest.getPath());
		}
		return true;
	}

	private void addDeleteOperation(String srcPath,
			SynchroContext<DataSynchroDatabaseConfiguration> context,
			SynchroResult result,
			SynchroOperationRepository buffer) {

		// Copy file
		File src = new File(getTargetDirectory(context), srcPath);

		// Warn if file already deleted, then continue
		if (!src.exists()) {
			log.warn(String.format("[%s] Unable to delete the non-existent file [%s]",
					DataSynchroTables.MEASUREMENT_FILE.name(),
					srcPath));
			return;
		}

		try {
			if (log.isDebugEnabled()) {
				log.debug(String.format("[%s] Delete file [%s]",
						DataSynchroTables.MEASUREMENT_FILE.name(),
						srcPath));
			}

			FileOperation delete;

			// If target DB = TempDB: no lock and undo
			if (context.getTarget().isMirrorDatabase()) {
				delete = FileOperationBuilder.prepareDelete(src)
						.build();
			}
			// If target DB is not TempDB: apply lock and undo
			else {
				delete = FileOperationBuilder.prepareDelete(src)
						.withLock()
						.withUndo()
						.build();
			}
			buffer.addFileOperation(delete);

			// Delete parent directory, is empty
			FileOperation deleteParentDirIfEmpty = FileOperationBuilder.prepareDeleteDir(src.getParentFile())
					.onlyIfEmpty()
					.build();
			buffer.addFileOperation(deleteParentDirIfEmpty);

			// Add to result
			result.addDeletedFiles(DataSynchroTables.MEASUREMENT_FILE.name(), 1);
		} catch (IOException e) {
			throw new SynchroTechnicalException("Could not delete file: " + src.getPath());
		}
	}

}
