package fr.ifremer.adagio.synchro.intercept.data.internal;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2017 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import fr.ifremer.adagio.synchro.intercept.data.BatchInterceptor;
import fr.ifremer.adagio.synchro.intercept.data.DataTableInterceptor;
import fr.ifremer.adagio.synchro.intercept.data.ObjectTypeHelper;
import fr.ifremer.adagio.synchro.meta.DatabaseColumns;
import fr.ifremer.adagio.synchro.meta.data.DataSynchroTables;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration;
import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.common.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.common.synchro.service.SynchroResult;

import java.io.IOException;
import java.sql.SQLException;
import java.util.*;

/**
 * Manage attachment child tables (MEASUREMENT_FILE, PHOTO).
 * Created by blavenie on 13/04/17.
 */
public class TableWithAttachmentsInterceptor extends SynchroInterceptorBase {

	private final String tableName;
	private final int pkColumnIndex;
	private final Set<String> objectTypeCodes;
	private final DataSynchroDatabaseConfiguration config;

	private Map<String, Object> cachedRemoteIdByLocalIdForExport = null;
	private final String selectIdFromRemoteIdQuery;

	public TableWithAttachmentsInterceptor(DataSynchroDatabaseConfiguration config,
			String tableName,
			int pkColumnIndex) {
		super();
		Preconditions.checkNotNull(config);
		Preconditions.checkNotNull(config.getDirection());
		Preconditions.checkNotNull(tableName);
		Preconditions.checkArgument(pkColumnIndex >= 0);
		this.config = config;
		this.tableName = tableName;
		this.pkColumnIndex = pkColumnIndex;
		this.objectTypeCodes = ObjectTypeHelper.getObjectTypeFromTableName(tableName, tableName);
		this.selectIdFromRemoteIdQuery = initSelectIdFromRemoteIdQuery(config, tableName);
		setEnableOnWrite(true);
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {
		return new TableWithAttachmentsInterceptor(
				config,
				tableName,
				pkColumnIndex);
	}

	@Override
	protected void doClose() throws IOException {
		super.doClose();
		this.cachedRemoteIdByLocalIdForExport = null;
	}

	/** {@inheritDoc} */
	@Override
	protected void doOnWrite(Object[] data,
			List<Object> pk,
			SynchroTableDao sourceDao,
			SynchroTableDao targetDao,
			SynchroOperationRepository buffer,
			boolean insert)
			throws SQLException {
		if (buffer == null)
			return;

		SynchroDirection direction = config.getDirection();

		Object objectId = data[pkColumnIndex];

		// EXPORT: Temp DB -> Server DB
		if (direction == SynchroDirection.EXPORT_TEMP2SERVER) {
			// Transform remote id as local
			objectId = getLocalIdFromRemoteId(sourceDao, buffer, objectId);

			// Make sure ExportPkRemoteIdInterceptor has been added BEFORE TableWithAttachmentsInterceptor
			if (objectId == null) {
				throw new SynchroTechnicalException(String.format(
						"Unable to get the remote_id. Make sure interceptor [%s] has been set BEFORE [%s] in 'META-INF/services'",
						DataTableInterceptor.class.getSimpleName(),
						BatchInterceptor.class.getSimpleName()
						));
			}
		}

		if (objectId != null) {
			doOnWrite(data, objectId, sourceDao, targetDao, buffer, insert);
		}
	}

	@Override
	protected void doOnDelete(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
		if (buffer == null)
			return;

		Object objectId = pk.get(0);

		if (objectId != null) {
			for (String objectTypeCode : objectTypeCodes) {
				addChildToDeleteFromManyColumns(objectId, objectTypeCode, sourceDao, targetDao, buffer);
			}
		}
	}

	/* -- Internal methods -- */

	protected void doOnWrite(Object[] data,
			Object objectId,
			SynchroTableDao sourceDao,
			SynchroTableDao targetDao,
			SynchroOperationRepository buffer,
			boolean insert)
			throws SQLException {

		// Subclasses can override this method (see BatchAttachmentInterceptor)
		for (String objectTypeCode : objectTypeCodes) {
			addChildToUpdateFromManyColumns(data, objectId, objectTypeCode, sourceDao, targetDao, buffer);
		}
	}

	protected void addChildToUpdateFromManyColumns(Object[] data,
			Object objectId,
			String objectTypeCode,
			SynchroTableDao sourceDao,
			SynchroTableDao targetDao,
			SynchroOperationRepository buffer) {
		// Add link to measurement file
		buffer.addChildToUpdateFromManyColumns(DataSynchroTables.MEASUREMENT_FILE.name(),
				ImmutableSet.of(DatabaseColumns.OBJECT_ID.name(), DatabaseColumns.OBJECT_TYPE_FK.name()),
				ImmutableList.of(objectId, objectTypeCode)
				);

		// Add link to photo
		// buffer.addChildToUpdateFromManyColumns(DataSynchroTables.PHOTO.name(),
		// ImmutableSet.of(DatabaseColumns.OBJECT_ID.name(), DatabaseColumns.OBJECT_TYPE_FK.name()),
		// ImmutableList.of(objectId, objectTypeCode)
		// );
	}

	protected void addChildToDeleteFromManyColumns(Object objectId, String objectTypeCode,
			SynchroTableDao sourceDao,
			SynchroTableDao targetDao,
			SynchroOperationRepository buffer) {
		// Delete cascade to measurement file
		buffer.addChildToDeleteFromManyColumns(DataSynchroTables.MEASUREMENT_FILE.name(),
				ImmutableSet.of(DatabaseColumns.OBJECT_ID.name(), DatabaseColumns.OBJECT_TYPE_FK.name()),
				ImmutableList.of(objectId, objectTypeCode)
				);

		// Delete cascade to photo
		// buffer.addChildToDeleteFromManyColumns(DataSynchroTables.PHOTO.name(),
		// ImmutableSet.of(DatabaseColumns.OBJECT_ID.name(), DatabaseColumns.OBJECT_TYPE_FK.name()),
		// ImmutableList.of(objectId, objectTypeCode)
		// );
	}

	protected Object getLocalIdFromRemoteId(SynchroTableDao sourceDao, SynchroOperationRepository buffer, final Object remoteId) {
		try {
			return getLocalIdFromSourceMissingUpdates(buffer, remoteId);
		} catch (SynchroTechnicalException e) {
			// Continue
		}

		// Lookup in the source DB
		try {
			return sourceDao.getUniqueTyped(selectIdFromRemoteIdQuery, new Object[] { remoteId });
		} catch (SQLException e) {
			throw new SynchroTechnicalException(String.format("Invalid sourceMissingUpdates map. Missing mapping for table [%s] and column [%s]",
					tableName, DatabaseColumns.REMOTE_ID.name().toLowerCase()));
		}
	}

	protected Object getLocalIdFromSourceMissingUpdates(SynchroOperationRepository buffer, final Object remoteId) {

		// Get the map created for the table, to apply remoteId on locale DB
		if (this.cachedRemoteIdByLocalIdForExport == null) {
			SynchroResult result = buffer.getSynchroContext().getResult();

			if (result.getSourceMissingUpdates().isEmpty()
					|| !result.getSourceMissingUpdates().containsKey(tableName)
					|| !result.getSourceMissingUpdates().get(tableName).containsKey(DatabaseColumns.REMOTE_ID.name().toLowerCase())) {
				throw new SynchroTechnicalException(String.format("Invalid sourceMissingUpdates map. Missing mapping for table [%s] and column [%s]",
						tableName, DatabaseColumns.REMOTE_ID.name().toLowerCase()));
			}

			this.cachedRemoteIdByLocalIdForExport = result.getSourceMissingUpdates()
					.get(tableName)
					.get(DatabaseColumns.REMOTE_ID.name().toLowerCase());
		}

		// Search the local id, from the remote id
		for (Map.Entry<String, Object> entry : cachedRemoteIdByLocalIdForExport.entrySet()) {
			if (Objects.equals(remoteId.toString(), entry.getValue().toString())) {
				return entry.getKey();
			}
		}

		throw new SynchroTechnicalException(String.format("No local id found in the sourceMissingUpdates map, for table [%s]",
				tableName));
	}

	protected String initSelectIdFromRemoteIdQuery(DataSynchroDatabaseConfiguration config, String tableName) {
		return String.format("SELECT %s FROM %s where %s=?",
				config.getColumnId(),
				tableName,
				config.getColumnRemoteId()
				);
	}

	protected DataSynchroDatabaseConfiguration getConfig() {
		return this.config;
	}

	protected int getPkColumnIndex() {
		return this.pkColumnIndex;
	}
}
