package fr.ifremer.adagio.synchro.intercept.data.internal;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.dao.SynchroBaseDao;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.common.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration;
import org.nuiton.i18n.I18n;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

/**
 * <p>
 * ImportRemoteIdInterceptor class.
 * </p>
 * 
 */
public class ImportRemoteIdInterceptor extends SynchroInterceptorBase {

	private final String tableName;

	private final String columnName;

	private final int columnIndex;

	private final boolean isNullable;

	private final String selectIdFromRemoteIdQuery;

	private DataSynchroDatabaseConfiguration config;

	private PreparedStatement selectIdFromRemoteIdStatement = null;

	/**
	 * <p>
	 * Constructor for ImportRemoteIdInterceptor.
	 * </p>
	 * 
	 * @param config
	 *            a {@link fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration} object.
	 * @param tableName
	 *            a {@link java.lang.String} object.
	 * @param columnName
	 *            a {@link java.lang.String} object.
	 * @param columnIndex
	 *            a int.
	 * @param isNullable
	 *            a boolean.
	 */
	public ImportRemoteIdInterceptor(DataSynchroDatabaseConfiguration config, String tableName, String columnName, int columnIndex, boolean isNullable) {
		super();
		Preconditions.checkArgument(columnIndex >= 0);
		this.tableName = tableName;
		this.columnName = columnName;
		this.columnIndex = columnIndex;
		this.isNullable = isNullable;
		this.config = config;
		this.selectIdFromRemoteIdQuery = initSelectIdFromRemoteIdQuery(config, tableName);
		setEnableOnRead(true);
		setEnableOnWrite(true);
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {
		return new ImportRemoteIdInterceptor(
				config,
				tableName,
				columnName,
				columnIndex,
				isNullable);
	}

	/** {@inheritDoc} */
	@Override
	protected void doOnWrite(Object[] data,
			List<Object> pk,
			SynchroTableDao sourceDao,
			SynchroTableDao targetDao,
			SynchroOperationRepository buffer,
			boolean insert)
			throws SQLException {
		Integer remoteId = (Integer) data[columnIndex];
		if (remoteId == null) {
			return;
		}

		Integer localId = getIdFromRemoteId(targetDao, remoteId);
		if (localId == null) {
			// Mandatory column, could not continue
			if (!isNullable) {
				throw new SynchroTechnicalException(
						I18n.t("adagio.synchro.synchronizeData.noIdFromRemoteId.error",
								config.getColumnId(),
								tableName.toUpperCase(),
								config.getColumnRemoteId(),
								remoteId));
			}

			// Optional column : add to pending changes (will be processed later)
			buffer.addMissingColumnUpdate(columnName, pk, remoteId);
			data[columnIndex] = null;
		}
		else {
			data[columnIndex] = localId;
		}
	}

	/** {@inheritDoc} */
	@Override
	protected void doOnRead(Object[] data, SynchroTableDao sourceDao, SynchroTableDao targetDao) throws SQLException {
		Integer remoteId = null;
		if (data[columnIndex] instanceof String) {
			remoteId = Integer.valueOf((String) data[columnIndex]);
		}
		else {
			remoteId = (Integer) data[columnIndex];
		}
		if (remoteId == null) {
			return;
		}

		Integer localId = getIdFromRemoteId(targetDao, remoteId);
		if (localId == null) {
			// no match: could not continue
			throw new SynchroTechnicalException(
					I18n.t("adagio.synchro.synchronizeData.noIdFromRemoteId.error",
							config.getColumnId(),
							tableName.toUpperCase(),
							config.getColumnRemoteId(),
							remoteId));
		}

		data[columnIndex] = localId;
	}

	/* -- Internal methods -- */

	/**
	 * <p>
	 * getIdFromRemoteId.
	 * </p>
	 * 
	 * @param dao
	 *            a {@link fr.ifremer.common.synchro.dao.SynchroBaseDao} object.
	 * @param remoteId
	 *            a {@link java.lang.Integer} object.
	 * @return a {@link java.lang.Integer} object.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected Integer getIdFromRemoteId(SynchroBaseDao dao, Integer remoteId) throws SQLException {

		Integer result = (Integer) dao.getUniqueTyped(selectIdFromRemoteIdQuery, new Object[] { remoteId });
		return result;
	}

	/** {@inheritDoc} */
	@Override
	protected void doClose() throws IOException {
		super.doClose();

		// Close statement
		Daos.closeSilently(selectIdFromRemoteIdStatement);
		selectIdFromRemoteIdStatement = null;
	}

	/**
	 * <p>
	 * initSelectIdFromRemoteIdQuery.
	 * </p>
	 * 
	 * @param config
	 *            a {@link fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration} object.
	 * @param tableName
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String initSelectIdFromRemoteIdQuery(DataSynchroDatabaseConfiguration config, String tableName) {
		return String.format("SELECT %s FROM %s where %s=?",
				config.getColumnId(),
				tableName,
				config.getColumnRemoteId()
				);
	}

	/** {@inheritDoc} */
	@Override
	public boolean equals(Object obj) {
		if (obj.getClass().isAssignableFrom(ImportRemoteIdInterceptor.class)) {
			ImportRemoteIdInterceptor importInterceptor = (ImportRemoteIdInterceptor) obj;
			if (importInterceptor.getTableName() == null
					|| importInterceptor.getColumnName() == null
					|| this.tableName == null
					|| this.columnName == null) {
				return false;
			} else if (importInterceptor.getTableName().equalsIgnoreCase(this.tableName)
					&& importInterceptor.getColumnName().equalsIgnoreCase(this.columnName)) {
				return true;
			}
			// or just return (this.hashCode() == importInterceptor.hashCode());
		}
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public int hashCode() {
		int hash = 7;
		hash = 67 * hash + (this.tableName != null ? this.tableName.hashCode() : 0);
		hash = 67 * hash + (this.columnName != null ? this.columnName.hashCode() : 0);
		return hash;
	}

	/**
	 * <p>
	 * Getter for the field <code>tableName</code>.
	 * </p>
	 * 
	 * @return a {@link java.lang.String} object.
	 */
	public String getTableName() {
		return tableName;
	}

	/**
	 * <p>
	 * Getter for the field <code>columnName</code>.
	 * </p>
	 * 
	 * @return a {@link java.lang.String} object.
	 */
	public String getColumnName() {
		return columnName;
	}

}
