package fr.ifremer.adagio.synchro.intercept.data.internal;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import fr.ifremer.adagio.core.dao.technical.synchronization.SynchronizationStatus;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.dao.SynchroBaseDao;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.intercept.SynchroBadUpdateDateRowException;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.common.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration;
import org.nuiton.i18n.I18n;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.List;

/**
 * <p>
 * ImportEditedRowInterceptor class.
 * </p>
 * 
 */
public class ImportEditedRowInterceptor extends SynchroInterceptorBase {

	private final String tableName;

	private final int remoteIdColumnIndex;

	private final int updateDateColumnIndex;

	private final String selectLocalIdEditedRowQuery;

	private DataSynchroDatabaseConfiguration config;

	private PreparedStatement selectIdFromRemoteIdStatement = null;

	/**
	 * <p>
	 * Constructor for ImportEditedRowInterceptor.
	 * </p>
	 * 
	 * @param config
	 *            a {@link fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration} object.
	 * @param tableName
	 *            a {@link java.lang.String} object.
	 * @param remoteIdColumnIndex
	 *            a int.
	 * @param updateDateColumnIndex
	 *            a int.
	 */
	public ImportEditedRowInterceptor(DataSynchroDatabaseConfiguration config, String tableName,
			int remoteIdColumnIndex,
			int updateDateColumnIndex) {
		super();
		Preconditions.checkArgument(remoteIdColumnIndex >= 0);
		Preconditions.checkArgument(updateDateColumnIndex >= 0);

		this.tableName = tableName;
		this.remoteIdColumnIndex = remoteIdColumnIndex;
		this.updateDateColumnIndex = updateDateColumnIndex;
		this.config = config;
		this.selectLocalIdEditedRowQuery = initSelectLocalIdEditedRowQuery(config, tableName);
		setEnableOnWrite(true);
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {
		return new ImportEditedRowInterceptor(
				config,
				tableName,
				remoteIdColumnIndex,
				updateDateColumnIndex);
	}

	/** {@inheritDoc} */
	@Override
	protected void doOnWrite(Object[] data,
			List<Object> pk,
			SynchroTableDao sourceDao,
			SynchroTableDao targetDao,
			SynchroOperationRepository buffer,
			boolean insert)
			throws SQLException {
		Integer remoteId = (Integer) data[remoteIdColumnIndex];
		if (remoteId == null) {
			return;
		}

		Integer editedLocalId = getLocalIdEditedRow(targetDao, remoteId);
		boolean canOverride = (editedLocalId == null);
		if (!canOverride) {
			Timestamp validUpdateDate = (Timestamp) data[updateDateColumnIndex];

			throw new SynchroBadUpdateDateRowException(
					I18n.t("adagio.synchro.synchronizeData.editedRow.error", tableName, editedLocalId),
					tableName,
					ImmutableList.<Object> of(remoteId), // sourcePk (see mantis #29565)
					ImmutableList.<Object> of(editedLocalId), // targetPk
					validUpdateDate);
		}
	}

	/* -- Internal methods -- */

	/**
	 * <p>
	 * getLocalIdEditedRow.
	 * </p>
	 * 
	 * @param dao
	 *            a {@link fr.ifremer.common.synchro.dao.SynchroBaseDao} object.
	 * @param remoteId
	 *            a {@link java.lang.Integer} object.
	 * @return a {@link java.lang.Integer} object.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected Integer getLocalIdEditedRow(SynchroBaseDao dao, Integer remoteId) throws SQLException {

		Integer result = (Integer) dao.getUniqueTyped(selectLocalIdEditedRowQuery, new Object[] { remoteId });
		return result;
	}

	/** {@inheritDoc} */
	@Override
	protected void doClose() throws IOException {
		super.doClose();

		// Close statement
		Daos.closeSilently(selectIdFromRemoteIdStatement);
		selectIdFromRemoteIdStatement = null;
	}

	/**
	 * <p>
	 * initSelectLocalIdEditedRowQuery.
	 * </p>
	 * 
	 * @param config
	 *            a {@link fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration} object.
	 * @param tableName
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String initSelectLocalIdEditedRowQuery(DataSynchroDatabaseConfiguration config, String tableName) {
		return String.format("SELECT %s FROM %s where %s=? and %s <> '%s'",
				config.getColumnId(),
				tableName,
				config.getColumnRemoteId(),
				config.getColumnSynchronizationStatus(),
				SynchronizationStatus.SYNCHRONIZED.getValue()
				);
	}

}
