package fr.ifremer.adagio.synchro.intercept.data.internal;

/*
 * #%L
 * SIH-Adagio :: Synchro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.common.synchro.intercept.SynchroOperationRepository;

import java.sql.SQLException;
import java.util.List;

/**
 * Read/Write Interceptor use to set as NULL any negative value.
 * 
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.15.4
 */
public class ExportRemoveNegativeValueInterceptor extends SynchroInterceptorBase {

	private final String tableName;

	private final int columnIndex;

	private final DataSynchroDatabaseConfiguration config;

	/**
	 * <p>
	 * Constructor for ExportRemoveNegativeValueInterceptor.
	 * </p>
	 * 
	 * @param config
	 *            a {@link fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration} object.
	 * @param tableName
	 *            a {@link java.lang.String} object.
	 * @param columnIndex
	 *            a int.
	 */
	public ExportRemoveNegativeValueInterceptor(DataSynchroDatabaseConfiguration config, String tableName,
			int columnIndex) {
		super();
		Preconditions.checkArgument(columnIndex >= 0);
		this.tableName = tableName;
		this.config = config;
		this.columnIndex = columnIndex;
		setEnableOnWrite(true);
		setEnableOnRead(true);
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {
		return new ExportRemoveNegativeValueInterceptor(
				config,
				tableName,
				columnIndex);
	}

	/** {@inheritDoc} */
	@Override
	protected void doOnWrite(Object[] data,
			List<Object> pk,
			SynchroTableDao sourceDao,
			SynchroTableDao targetDao,
			SynchroOperationRepository operationContext,
			boolean insert) throws SQLException {
		if (data[columnIndex] == null) {
			return;
		}
		long value = Long.parseLong(data[columnIndex].toString());

		if (value < 0) {
			data[columnIndex] = null;
		}
	}

	/** {@inheritDoc} */
	@Override
	protected void doOnRead(Object[] data, SynchroTableDao sourceDao, SynchroTableDao targetDao) throws SQLException {
		if (data[columnIndex] == null) {
			return;
		}
		long value = Long.parseLong(data[columnIndex].toString());
		if (value < 0) {
			data[columnIndex] = null;
		}
	}

	/* -- Internal methods -- */

}
