package fr.ifremer.adagio.synchro.intercept.data.internal;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration;
import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.dao.SynchroBaseDao;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.common.synchro.intercept.SynchroOperationRepository;
import org.nuiton.i18n.I18n;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

/**
 * <p>
 * ExportFkRemoteIdInterceptor class.
 * </p>
 * 
 */
public class ExportFkRemoteIdInterceptor extends SynchroInterceptorBase {

	private final String tableName;

	private final String columnName;

	private final int columnIndex;

	private final boolean isNullable;

	private final String selectRemoteIdFromIdQuery;

	private final DataSynchroDatabaseConfiguration config;

	private PreparedStatement selectRemoteIdFromIdStatement = null;

	/**
	 * <p>
	 * Constructor for ExportFkRemoteIdInterceptor.
	 * </p>
	 * 
	 * @param config
	 *            a {@link fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration} object.
	 * @param tableName
	 *            a {@link java.lang.String} object.
	 * @param columnName
	 *            a {@link java.lang.String} object.
	 * @param columnIndex
	 *            a int.
	 * @param isNullable
	 *            a boolean.
	 */
	public ExportFkRemoteIdInterceptor(DataSynchroDatabaseConfiguration config, String tableName, String columnName, int columnIndex,
			boolean isNullable) {
		super();
		Preconditions.checkArgument(columnIndex >= 0);
		this.config = config;
		this.tableName = tableName;
		this.columnName = columnName;
		this.columnIndex = columnIndex;
		this.isNullable = isNullable;
		this.selectRemoteIdFromIdQuery = initSelectRemoteIdFromIdQuery(config, tableName);
		setEnableOnWrite(true);
		setEnableOnRead(true);
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {
		return new ExportFkRemoteIdInterceptor(
				config,
				tableName,
				columnName,
				columnIndex,
				isNullable);
	}

	/** {@inheritDoc} */
	@Override
	protected void doOnWrite(Object[] data,
			List<Object> pk,
			SynchroTableDao sourceDao,
			SynchroTableDao targetDao,
			SynchroOperationRepository operationContext,
			boolean insert)
			throws SQLException {

		if (data[columnIndex] == null) {
			return;
		}
		long localId = Long.parseLong(data[columnIndex].toString());

		Number remoteId = getRemoteIdFromId(sourceDao, localId);
		if (remoteId == null) {
			// Mandatory column, could not continue
			if (!isNullable) {
				throw new SynchroTechnicalException(
						I18n.t("adagio.synchro.synchronizeData.noRemoteIdFromId.error",
								config.getColumnRemoteId(),
								tableName.toUpperCase(),
								config.getColumnId(),
								localId));
			}

			// Optional column : add to pending changes (will be processed later)
			operationContext.addMissingColumnUpdate(columnName, pk, localId);
			data[columnIndex] = null;
		}
		else {
			data[columnIndex] = remoteId;
		}
	}

	/** {@inheritDoc} */
	@Override
	protected void doOnRead(Object[] data, SynchroTableDao sourceDao, SynchroTableDao targetDao) throws SQLException {
		if (data[columnIndex] == null) {
			return;
		}
		long localId = Long.parseLong(data[columnIndex].toString());

		Number remoteId = getRemoteIdFromId(sourceDao, localId);
		if (remoteId == null) {
			// no match: maybe be the referenced row will be inserted later
			data[columnIndex] = null;
			return;
		}

		data[columnIndex] = remoteId;
	}

	/* -- Internal methods -- */

	/**
	 * <p>
	 * getRemoteIdFromId.
	 * </p>
	 * 
	 * @param dao
	 *            a {@link fr.ifremer.common.synchro.dao.SynchroBaseDao} object.
	 * @param localId
	 *            a long.
	 * @return a {@link java.lang.Number} object.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected Number getRemoteIdFromId(SynchroBaseDao dao, long localId) throws SQLException {

		Number result = (Number) dao.getUniqueTyped(selectRemoteIdFromIdQuery, new Object[] { localId });

		return result;
	}

	/** {@inheritDoc} */
	@Override
	protected void doClose() throws IOException {
		super.doClose();

		// Close statement
		Daos.closeSilently(selectRemoteIdFromIdStatement);
		selectRemoteIdFromIdStatement = null;
	}

	/**
	 * <p>
	 * initSelectRemoteIdFromIdQuery.
	 * </p>
	 * 
	 * @param config
	 *            a {@link fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration} object.
	 * @param tableName
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String initSelectRemoteIdFromIdQuery(DataSynchroDatabaseConfiguration config, String tableName) {
		return String.format("SELECT %s FROM %s where %s=?",
				config.getColumnRemoteId(),
				tableName,
				config.getColumnId()
				);
	}

	/** {@inheritDoc} */
	@Override
	public boolean equals(Object obj) {
		if (obj.getClass().isAssignableFrom(ExportFkRemoteIdInterceptor.class)) {
			ExportFkRemoteIdInterceptor exportInterceptor = (ExportFkRemoteIdInterceptor) obj;
			if (exportInterceptor.getTableName() == null
					|| exportInterceptor.getColumnName() == null
					|| this.tableName == null
					|| this.columnName == null) {
				return false;
			} else if (exportInterceptor.getTableName().equalsIgnoreCase(this.tableName)
					&& exportInterceptor.getColumnName().equalsIgnoreCase(this.columnName)) {
				return true;
			}
			// or after cast just return (this.hashCode() == importInterceptor.hashCode());
		}
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public int hashCode() {
		int hash = 7;
		hash = 79 * hash + (this.tableName != null ? this.tableName.hashCode() : 0);
		hash = 79 * hash + (this.columnName != null ? this.columnName.hashCode() : 0);
		return hash;
	}

	/**
	 * <p>
	 * Getter for the field <code>tableName</code>.
	 * </p>
	 * 
	 * @return a {@link java.lang.String} object.
	 */
	public String getTableName() {
		return tableName;
	}

	/**
	 * <p>
	 * Getter for the field <code>columnName</code>.
	 * </p>
	 * 
	 * @return a {@link java.lang.String} object.
	 */
	public String getColumnName() {
		return columnName;
	}

}
