package fr.ifremer.adagio.synchro.intercept.data;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.synchro.intercept.data.internal.ExportUpdateDateInterceptor;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.event.LoadTableEvent;
import fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.sql.Timestamp;

/**
 * replace update_date column by a system timestamp, when exported to server.
 * This interceptor is only used for table without column 'id' AND 'remote_id', because this tables are managed by
 * DataTableInterceptor.
 * 
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.7.1
 */
public class TableWithUpdateDateInterceptor extends AbstractDataInterceptor {

	private static final Log log = LogFactory.getLog(TableWithUpdateDateInterceptor.class);

	private Timestamp systimestamp = null;

	/**
	 * <p>
	 * Constructor for TableWithUpdateDateInterceptor.
	 * </p>
	 */
	public TableWithUpdateDateInterceptor() {
		super(SynchroDirection.EXPORT_TEMP2SERVER);
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {
		TableWithUpdateDateInterceptor result = (TableWithUpdateDateInterceptor) super.clone();
		result.systimestamp = this.systimestamp;
		return result;
	}

	/** {@inheritDoc} */
	public boolean doApply(SynchroDatabaseMetadata dbMeta, org.hibernate.tool.hbm2ddl.TableMetadata table) {

		return hasColumns(table, getConfig().getColumnUpdateDate())
				// Exclude tables already managed by DataTableInterceptor
				&& !hasColumns(table, getConfig().getColumnId(), getConfig().getColumnRemoteId());
	};

	/**
	 * <p>
	 * handleTableLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadTableEvent} object.
	 */
	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {
		if (systimestamp == null) {
			systimestamp = checkAndGetSystemTimestamp(getConfig());
		}

		int updateDateColumnIndex = e.table.getSelectColumnIndex(getConfig().getColumnUpdateDate());
		if (updateDateColumnIndex <= 0) {
			log.warn(String.format("Unable to find %s column in the select query. %s will be ignore.",
					getConfig().getColumnUpdateDate(),
					getClass().getSimpleName()));
		}
		else {
			ExportUpdateDateInterceptor setUpdateDateInterceptor =
					new ExportUpdateDateInterceptor(updateDateColumnIndex, systimestamp);

			e.table.addInterceptor(setUpdateDateInterceptor);
		}
	}

	/* -- Protected methods -- */

	/**
	 * <p>
	 * checkAndGetSystemTimestamp.
	 * </p>
	 * 
	 * @param configuration
	 *            a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @return a {@link java.sql.Timestamp} object.
	 */
	protected Timestamp checkAndGetSystemTimestamp(SynchroDatabaseConfiguration configuration) {
		Timestamp systimestamp = configuration.getSystemTimestamp();
		Preconditions.checkNotNull(systimestamp,
				String.format("Could not found system timestamp in database configuration. This is need for %s", getClass().getSimpleName()));
		return systimestamp;
	}

}
