package fr.ifremer.adagio.synchro.intercept.data;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.core.dao.technical.synchronization.SynchronizationStatus;
import fr.ifremer.adagio.synchro.meta.DatabaseColumns;
import fr.ifremer.adagio.synchro.meta.data.DataSynchroTables;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.common.synchro.meta.event.LoadTableEvent;
import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryName;
import fr.ifremer.common.synchro.query.SynchroQueryOperator;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.util.Set;

/**
 * <p>
 * ScientificCruiseInterceptor class.
 * </p>
 * 
 * @author Nolwenn Rannou (nolwenn.rannou@ifremer.fr)
 * @since 4.0.3
 */
public class ScientificCruiseInterceptor extends AbstractDataInterceptor {

	private static final String TABLE = DataSynchroTables.SCIENTIFIC_CRUISE.name();
	private static final String COLUMN_ID = DatabaseColumns.ID.name();

	/**
	 * <p>
	 * Constructor for ScientificCruiseInterceptor.
	 * </p>
	 */
	public ScientificCruiseInterceptor() {
		super();
	}

	/** {@inheritDoc} */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return TABLE.equalsIgnoreCase(table.getName());
	}

	/**
	 * <p>
	 * handleTableLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link LoadTableEvent} object.
	 */
	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {

		// Table should be set as root manually, inside specific interceptor (cf RootDataTableInterceptor)
		e.table.setRoot(true);
	}

	/**
	 * <p>
	 * handleQuery.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.CreateQueryEvent} object.
	 */
	@Subscribe
	public void handleQuery(CreateQueryEvent e) {
		SynchroDirection direction = getConfig().getDirection();

		switch (e.queryName) {
		// Select queries : remove unused columns
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
		case selectMaxUpdateDate:
			if (direction == SynchroDirection.IMPORT_SERVER2TEMP) {
				// Add restriction (pk includes, rights...)
				e.sql = addRestrictionOnImportServer2TempDb(e.source, e.queryName, e.sql);
			}
			else if (direction == SynchroDirection.IMPORT_TEMP2LOCAL) {
				// Add restriction (pk includes, ...)
				e.sql = addRestrictionOnImportTemp2LocalDb(e.source, e.queryName, e.sql);
			}
			else if (direction == SynchroDirection.EXPORT_LOCAL2TEMP) {
				// Add restriction (pk includes, rights...)
				e.sql = addRestrictionOnExport(e.source, e.queryName, e.sql);
			}
			break;

		default:
			break;
		}
	}

	/**
	 * <p>
	 * addRestrictionOnImportServer2TempDb.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param queryName
	 *            a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String addRestrictionOnImportServer2TempDb(SynchroTableMetadata table, SynchroQueryName queryName, String sql) {

		DataSynchroDatabaseConfiguration databaseConfiguration = getConfig();

		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(queryName, sql);

		// select
		// -> need to add a 'distinct' because of inner join on LANDING
		if (queryName == SynchroQueryName.count
				|| queryName == SynchroQueryName.countFromUpdateDate) {
			queryBuilder.replaceColumn("count(*)", "count(distinct t.ID)");
		}
		else {
			queryBuilder.setColumnDistinct(true);
		}

		// Retrieve the person id
		int personId = checkAndGetPersonId();

		// where: if administrator or program rights
		queryBuilder.addWhere(SynchroQueryOperator.AND,
				String.format(
						"(EXISTS ("
								+ "SELECT 1 FROM PROGRAM2PERSON"
								+ " WHERE PROGRAM_FK = T.PROGRAM_FK AND PERSON_FK = %s"
								+ ") OR EXISTS ("
								+ "SELECT 1 FROM PERSON2USER_PROFIL"
								+ " WHERE PERSON_FK = %s"
								+ "   AND USER_PROFIL_FK = %s"
								+ "))",
						personId,
						personId,
						getConfig().getUserProfilIdReferentailAdministrator()
						));

		// where: limit to pks (for import by Pk)
		String pkFilter = createPkFilter(table.getName(), COLUMN_ID);
		if (StringUtils.isNotBlank(pkFilter)) {
			// Apply Pk filter, but do not apply date restriction
			queryBuilder.addWhere(SynchroQueryOperator.AND, pkFilter);
		}

		// where: filter on period, if a date column exists AND no Pk filter
		else if (databaseConfiguration.getDataStartDate() != null
				&& databaseConfiguration.getDataEndDate() != null) {
			// Retrieve the date column to user
			String dateColumnName = getFirstExistingColumnName(table,
					"return_date_time",
					"sale_start_date",
					"landing_date_time",
					"end_date_time",
					"start_date_time"
					);
			// If found, add a filter on start/end dates
			if (dateColumnName != null) {
				queryBuilder.addWhere(SynchroQueryOperator.AND, String.format(
						"t.%s  >= :startDate AND t.%s <= :endDate", dateColumnName, dateColumnName));
			}
		}

		return queryBuilder.build();
	}

	/**
	 * <p>
	 * addRestrictionOnImportTemp2LocalDb.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param queryName
	 *            a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String addRestrictionOnImportTemp2LocalDb(SynchroTableMetadata table, SynchroQueryName queryName, String sql) {

		String pkFilter = createPkFilter(table.getName(), COLUMN_ID);
		if (StringUtils.isBlank(pkFilter)) {
			return sql;
		}

		// where: limit to pks (for import by Pk) - need for batch import (see mantis #27275)
		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(queryName, sql);
		queryBuilder.addWhere(SynchroQueryOperator.AND, pkFilter);
		return queryBuilder.build();
	}

	/**
	 * <p>
	 * addRestrictionOnExport.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param queryName
	 *            a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String addRestrictionOnExport(SynchroTableMetadata table, SynchroQueryName queryName, String sql) {

		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(queryName, sql);

		// select
		// -> need to add a 'distinct' because of inner join on LANDING
		if (queryName == SynchroQueryName.count
				|| queryName == SynchroQueryName.countFromUpdateDate) {
			queryBuilder.replaceColumn("count(*)", "count(distinct t.ID)");
		}
		else {
			queryBuilder.setColumnDistinct(true);
		}

		// where: limit to pks (for import by Pk - mantis #31458)
		String pkFilter = createPkFilter(table.getName(), COLUMN_ID);
		if (StringUtils.isNotBlank(pkFilter)) {
			// Apply Pk filter, but do not apply date restriction
			queryBuilder.addWhere(SynchroQueryOperator.AND, pkFilter);

			// where: only 'ready to sync' data
			queryBuilder.addWhere(SynchroQueryOperator.AND,
					String.format("%s='%s'", DatabaseColumns.SYNCHRONIZATION_STATUS.name(), SynchronizationStatus.READY_TO_SYNCHRONIZE.value()));
		}

		// If no PK includes: export by user rights
		else {
			// Retrieve the person id
			int personId = checkAndGetPersonId();

			// where: if administrator or program rights
			queryBuilder.addWhere(SynchroQueryOperator.AND,
					String.format(
							"(EXISTS ("
									+ "SELECT 1 FROM PROGRAM2PERSON"
									+ " WHERE PROGRAM_FK = T.PROGRAM_FK AND PERSON_FK = %s"
									+ ") OR EXISTS ("
									+ "SELECT 1 FROM PERSON2USER_PROFIL"
									+ " WHERE PERSON_FK = %s"
									+ "   AND USER_PROFIL_FK = %s"
									+ "))",
							personId,
							personId,
							getConfig().getUserProfilIdReferentailAdministrator()));

			// where: only 'ready to sync' data
			queryBuilder.addWhere(SynchroQueryOperator.AND,
					String.format("%s='%s'", DatabaseColumns.SYNCHRONIZATION_STATUS.name(), SynchronizationStatus.READY_TO_SYNCHRONIZE.value()));
		}

		return queryBuilder.build();
	}

	/**
	 * <p>
	 * getFirstExistingColumnName.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param columnNames
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String getFirstExistingColumnName(SynchroTableMetadata table, String... columnNames) {
		Set<String> delegateColumns = table.getColumnNames();
		for (String columnName : columnNames) {
			if (delegateColumns.contains(columnName.toLowerCase())) {
				return columnName;
			}
		}
		return null;
	}
}
