package fr.ifremer.adagio.synchro.intercept.data;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.dao.technical.synchronization.SynchronizationStatus;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryName;
import fr.ifremer.common.synchro.query.SynchroQueryOperator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.util.Set;

/**
 * Default interceptor applied to all table suspected to be root. <br>
 * Table should be set as root manually, inside specific interceptor (e.g. because Landing should not be define as root)
 */
public class RootDataTableInterceptor extends AbstractDataInterceptor {

	/** Constant <code>COLUMN_ID="id"</code> */
	public static final String COLUMN_ID = "id";
	/** Constant <code>COLUMN_VESSEL_FK="vessel_fk"</code> */
	public static final String COLUMN_VESSEL_FK = "vessel_fk";
	/** Constant <code>COLUMN_PROGRAM_FK="program_fk"</code> */
	public static final String COLUMN_PROGRAM_FK = "program_fk";
	/** Constant <code>COLUMN_SYNCHRONIZATION_STATUS="synchronization_status"</code> */
	public static final String COLUMN_SYNCHRONIZATION_STATUS = "synchronization_status";

	/** {@inheritDoc} */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		boolean isRootTable = isRootTable(table);
		return isRootTable;
	}

	// Table should be set as root manually
	// (because landing should not be define as root)
	/*
	 * @Subscribe
	 * public void handleLoadTable(LoadTableEvent e) {
	 * // if (!table.isRoot()) {
	 * // e.table.setRoot(true);
	 * // }
	 * }
	 */

	/**
	 * <p>
	 * handleQuery.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.CreateQueryEvent} object.
	 */
	@Subscribe
	public void handleQuery(CreateQueryEvent e) {
		SynchroDirection direction = getConfig().getDirection();

		switch (e.queryName) {
		// Select queries : remove unused columns
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
		case selectMaxUpdateDate:
			if (direction == SynchroDirection.IMPORT_SERVER2TEMP) {
				// Add restriction (person session, etc.)
				e.sql = addRestrictionOnImportServer2TempDb(e.source, e.queryName, e.sql);

				// Add a filter for debug
				e.sql = addDebugRestriction(e.sql);
			}
			else if (direction == SynchroDirection.IMPORT_TEMP2LOCAL) {
				// Add restriction (pk includes, ...)
				e.sql = addRestrictionOnImportTemp2LocalDb(e.source, e.queryName, e.sql);
			}
			else if (direction == SynchroDirection.EXPORT_LOCAL2TEMP) {
				// Add restriction on person session
				e.sql = addRestrictionOnExport(e.source, e.queryName, e.sql);

				// Add a filter for debug
				e.sql = addDebugRestriction(e.sql);
			}
			break;

		default:
			break;
		}
	}

	/**
	 * <p>
	 * addRestrictionOnImportServer2TempDb.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param queryName
	 *            a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String addRestrictionOnImportServer2TempDb(SynchroTableMetadata table, SynchroQueryName queryName, String sql) {

		DataSynchroDatabaseConfiguration databaseConfiguration = getConfig();
		boolean enableUpdateDateFilter = SynchroQueryName.withUpdateDate(queryName);

		Set<String> objectTypes = ObjectTypeHelper.getObjectTypeFromTableName(table.getName());
		Preconditions.checkNotNull(CollectionUtils.isNotEmpty(objectTypes));
		String objectTypeList = new StringBuilder().append("'").append(Joiner.on("','").join(objectTypes)).append("'")
				.toString();

		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(queryName, sql);

		// Retrieve the session id
		int personSessionId = checkAndGetPersonSessionId();

		// select
		// -> need to add a 'distinct' because of inner join on LANDING
		if (queryName == SynchroQueryName.count
				|| queryName == SynchroQueryName.countFromUpdateDate) {
			queryBuilder.replaceColumn("count(*)", "count(distinct t.ID)");
		}
		else {
			queryBuilder.setColumnDistinct(true);
		}

		// from
		queryBuilder.addJoin(
				"INNER JOIN PERSON_SESSION_VESSEL psv"
						+ " ON psv.vessel_fk=t.vessel_fk"
						+ String.format(" AND psv.person_session_fk=%s", personSessionId)
						+ " AND psv.program_fk=t.program_fk");

		// where: if rights updated
		if (enableUpdateDateFilter) {
			queryBuilder.addWhere(SynchroQueryOperator.OR, "psv.update_date", ">", ":updateDate");
		}

		// where: limit rights on object types
		queryBuilder.addWhere(SynchroQueryOperator.AND,
				String.format("psv.object_type_fk IN (%s)", objectTypeList));

		// where: programs
		String programFilter = createProgramCodesFilter("t.program_fk IN (%s)");
		if (StringUtils.isNotBlank(programFilter)) {
			queryBuilder.addWhere(SynchroQueryOperator.AND, programFilter);
		}

		// where: limit to pks (for import by Pk)
		String pkFilter = createPkFilter(table.getName(), COLUMN_ID);
		if (StringUtils.isNotBlank(pkFilter)) {
			// Apply Pk filter, but do not apply date restriction
			queryBuilder.addWhere(SynchroQueryOperator.AND, pkFilter);
		}

		// where: filter on period, if a date column exists AND no Pk filter
		else if (databaseConfiguration.getDataStartDate() != null
				&& databaseConfiguration.getDataEndDate() != null) {
			// Retrieve the date column to user
			String dateColumnName = getFirstExistingColumnName(table,
					"return_date_time",
					"sale_start_date",
					"landing_date_time",
					"end_date_time",
					"start_date_time"
					);
			// If found, add a filter on start/end dates
			if (dateColumnName != null) {
				queryBuilder.addWhere(SynchroQueryOperator.AND, String.format(
						"t.%s  >= :startDate AND t.%s <= :endDate", dateColumnName, dateColumnName));
			}
		}

		return queryBuilder.build();
	}

	/**
	 * <p>
	 * addRestrictionOnImportTemp2LocalDb.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param queryName
	 *            a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String addRestrictionOnImportTemp2LocalDb(SynchroTableMetadata table, SynchroQueryName queryName, String sql) {

		String pkFilter = createPkFilter(table.getName(), COLUMN_ID);
		if (StringUtils.isBlank(pkFilter)) {
			return sql;
		}

		// where: limit to pks (for import by Pk) - need for batch import (see mantis #27275)
		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(queryName, sql);
		queryBuilder.addWhere(SynchroQueryOperator.AND, pkFilter);
		return queryBuilder.build();
	}

	/**
	 * <p>
	 * addRestrictionOnExport.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param queryName
	 *            a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String addRestrictionOnExport(SynchroTableMetadata table, SynchroQueryName queryName, String sql) {

		Set<String> objectTypes = ObjectTypeHelper.getObjectTypeFromTableName(table.getName());
		Preconditions.checkNotNull(CollectionUtils.isNotEmpty(objectTypes));
		String objectTypeList = new StringBuilder().append("'").append(Joiner.on("','").join(objectTypes)).append("'")
				.toString();

		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(queryName, sql);

		// select
		// -> need to add a 'distinct' because of inner join on LANDING
		if (queryName == SynchroQueryName.count
				|| queryName == SynchroQueryName.countFromUpdateDate) {
			queryBuilder.replaceColumn("count(*)", "count(distinct t.ID)");
		}
		else {
			queryBuilder.setColumnDistinct(true);
		}

		// where: limit to pks (for import by Pk - mantis #31458)
		String pkFilter = createPkFilter(table.getName(), COLUMN_ID);
		if (StringUtils.isNotBlank(pkFilter)) {
			// Apply Pk filter, but do not apply date restriction
			queryBuilder.addWhere(SynchroQueryOperator.AND, pkFilter);

			// where: only 'ready to sync' data
			queryBuilder.addWhere(SynchroQueryOperator.AND,
					String.format("%s='%s'", COLUMN_SYNCHRONIZATION_STATUS, SynchronizationStatus.READY_TO_SYNCHRONIZE.value()));
		}

		// If no PK includes: export by user rights
		else {
			// Retrieve the person id
			int personId = checkAndGetPersonId();

			// join: on PERSON_SESSION
			queryBuilder.addJoin(
					"INNER JOIN PERSON_SESSION_VESSEL psv"
							+ " ON psv.vessel_fk=t.vessel_fk"
							+ " AND psv.program_fk=t.program_fk"
							+ " INNER JOIN PERSON_SESSION ps ON ps.id=psv.person_session_fk"
							+ " AND ps.person_fk=" + personId);

			// where: only 'ready to sync' data
			queryBuilder.addWhere(SynchroQueryOperator.AND,
					String.format("%s='%s'", COLUMN_SYNCHRONIZATION_STATUS, SynchronizationStatus.READY_TO_SYNCHRONIZE.value()));

			// where: limit rights on object types
			queryBuilder.addWhere(SynchroQueryOperator.AND,
					String.format("psv.object_type_fk IN (%s)", objectTypeList));

			// where: programs
			String programFilter = createProgramCodesFilter("t.program_fk IN (%s)");
			if (StringUtils.isNotBlank(programFilter)) {
				queryBuilder.addWhere(SynchroQueryOperator.AND, programFilter);
			}
		}

		return queryBuilder.build();
	}

	/**
	 * <p>
	 * isRootTable.
	 * </p>
	 * 
	 * @param table
	 *            a {@link org.hibernate.tool.hbm2ddl.TableMetadata} object.
	 * @return a boolean.
	 */
	protected boolean isRootTable(TableMetadata table) {
		boolean hasRootColumns = hasColumns(table, COLUMN_VESSEL_FK, COLUMN_PROGRAM_FK, getConfig().getColumnUpdateDate());
		boolean hasObjectType = CollectionUtils.isNotEmpty(ObjectTypeHelper.getObjectTypeFromTableName(table.getName()));
		boolean isExcluded = "SCIENTIFIC_CRUISE".equals(table.getName());

		return hasRootColumns && hasObjectType && !isExcluded;
	}

	/**
	 * <p>
	 * addDebugRestriction.
	 * </p>
	 * 
	 * @param sql
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String addDebugRestriction(String sql) {

		String vesselIncludesStr = AdagioConfiguration.getInstance().getApplicationConfig().getOption("adagio.synchro.import.vessels.includes");
		if (StringUtils.isBlank(vesselIncludesStr)) {
			return sql;
		}

		// Vessels to includes
		StringBuilder vesselsParam = new StringBuilder();
		for (String vesselCode : vesselIncludesStr.split(",")) {
			vesselsParam.append(",'")
					.append(vesselCode)
					.append("'");
		}
		return SynchroQueryBuilder.newBuilder(sql)
				.addWhere(SynchroQueryOperator.AND, String.format("t.vessel_fk in (%s)", vesselsParam.substring(1)))
				.build();
	}

	/**
	 * <p>
	 * createProgramCodesFilter.
	 * </p>
	 * 
	 * @param stringToFormat
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createProgramCodesFilter(String stringToFormat) {
		String programCodes = AdagioConfiguration.getInstance().getImportProgramCodes();
		if (StringUtils.isBlank(programCodes)) {
			return null;
		}
		return String.format(stringToFormat,
				"'"
						+ Joiner.on("','").join(Splitter.on(',').split(programCodes))
						+ "'");
	}

	/**
	 * <p>
	 * getFirstExistingColumnName.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param columnNames
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String getFirstExistingColumnName(SynchroTableMetadata table, String... columnNames) {
		Set<String> delegateColumns = table.getColumnNames();
		for (String columnName : columnNames) {
			if (delegateColumns.contains(columnName.toLowerCase())) {
				return columnName;
			}
		}
		return null;
	}
}
