package fr.ifremer.adagio.synchro.intercept.data;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.dao.technical.synchronization.SynchronizationStatus;
import fr.ifremer.adagio.synchro.intercept.data.internal.ExportRemoveNegativeValueInterceptor;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy;
import fr.ifremer.common.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.common.synchro.meta.event.LoadTableEvent;
import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryName;
import fr.ifremer.common.synchro.query.SynchroQueryOperator;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.LockMode;
import org.hibernate.tool.hbm2ddl.ColumnMetadata;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * <p>
 * ObservedLocationInterceptor class.
 * </p>
 * 
 */
public class ObservedLocationInterceptor extends AbstractDataInterceptor {

	/** Constant <code>COLUMN_PROGRAM_FK="program_fk"</code> */
	public static final String COLUMN_PROGRAM_FK = "program_fk";
	/** Constant <code>COLUMN_START_DATE_TIME="start_date_time"</code> */
	public static final String COLUMN_START_DATE_TIME = "start_date_time";
	/** Constant <code>COLUMN_LOCATION_FK="location_fk"</code> */
	public static final String COLUMN_LOCATION_FK = "location_fk";
	/** Constant <code>COLUMN_RECORDER_PERSON_FK="recorder_person_fk"</code> */
	public static final String COLUMN_RECORDER_PERSON_FK = "recorder_person_fk";
	/** Constant <code>COLUMN_SYNCHRONIZATION_STATUS="synchronization_status"</code> */
	public static final String COLUMN_SYNCHRONIZATION_STATUS = "synchronization_status";

	/** Constant <code>NATURAL_ID_COLUMN_NAMES</code> */
	public static final List<String> NATURAL_ID_COLUMN_NAMES = ImmutableList.of(COLUMN_PROGRAM_FK, COLUMN_START_DATE_TIME, COLUMN_LOCATION_FK,
			COLUMN_RECORDER_PERSON_FK);

	/** {@inheritDoc} */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		if (!"OBSERVED_LOCATION".equalsIgnoreCase(table.getName())) {
			return false;
		}

		Map<String, ColumnMetadata> delegateColumns = SynchroTableMetadata.getColumns(table);
		boolean hasNeedColumns = delegateColumns.containsKey(COLUMN_PROGRAM_FK);

		return hasNeedColumns;
	}

	/**
	 * <p>
	 * handleTableLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadTableEvent} object.
	 */
	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {
		SynchroTableMetadata table = e.table;
		SynchroDirection direction = getConfig().getDirection();

		// Define as root table
		table.setRoot(true);

		// Export: Local DB -> Temp DB
		if (direction == SynchroDirection.EXPORT_LOCAL2TEMP) {
			int remoteIdColumnIndex = table.getSelectColumnIndex(getConfig().getColumnRemoteId());

			// Remove fake remote_id (added for link with PersonSessionItem)
			// See ObsdebObservedLocationDaoImpl.insertTemporaryPersonSessionItems()
			ExportRemoveNegativeValueInterceptor remoteIdInterceptor = new ExportRemoveNegativeValueInterceptor(
					getConfig(),
					table.getName(),
					remoteIdColumnIndex);

			table.addInterceptor(remoteIdInterceptor);
		}

		// Export: Temp DB -> Server DB
		else if (direction == SynchroDirection.EXPORT_TEMP2SERVER) {
			// Define natural id
			table.addUniqueConstraint("NATURAL_ID_UNIQUE_C", NATURAL_ID_COLUMN_NAMES, DuplicateKeyStrategy.REJECT);

			// Enable lock on update
			table.setLockOnUpdate(LockMode.UPGRADE_NOWAIT);
		}
	}

	/**
	 * <p>
	 * handleQuery.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.CreateQueryEvent} object.
	 */
	@Subscribe
	public void handleQuery(CreateQueryEvent e) {
		SynchroDirection direction = getConfig().getDirection();

		switch (e.queryName) {
		// Select queries : remove unsed columns
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
		case selectMaxUpdateDate:
			// Import: Server DB -> Temp DB
			if (direction == SynchroDirection.IMPORT_SERVER2TEMP) {
				// Add restriction on person session, etc.
				e.sql = addRestrictionOnImportServer2TempDb(e.source, e.queryName, e.sql);

				// Add a filter for debug, on ID and VESSEL_FK
				e.sql = addDebugRestriction(e.sql, getConfig().getColumnId());
			}

			// Import: Temp DB -> Local DB
			else if (direction == SynchroDirection.IMPORT_TEMP2LOCAL) {
				// Add restriction on pkIncludes
				e.sql = addRestrictionOnImportTemp2LocalDb(e.source, e.queryName, e.sql);
			}

			// Export: Local DB -> Temp DB
			else if (direction == SynchroDirection.EXPORT_LOCAL2TEMP) {
				// Add restriction on person session
				e.sql = addRestrictionOnExport(e.source, e.queryName, e.sql);

				// Add a filter for debug, on REMOTE_ID and VESSEL_FK
				e.sql = addDebugRestriction(e.sql, getConfig().getColumnRemoteId());
			}

			break;

		default:
			break;
		}
	}

	/**
	 * <p>
	 * addRestrictionOnImportServer2TempDb.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param queryName
	 *            a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String addRestrictionOnImportServer2TempDb(SynchroTableMetadata table, SynchroQueryName queryName, String sql) {

		boolean enableUpdateDateFilter = SynchroQueryName.withUpdateDate(queryName);

		Set<String> objectTypes = ObjectTypeHelper.getObjectTypeFromTableName(table.getName());
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(objectTypes));
		String objectTypeList = new StringBuilder().append("'").append(Joiner.on("','").join(objectTypes)).append("'")
				.toString();

		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(queryName, sql);

		// Retrieve the session id
		int personSessionId = checkAndGetPersonSessionId();

		// select
		// -> need to add a 'distinct' because of inner join on LANDING
		if (queryName == SynchroQueryName.count
				|| queryName == SynchroQueryName.countFromUpdateDate) {
			queryBuilder.replaceColumn("count(*)", "count(distinct t.ID)");
		}
		else {
			queryBuilder.setColumnDistinct(true);
		}

		// Add join on PERSON_SESSION_VESSEL and PERSON_SESSION_ITEM (mantis #24182)
		queryBuilder.addJoin(
				" LEFT OUTER JOIN LANDING l ON l.observed_location_fk=t.id"
						+ " LEFT OUTER JOIN PERSON_SESSION_VESSEL psv ON psv.vessel_fk=l.vessel_fk AND psv.program_fk=t.program_fk"
						+ " AND psv.person_session_fk=" + personSessionId
						+ " LEFT OUTER JOIN PERSON_SESSION_ITEM psi ON psi.object_id=t.id"
						+ " AND psi.person_session_fk=" + personSessionId);

		// where: detect access rights changes (using UPDATE_DATE from PERSON_SESSION_xxx tables)
		if (enableUpdateDateFilter) {
			queryBuilder.addWhere(SynchroQueryOperator.OR,
					"((psv.id IS NOT NULL AND psv.update_date > :updateDate)"
							+ " OR (psi.id IS NOT NULL AND psi.update_date > :updateDate))");
		}

		// where: select observed_location
		// - with right on vessel (PERSON_SESSION_VESSEL)
		// - with right by id (PERSON_SESSION_ITEM)
		// - without landing, and never exported

		// WARNING: make sure there is left & right parenthesis (mantis #25174)
		queryBuilder.addWhere(SynchroQueryOperator.AND, String.format(
				"((psv.object_type_fk IN (%s) AND NOT (t.start_date_time > psv.end_date_time OR t.end_date_time < psv.start_date_time))"
						+ " OR (psi.object_type_fk IN (%s)))",
				objectTypeList,
				objectTypeList
				));

		// where: Filter on programs
		String programFilter = createProgramCodesFilter("t.program_fk IN (%s)");
		if (StringUtils.isNotBlank(programFilter)) {
			queryBuilder.addWhere(SynchroQueryOperator.AND, programFilter);
		}

		// where: limit to pks (for import by Pk)
		String pkFilter = createPkFilter(table.getName(), getConfig().getColumnId());
		if (StringUtils.isNotBlank(pkFilter)) {
			queryBuilder.addWhere(SynchroQueryOperator.AND, pkFilter);
		}

		// where: period, if a date column exists AND no Pk filter
		else if (getConfig().getDataStartDate() != null
				&& getConfig().getDataEndDate() != null) {

			queryBuilder.addWhere(SynchroQueryOperator.AND, "t.start_date_time >= :startDate AND t.start_date_time <= :endDate");
		}
		return queryBuilder.build();
	}

	/**
	 * <p>
	 * addRestrictionOnImportTemp2LocalDb.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param queryName
	 *            a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String addRestrictionOnImportTemp2LocalDb(SynchroTableMetadata table, SynchroQueryName queryName, String sql) {
		String pkFilter = createPkFilter(table.getName(), getConfig().getColumnId());
		if (StringUtils.isBlank(pkFilter)) {
			return sql;
		}

		// where: limit to pks (for import by Pk) - need for batch import (see mantis #27275)
		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(queryName, sql);
		queryBuilder.addWhere(SynchroQueryOperator.AND, pkFilter);
		return queryBuilder.build();
	}

	/**
	 * <p>
	 * addRestrictionOnExport.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @param queryName
	 *            a {@link fr.ifremer.common.synchro.query.SynchroQueryName} object.
	 * @param sql
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String addRestrictionOnExport(SynchroTableMetadata table, SynchroQueryName queryName, String sql) {

		Set<String> objectTypes = ObjectTypeHelper.getObjectTypeFromTableName(table.getName());
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(objectTypes));
		String objectTypeList = new StringBuilder().append("'").append(Joiner.on("','").join(objectTypes)).append("'")
				.toString();

		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(queryName, sql);

		// Retrieve the person id
		int personId = checkAndGetPersonId();

		// select
		// -> need to add a 'distinct' because of inner join on LANDING
		if (queryName == SynchroQueryName.count
				|| queryName == SynchroQueryName.countFromUpdateDate) {
			queryBuilder.replaceColumn("count(*)", "count(distinct t.ID)");
		}
		else {
			queryBuilder.setColumnDistinct(true);
		}

		// Add join on PERSON_SESSION
		queryBuilder.addJoin(
				" INNER JOIN PERSON_SESSION ps ON ps.person_fk=" + personId
						+ " LEFT OUTER JOIN LANDING l ON l.observed_location_fk=t.id"
						+ " LEFT OUTER JOIN PERSON_SESSION_VESSEL psv ON psv.vessel_fk=l.vessel_fk AND psv.program_fk = t.program_fk"
						+ " AND psv.person_session_fk=ps.id"
						+ " LEFT OUTER JOIN PERSON_SESSION_ITEM psi ON psi.object_id=t.remote_id"
						+ " AND psi.person_session_fk=ps.id");

		// where: synchronisation_status
		queryBuilder.addWhere(SynchroQueryOperator.AND,
				String.format("%s='%s'", COLUMN_SYNCHRONIZATION_STATUS, SynchronizationStatus.READY_TO_SYNCHRONIZE.value()));

		// where: select observed_location :
		// - with right on vessel (PERSON_SESSION_VESSEL)
		// - with right by id (PERSON_SESSION_ITEM)
		// - without landing, and never exported

		// WARNING: make sure there is left & right parenthesis (mantis #25174)
		queryBuilder.addWhere(SynchroQueryOperator.AND, String.format(
				"((psv.object_type_fk IN (%s) AND NOT (t.start_date_time > psv.end_date_time OR t.end_date_time < psv.start_date_time))"
						+ " OR (psi.object_type_fk IN (%s))"
						// Comment this, since mantis #28889 : we MUST NOT export observations WITHOUT vessel IF not
						// PersonSessionItem
						// even for the first exportation (See
						// ObsdebObservedLocationDaoImpl.insertTemporaryPersonSessionItems())
						// // Force to export observed location WITHOUT vessel, if has never been exported
						// " OR (l.id IS NULL AND t.remote_id IS NULL)"
						+ ")",
				objectTypeList,
				objectTypeList));

		// where: programs
		String programFilter = createProgramCodesFilter("t.program_fk IN (%s)");
		if (StringUtils.isNotBlank(programFilter)) {
			queryBuilder.addWhere(SynchroQueryOperator.AND, programFilter);
		}

		return queryBuilder.build();
	}

	/**
	 * <p>
	 * addDebugRestriction.
	 * </p>
	 * 
	 * @param sql
	 *            a {@link java.lang.String} object.
	 * @param observedLocationFilterColumnName
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String addDebugRestriction(String sql, String observedLocationFilterColumnName) {

		String idIncludes = AdagioConfiguration.getInstance().getApplicationConfig().getOption("adagio.synchro.import.observedLocation.includes");
		String vesselIncludesStr = AdagioConfiguration.getInstance().getApplicationConfig().getOption("adagio.synchro.import.vessels.includes");
		if (StringUtils.isBlank(idIncludes) && StringUtils.isBlank(vesselIncludesStr)) {
			return sql;
		}

		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(sql);

		// IDs to include
		if (StringUtils.isNotBlank(idIncludes)) {
			queryBuilder.addWhere(SynchroQueryOperator.AND, String.format("t.%s in (%s)",
					observedLocationFilterColumnName,
					idIncludes));
		}

		// Vessels to includes
		if (StringUtils.isNotBlank(vesselIncludesStr)) {
			StringBuilder vesselParam = new StringBuilder();
			for (String vesselCode : vesselIncludesStr.split(",")) {
				vesselParam.append(",'")
						.append(vesselCode)
						.append("'");
			}
			queryBuilder.addWhere(SynchroQueryOperator.AND,
					String.format("t.id in (select distinct l.observed_location_fk from landing l where l.vessel_fk in (%s))",
							vesselParam.substring(1)));
		}

		return queryBuilder.build();
	}

	/**
	 * <p>
	 * createProgramCodesFilter.
	 * </p>
	 * 
	 * @param stringToFormat
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createProgramCodesFilter(String stringToFormat) {
		String programCodes = AdagioConfiguration.getInstance().getImportProgramCodes();
		if (StringUtils.isBlank(programCodes)) {
			return "";
		}
		return String.format(stringToFormat,
				"'"
						+ Joiner.on("','").join(Splitter.on(',').split(programCodes))
						+ "'");
	}
}
