package fr.ifremer.adagio.synchro.intercept.data;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import org.apache.commons.collections4.CollectionUtils;

import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

/**
 * <p>
 * ObjectTypeHelper class.
 * </p>
 * 
 */
public class ObjectTypeHelper {

	/** Constant <code>instance</code> */
	protected static ObjectTypeHelper instance = null;

	/** Constant <code>table2objectMap</code> */
	protected static final Map<String, Set<String>> table2objectMap = initMap();

	private ObjectTypeHelper() {
		// helper class : no constructor
	}

	/**
	 * <p>
	 * getObjectTypeFromTableName.
	 * </p>
	 * 
	 * @param tableName
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.util.Set} object.
	 */
	public static Set<String> getObjectTypeFromTableName(String tableName) {
		Preconditions.checkNotNull(tableName);
		Set<String> result = table2objectMap.get(tableName);
		return result;
	}

	/**
	 * <p>
	 * getObjectTypeFromTableName.
	 * </p>
	 * 
	 * @param tableName
	 *            a {@link java.lang.String} object.
	 * @param defaultValue
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.util.Set} object.
	 */
	public static Set<String> getObjectTypeFromTableName(String tableName, String defaultValue) {
		Preconditions.checkNotNull(tableName);
		Set<String> result = table2objectMap.get(tableName);
		if (CollectionUtils.isEmpty(result) && defaultValue != null) {
			return ImmutableSet.of(defaultValue);
		}
		return result;
	}

	/**
	 * <p>
	 * getFirstObjectTypeFromTableName.
	 * </p>
	 * 
	 * @param tableName
	 *            a {@link java.lang.String} object.
	 * @param defaultValue
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	public static String getFirstObjectTypeFromTableName(String tableName, String defaultValue) {
		Set<String> objectTypes = getObjectTypeFromTableName(tableName);
		if (CollectionUtils.isEmpty(objectTypes)) {
			return defaultValue;
		}
		return objectTypes.iterator().next();
	}

	/**
	 * <p>
	 * getTableNameFromObjectType.
	 * </p>
	 * 
	 * @param objectType
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	public static String getTableNameFromObjectType(String objectType) {
		Preconditions.checkNotNull(objectType);

		for (Entry<String, Set<String>> entry : table2objectMap.entrySet()) {
			String tableName = entry.getKey();
			Set<String> objectTypes = entry.getValue();
			if (objectTypes.contains(objectType)) {
				return tableName;
			}
		}

		// not found, return itself
		return objectType;
	}

	/**
	 * Renvoie le code de type d'object par défaut correspondant à la classe 'aClass'. Il s'agit du nom simple (
	 * {@link java.lang.Class#getSimpleName()}) de la classe 'aClass' converti en majuscules et avec un caractère
	 * souligné '_' avant chaque majuscule du nom d'origine.<br>
	 * Par exemple pour ActivityCalendar, on renvoie "ACTIVITY_CALENDAR".
	 * 
	 * @param aClass
	 *            a {@link java.lang.Class} object.
	 * @return null lorsque aClass est nul
	 */
	public static final String toDefaultObjectTypeCode(Class<?> aClass) {
		String result = null;
		if (aClass != null) {
			StringBuilder sb = new StringBuilder();
			String name = aClass.getSimpleName();
			if (name != null && name.length() > 0) {
				sb.append(Character.toUpperCase(name.charAt(0)));
				for (int i = 1; i < name.length(); i++) {
					char caract = name.charAt(i);
					if (Character.isUpperCase(caract)) {
						sb.append("_"); //$NON-NLS-1$
					} else {
						caract = Character.toUpperCase(caract);
					}
					sb.append(caract);
				}
			}
			result = sb.toString();
		}
		return result;
	}

	/* -- Init map -- */

	/**
	 * <p>
	 * initMap.
	 * </p>
	 * 
	 * @return a {@link java.util.Map} object.
	 */
	protected static Map<String, Set<String>> initMap() {
		Map<String, Set<String>> result = Maps.newHashMap();

		// Warning : need only to set special case (OBJECT_TYPE_FK != TABLE_NAME)
		// -------------------------------------------------------------
		// Referential
		// -------------------------------------------------------------

		// Need to specify referential tables, to be sure to imported deletion (mantis #24124)
		result.put("VESSEL_FEATURES", ImmutableSet.of("VESSEL_FEATURES"));
		result.put("LOCATION", ImmutableSet.of("LOCATION"));
		result.put("VESSEL", ImmutableSet.of("VESSEL"));
		result.put("REFERENCE_TAXON", ImmutableSet.of("REFERENCE_TAXON"));
		result.put("TAXON_GROUP", ImmutableSet.of("TAXON_GROUP"));
		result.put("GEAR", ImmutableSet.of("GEAR"));
		result.put("TAXON_GROUP", ImmutableSet.of("TAXON_GROUP"));
		result.put("METIER", ImmutableSet.of("METIER"));
		result.put("PMFM", ImmutableSet.of("PMFM"));
		result.put("LOCATION_ASSOCIATION", ImmutableSet.of("LOCATION_ASSOCIATION"));
		result.put("TAXON_NAME", ImmutableSet.of("TAXON_NAME"));
		result.put("PARAMETER", ImmutableSet.of("PARAMETER"));
		result.put("LOCATION_LEVEL", ImmutableSet.of("LOCATION_LEVEL"));
		result.put("VESSEL_OWNER", ImmutableSet.of("VESSEL_OWNER"));
		result.put("ROUND_WEIGHT_CONVERSION", ImmutableSet.of("ROUND_WEIGHT_CONVERSION"));
		result.put("QUALITATIVE_VALUE", ImmutableSet.of("QUALITATIVE_VALUE"));
		result.put("PERSON", ImmutableSet.of("PERSON"));
		result.put("DEPTH_GRADIENT", ImmutableSet.of("DEPTH_GRADIENT"));
		result.put("DISTANCE_TO_COAST_GRADIENT", ImmutableSet.of("DISTANCE_TO_COAST_GRADIENT"));
		result.put("NEARBY_SPECIFIC_AREA", ImmutableSet.of("NEARBY_SPECIFIC_AREA"));
		result.put("PROGRAM", ImmutableSet.of("PROGRAM"));
		result.put("GEAR_CLASSIFICATION", ImmutableSet.of("GEAR_CLASSIFICATION"));

		// -------------------------------------------------------------
		// Data
		// -------------------------------------------------------------
		result.put("FISHING_TRIP", ImmutableSet.of("OBSERVED_FISHING_TRIP", "FISHING_TRIP"));
		result.put("SALE", ImmutableSet.of("SALE", "OBSERVED_SALE"));
		result.put("BATCH", ImmutableSet.of("CATCH_BATCH", "BATCH"));
		result.put("LANDING", ImmutableSet.of("OBSERVED_LANDING", "LANDING"));
		// observed location: use observed LANDING instead
		result.put("OBSERVED_LOCATION", ImmutableSet.of("OBSERVED_LOCATION", "OBSERVED_LANDING"));

		result.put("SCIENTIFIC_CRUISE", ImmutableSet.of("SCIENTIFIC_CRUISE"));
		result.put("ACTIVITY_CALENDAR", ImmutableSet.of("ACTIVITY_CALENDAR"));
		result.put("DAILY_ACTIVITY_CALENDAR", ImmutableSet.of("DAILY_ACTIVITY_CALENDAR"));
		result.put("DECLARED_DOCUMENT_REFERENCE", ImmutableSet.of("DECLARED_DOCUMENT_REFERENCE"));
		// result.put("GEAR_USE_FEATURES", ImmutableSet.of("GEAR_USE_FEATURES"));
		// result.put("GEAR_PHYSICAL_FEATURES", ImmutableSet.of("GEAR_PHYSICAL_FEATURES"));
		result.put("PERSON_SESSION_VESSEL", ImmutableSet.of("PERSON_SESSION_VESSEL"));
		result.put("SAMPLE", ImmutableSet.of("SAMPLE"));

		return result;
	}
}
