package fr.ifremer.adagio.synchro.intercept.data;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.ImmutableList;
import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.core.dao.technical.synchronization.SynchronizationStatus;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.common.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy;
import fr.ifremer.common.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.common.synchro.meta.event.LoadJoinEvent;
import fr.ifremer.common.synchro.meta.event.LoadTableEvent;
import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryOperator;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

/**
 * <p>
 * LandingInterceptor class.
 * </p>
 * 
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.5.0
 */
public class LandingInterceptor extends AbstractDataInterceptor {

	/** Constant <code>COLUMN_PROGRAM_FK="program_fk"</code> */
	public static final String COLUMN_PROGRAM_FK = "program_fk";

	private int fishingTripFkColumnIndex = -1;

	private static final String TABLE_LANDING = "LANDING";
	private static final String TABLE_PRODUCE = "PRODUCE";
	private static final String TABLE_FISHING_TRIP = "FISHING_TRIP";
	private static final String TABLE_BATCH = "BATCH";
	private static final String LANDING_COLUMN_ID = "id";
	private static final String LANDING_COLUMN_FISHING_TRIP_FK = "fishing_trip_fk";
	private static final String LANDING_COLUMN_OBSERVED_LOCATION_FK = "observed_location_fk";
	private static final String FISHING_TRIP_COLUMN_ID = "id";
	private static final String FK_COLUMN_NAME = "OBSERVED_LOCATION_FK";
	private static final String SYNCHRONIZATION_STATUS_COLUMN_NAME = "SYNCHRONIZATION_STATUS";

	private static final List<String> LANDING_UNIQUE_KEY = ImmutableList.of("LANDING_DATE_TIME", "VESSEL_FK", "PROGRAM_FK", "RANK_ORDER");

	private PreparedStatement selectFishingTripIdStatement = null;

	/**
	 * <p>
	 * Constructor for LandingInterceptor.
	 * </p>
	 */
	public LandingInterceptor() {
		super();
		setEnableOnWrite(true);
	}

	/** {@inheritDoc} */
	@Override
	public void init(DataSynchroDatabaseConfiguration config) {
		super.init(config);
		enableIntegrityConstraints = config.getDirection() == SynchroDirection.EXPORT_TEMP2SERVER
				|| config.getDirection() == SynchroDirection.IMPORT_TEMP2LOCAL;
	}

	/** {@inheritDoc} */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return TABLE_LANDING.equalsIgnoreCase(table.getName());
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {
		LandingInterceptor newBean = (LandingInterceptor) super.clone();
		newBean.fishingTripFkColumnIndex = this.fishingTripFkColumnIndex;
		return newBean;
	}

	/**
	 * <p>
	 * handleQuery.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.CreateQueryEvent} object.
	 */
	@Subscribe
	public void handleQuery(CreateQueryEvent e) {
		switch (e.queryName) {
		// Select queries : remove unsed columns
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
		case selectMaxUpdateDate:
			// Exclude Landings that are child of OBSERVED_LOCATION or FISHING_TRIP
			e.sql = SynchroQueryBuilder.newBuilder(e.sql)
					.addWhere(SynchroQueryOperator.AND, String.format("t.%s is null", LANDING_COLUMN_OBSERVED_LOCATION_FK))
					.addWhere(SynchroQueryOperator.AND, String.format("t.%s is null", LANDING_COLUMN_FISHING_TRIP_FK))
					.build();
			break;

		default:
			break;
		}
	}

	/**
	 * <p>
	 * handleTableLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadTableEvent} object.
	 */
	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {

		fishingTripFkColumnIndex = e.table.getSelectColumnIndex(LANDING_COLUMN_FISHING_TRIP_FK);

		e.table.setRoot(false);

		// Define a natural Id, on server
		if (getConfig().getDirection() == SynchroDirection.EXPORT_TEMP2SERVER) {
			// The landing have to be update after children updates or deletes (Mantis #30027 & #30314)
			e.table.addUniqueConstraint("LANDING_UNIQUE_KEY", LANDING_UNIQUE_KEY, DuplicateKeyStrategy.REPLACE_LOW_PRIORITY);
		}
		else if (getConfig().getDirection() == SynchroDirection.EXPORT_LOCAL2TEMP) {
			// When select by OBSERVED_LOCATION_FK, SYNCHRONIZATION_STATUS must not be DELETED
			e.table.addSelectByFksWhereClause(FK_COLUMN_NAME,
					String.format("t.%s<>'%s'", SYNCHRONIZATION_STATUS_COLUMN_NAME, SynchronizationStatus.DELETED.getValue()));
		}
	}

	/**
	 * <p>
	 * handleJoinLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadJoinEvent} object.
	 */
	@Subscribe
	public void handleJoinLoad(LoadJoinEvent e) {

		if (!e.join.isValid()) {
			return;
		}

		String targetTableName = e.join.getTargetTable().getName();

		// disable join to CatchBatch (not used)
		if (TABLE_BATCH.equalsIgnoreCase(targetTableName)) {
			e.join.setIsValid(false);
		}

		// disable join to Produce (will used Landing->ExpectedSale->Produce) - mantis #23253
		if (TABLE_PRODUCE.equalsIgnoreCase(targetTableName)) {
			e.join.setIsValid(false);
		}

	}

	/** {@inheritDoc} */
	@Override
	protected void doOnWrite(Object[] data,
			List<Object> pk,
			SynchroTableDao sourceDao,
			SynchroTableDao targetDao,
			SynchroOperationRepository buffer,
			boolean insert)
			throws SQLException {
		// If root entity: no transformation
		if (buffer == null) {
			return;
		}
		Object fishingTripRemoteId = data[fishingTripFkColumnIndex];
		if (fishingTripRemoteId != null) {
			buffer.addChildToUpdateFromOneColumn(TABLE_FISHING_TRIP, FISHING_TRIP_COLUMN_ID, fishingTripRemoteId);

			if (enableIntegrityConstraints) {
				// set as Null, and update later
				data[fishingTripFkColumnIndex] = null;

				buffer.addMissingColumnUpdate(LANDING_COLUMN_FISHING_TRIP_FK, pk, fishingTripRemoteId);
			}
		}
	}

	/** {@inheritDoc} */
	@Override
	protected void doOnDelete(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
		// Warning: use parseInt because ID could a String or an Integer
		Integer landingId = Integer.parseInt(pk.get(0).toString());

		// Retrieve fishing trip id
		Integer fishingTripId = getFishingTripId(targetDao.getConnection(), landingId);

		// If exists, add to children to delete
		if (fishingTripId != null) {
			buffer.addChildToDeleteFromOneColumn(TABLE_FISHING_TRIP, FISHING_TRIP_COLUMN_ID, fishingTripId);
		}
	}

	/** {@inheritDoc} */
	@Override
	protected void doOnDetach(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
		// Warning: use parseInt because ID could a String or an Integer
		Integer landingId = Integer.parseInt(pk.get(0).toString());

		// Retrieve fishing trip id
		Integer fishingTripId = getFishingTripId(targetDao.getConnection(), landingId);

		// If exists, add to children to delete
		if (fishingTripId != null) {
			buffer.addChildToDetachFromOneColumn(TABLE_FISHING_TRIP, FISHING_TRIP_COLUMN_ID, fishingTripId);
		}
	}

	/** {@inheritDoc} */
	@Override
	protected void doClose() throws IOException {
		super.doClose();

		Daos.closeSilently(selectFishingTripIdStatement);
		selectFishingTripIdStatement = null;
	}

	/* -- internal methods -- */

	/**
	 * <p>
	 * getFishingTripId.
	 * </p>
	 * 
	 * @param connection
	 *            a {@link java.sql.Connection} object.
	 * @param landingId
	 *            a int.
	 * @return a {@link java.lang.Integer} object.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected Integer getFishingTripId(Connection connection, int landingId) throws SQLException {
		if (selectFishingTripIdStatement == null) {
			selectFishingTripIdStatement = connection.prepareStatement(initSelectFishingTripIdStatement());
		}

		selectFishingTripIdStatement.setInt(1, landingId);

		ResultSet resultSet = null;
		try {
			resultSet = selectFishingTripIdStatement.executeQuery();

			if (!resultSet.next()) {
				return null;
			}
			if (resultSet.getObject(1) == null) {
				return null;
			}
			return resultSet.getInt(1);
		} finally {
			Daos.closeSilently(resultSet);
		}
	}

	/**
	 * <p>
	 * initSelectFishingTripIdStatement.
	 * </p>
	 * 
	 * @return a {@link java.lang.String} object.
	 */
	protected String initSelectFishingTripIdStatement() {
		return String.format("SELECT %s FROM %s where %s=?",
				LANDING_COLUMN_FISHING_TRIP_FK,
				TABLE_LANDING,
				LANDING_COLUMN_ID
				);
	}
}
