package fr.ifremer.adagio.synchro.intercept.data;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.ImmutableList;
import com.google.common.eventbus.Subscribe;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy;
import fr.ifremer.common.synchro.meta.event.LoadJoinEvent;
import fr.ifremer.common.synchro.meta.event.LoadTableEvent;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.util.List;

/**
 * <p>
 * GearPhysicalFeaturesInterceptor class.
 * </p>
 * 
 */
public class GearPhysicalFeaturesInterceptor extends AbstractDataInterceptor {

	/** Constant <code>TARGET_TABLE_OPERATION="OPERATION"</code> */
	public static final String TARGET_TABLE_OPERATION = "OPERATION";
	/** Constant <code>TARGET_COLUMN_GEAR_PHYSICAL_FEATURES="gear_physical_features_fk"</code> */
	public static final String TARGET_COLUMN_GEAR_PHYSICAL_FEATURES = "gear_physical_features_fk";

	private static final String UNIQUE_KEY_NAME = "GEAR_PHYS_FEAT_UNIQUE_KEY";
	private static final List<String> UNIQUE_KEY_COLS = ImmutableList.of("VESSEL_FK", "PROGRAM_FK", "GEAR_FK",
			"PHYSICAL_GEAR_SURVEY_FK", "FISHING_TRIP_FK", "RANK_ORDER", "START_DATE");

	/** {@inheritDoc} */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return "GEAR_PHYSICAL_FEATURES".equalsIgnoreCase(table.getName());
	}

	/**
	 * <p>
	 * handleJoinLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadJoinEvent} object.
	 */
	@Subscribe
	public void handleJoinLoad(LoadJoinEvent e) {
		if (!e.join.isValid()) {
			return;
		}

		// Disable join between OPERATION and GEAR_PHYSICAL_FEATURES
		String targetTableName = e.join.getTargetTable().getName();
		String targetColumnName = e.join.getTargetColumn().getName();
		boolean isJoinToOperationTable = TARGET_TABLE_OPERATION.equalsIgnoreCase(targetTableName)
				&& TARGET_COLUMN_GEAR_PHYSICAL_FEATURES.equalsIgnoreCase(targetColumnName);

		if (isJoinToOperationTable) {
			e.join.setIsValid(false);
		}
	}

	/**
	 * <p>
	 * handleTableLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadTableEvent} object.
	 */
	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {

		// Define a natural Id, on server
		if (getConfig().getDirection() == SynchroDirection.EXPORT_TEMP2SERVER) {
			// Add the unique key - mantis #23769
			// Should be configure as 'REPLACE', because it's possible (in ObsDeb), to remove then recreate the same
			// data
			// -> Define replacement has low priority (see mantis #29693) to prefer replacement by REMOTE_ID
			e.table.addUniqueConstraint(UNIQUE_KEY_NAME, UNIQUE_KEY_COLS, DuplicateKeyStrategy.REPLACE_LOW_PRIORITY);
		}
	}
}
